<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class PlatformSetting extends Model
{
    use HasFactory;

    protected $fillable = [
        'key',
        'value',
        'type',
        'group',
        'description',
        'is_encrypted'
    ];

    protected $casts = [
        'is_encrypted' => 'boolean',
    ];

    /**
     * Get a setting value by key
     */
    public static function get(string $key, $default = null)
    {
        $setting = static::where('key', $key)->first();
        
        if (!$setting) {
            return $default;
        }

        $value = $setting->value;

        // Decrypt if encrypted
        if ($setting->is_encrypted && $value) {
            try {
                $value = decrypt($value);
            } catch (\Exception $e) {
                return $default;
            }
        }

        // Cast based on type
        return match ($setting->type) {
            'boolean' => filter_var($value, FILTER_VALIDATE_BOOLEAN),
            'integer' => (int) $value,
            'float' => (float) $value,
            'array', 'json' => json_decode($value, true),
            default => $value,
        };
    }

    /**
     * Set a setting value
     */
    public static function set(string $key, $value, string $type = 'string', bool $encrypt = false, string $group = 'general', string $description = ''): void
    {
        // Convert value to string for storage
        $stringValue = match ($type) {
            'array', 'json' => json_encode($value),
            'boolean' => $value ? '1' : '0',
            default => (string) $value,
        };

        // Encrypt if needed
        if ($encrypt && $stringValue) {
            $stringValue = encrypt($stringValue);
        }

        static::updateOrCreate(
            ['key' => $key],
            [
                'value' => $stringValue,
                'type' => $type,
                'group' => $group,
                'description' => $description,
                'is_encrypted' => $encrypt,
            ]
        );
    }

    /**
     * Get all settings by group
     */
    public static function getByGroup(string $group): array
    {
        $settings = static::where('group', $group)->get();
        $result = [];

        foreach ($settings as $setting) {
            $value = $setting->value;

            // Decrypt if encrypted
            if ($setting->is_encrypted && $value) {
                try {
                    $value = decrypt($value);
                } catch (\Exception $e) {
                    continue;
                }
            }

            // Cast based on type
            $result[$setting->key] = match ($setting->type) {
                'boolean' => filter_var($value, FILTER_VALIDATE_BOOLEAN),
                'integer' => (int) $value,
                'float' => (float) $value,
                'array', 'json' => json_decode($value, true),
                default => $value,
            };
        }

        return $result;
    }

    /**
     * Delete a setting
     */
    public static function remove(string $key): bool
    {
        return static::where('key', $key)->delete() > 0;
    }
}