<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;

class PopupNotification extends Model
{
    use HasFactory;

    protected $fillable = [
        'title',
        'message',
        'type',
        'is_active',
        'start_date',
        'end_date',
        'target_users',
        'created_by',
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'start_date' => 'datetime',
        'end_date' => 'datetime',
        'target_users' => 'array',
    ];

    /**
     * Notification types
     */
    const TYPE_INFO = 'info';
    const TYPE_WARNING = 'warning';
    const TYPE_SUCCESS = 'success';
    const TYPE_ANNOUNCEMENT = 'announcement';

    /**
     * Get the admin who created this notification
     */
    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Users who have seen this notification
     */
    public function seenBy(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'user_popup_notifications')
                    ->withPivot('seen_at')
                    ->withTimestamps();
    }

    /**
     * Check if notification is currently active
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true)
                    ->where(function ($q) {
                        $q->whereNull('start_date')
                          ->orWhere('start_date', '<=', now());
                    })
                    ->where(function ($q) {
                        $q->whereNull('end_date')
                          ->orWhere('end_date', '>=', now());
                    });
    }

    /**
     * Check if user should see this notification
     */
    public function shouldShowToUser(User $user): bool
    {
        // Check if notification is active
        if (!$this->is_active) {
            return false;
        }

        // Check date range
        if ($this->start_date && $this->start_date > now()) {
            return false;
        }

        if ($this->end_date && $this->end_date < now()) {
            return false;
        }

        // Check if user has already seen it
        if ($this->seenBy()->where('user_id', $user->id)->exists()) {
            return false;
        }

        // Check target users
        if ($this->target_users) {
            $targetUsers = $this->target_users;
            
            // If specific user IDs are targeted
            if (isset($targetUsers['user_ids']) && !in_array($user->id, $targetUsers['user_ids'])) {
                return false;
            }

            // If specific levels are targeted
            if (isset($targetUsers['levels']) && !in_array($user->level, $targetUsers['levels'])) {
                return false;
            }

            // If minimum level is required
            if (isset($targetUsers['min_level']) && $user->level < $targetUsers['min_level']) {
                return false;
            }
        }

        return true;
    }

    /**
     * Mark as seen by user
     */
    public function markAsSeenBy(User $user): void
    {
        $this->seenBy()->syncWithoutDetaching([
            $user->id => ['seen_at' => now()]
        ]);
    }

    /**
     * Get notification icon based on type
     */
    public function getIconAttribute(): string
    {
        return match($this->type) {
            self::TYPE_INFO => '📢',
            self::TYPE_WARNING => '⚠️',
            self::TYPE_SUCCESS => '✅',
            self::TYPE_ANNOUNCEMENT => '🎉',
            default => '📢'
        };
    }

    /**
     * Get CSS class for notification type
     */
    public function getCssClassAttribute(): string
    {
        return match($this->type) {
            self::TYPE_INFO => 'info',
            self::TYPE_WARNING => 'warning',
            self::TYPE_SUCCESS => 'success',
            self::TYPE_ANNOUNCEMENT => 'announcement',
            default => 'info'
        };
    }
}
