<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Str;

class Post extends Model
{
    use HasFactory;

    protected $fillable = [
        'title',
        'slug',
        'content',
        'excerpt',
        'type',
        'status',
        'pending_approval',
        'featured_image',
        'metadata',
        'author_id',
        'published_at',
        'views',
    ];

    protected $casts = [
        'metadata' => 'array',
        'published_at' => 'datetime',
        'pending_approval' => 'boolean',
    ];

    /**
     * Post types
     */
    const TYPE_BLOG = 'blog';
    const TYPE_NEWS = 'news';
    const TYPE_HIGHLIGHT = 'highlight';
    const TYPE_TOURNAMENT = 'tournament';
    const TYPE_UPDATE = 'update';

    /**
     * Post statuses
     */
    const STATUS_DRAFT = 'draft';
    const STATUS_PUBLISHED = 'published';
    const STATUS_ARCHIVED = 'archived';
    
    // For creator submissions awaiting approval, we'll use draft status with a flag
    const STATUS_PENDING_APPROVAL = 'draft'; // Use draft for pending approval for now

    /**
     * Get the author of the post
     */
    public function author(): BelongsTo
    {
        return $this->belongsTo(User::class, 'author_id');
    }

    /**
     * Get posts that are published
     */
    public function scopePublished($query)
    {
        return $query->where('status', self::STATUS_PUBLISHED)
                    ->where('published_at', '<=', now());
    }

    /**
     * Get posts that are pending approval
     */
    public function scopePending($query)
    {
        return $query->where('pending_approval', true)->where('status', 'draft');
    }

    /**
     * Get posts by type
     */
    public function scopeOfType($query, $type)
    {
        return $query->where('type', $type);
    }

    /**
     * Generate slug from title
     */
    public static function generateSlug($title)
    {
        $slug = Str::slug($title);
        $count = static::where('slug', 'LIKE', "{$slug}%")->count();
        
        return $count > 0 ? "{$slug}-" . ($count + 1) : $slug;
    }

    /**
     * Get the URL for the post
     */
    public function getUrlAttribute()
    {
        return route('posts.show', $this->slug);
    }

    /**
     * Get excerpt or generate from content
     */
    public function getExcerptAttribute($value)
    {
        if ($value) {
            return $value;
        }

        return Str::limit(strip_tags($this->content), 150);
    }

    /**
     * Get reading time estimate
     */
    public function getReadingTimeAttribute()
    {
        $wordCount = str_word_count(strip_tags($this->content));
        $readingTime = ceil($wordCount / 200); // Average reading speed
        
        return max(1, $readingTime);
    }

    /**
     * Increment view count
     */
    public function incrementViews()
    {
        $this->increment('views');
    }
}
