<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Tournament extends Model
{
    use HasFactory;

    protected $fillable = [
        'host_id',
        'game_id',
        'type',
        'title',
        'description',
        'entry_fee',
        'prize_pool',
        'livestream_link',
        'status',
        'bracket_json',
        'starts_at',
        'ends_at',
    ];

    protected $casts = [
        'entry_fee' => 'decimal:2',
        'prize_pool' => 'decimal:2',
        'bracket_json' => 'array',
        'starts_at' => 'datetime',
        'ends_at' => 'datetime',
    ];

    /**
     * The host of the tournament
     */
    public function host(): BelongsTo
    {
        return $this->belongsTo(User::class, 'host_id');
    }

    /**
     * The game for this tournament
     */
    public function game(): BelongsTo
    {
        return $this->belongsTo(Game::class);
    }

    /**
     * Tournament participants
     */
    public function participants(): HasMany
    {
        return $this->hasMany(TournamentParticipant::class);
    }

    /**
     * Users participating in the tournament
     */
    public function users()
    {
        return $this->belongsToMany(User::class, 'tournament_participants')
                    ->withPivot('seed', 'result')
                    ->withTimestamps();
    }

    /**
     * Tournament matches
     */
    public function matches(): HasMany
    {
        return $this->hasMany(TournamentMatch::class);
    }

    /**
     * Tournament invites
     */
    public function invites(): HasMany
    {
        return $this->hasMany(TournamentInvite::class);
    }

    /**
     * Pending tournament invites
     */
    public function pendingInvites(): HasMany
    {
        return $this->hasMany(TournamentInvite::class)->where('status', 'pending');
    }

    /**
     * Check if tournament is upcoming
     */
    public function isUpcoming(): bool
    {
        return $this->status === 'upcoming';
    }

    /**
     * Check if tournament is ongoing
     */
    public function isOngoing(): bool
    {
        return $this->status === 'ongoing';
    }

    /**
     * Check if tournament is completed
     */
    public function isCompleted(): bool
    {
        return $this->status === 'completed';
    }

    /**
     * Check if user can join the tournament
     */
    public function canBeJoinedBy(User $user): bool
    {
        return $this->isUpcoming() 
            && !$this->users()->where('user_id', $user->id)->exists()
            && $user->wallet_balance >= $this->entry_fee
            && !$this->hasParticipantLimit();
    }

    /**
     * Check if tournament has reached participant limit for even number requirement
     */
    public function hasParticipantLimit(): bool
    {
        if ($this->type !== 'knockout') {
            return false;
        }

        $currentCount = $this->participants()->count();
        $maxParticipants = $this->getMaxParticipants();
        
        return $currentCount >= $maxParticipants;
    }

    /**
     * Get maximum participants for knockout tournament (must be power of 2)
     */
    public function getMaxParticipants(): int
    {
        // Default maximum for knockout tournaments (can be made configurable)
        return 16; // 16 = 2^4, allows for 4 rounds
    }

    /**
     * Check if tournament needs even number of participants
     */
    public function requiresEvenParticipants(): bool
    {
        return $this->type === 'knockout';
    }

    /**
     * Check if tournament can start
     */
    public function canStart(): bool
    {
        if (!$this->isUpcoming()) {
            return false;
        }

        $participantCount = $this->participants()->count();
        
        if ($this->requiresEvenParticipants()) {
            // Must be power of 2 and at least 2 participants
            return $participantCount >= 2 && ($participantCount & ($participantCount - 1)) === 0;
        }

        return $participantCount >= 2;
    }

    /**
     * Invite a user to the tournament
     */
    public function inviteUser(User $invitee, User $inviter, string $message = null): ?TournamentInvite
    {
        // Check if user is already invited or participating
        if ($this->hasUserInviteOrParticipation($invitee)) {
            return null;
        }

        // Check if tournament can still accept participants
        if ($this->hasParticipantLimit()) {
            return null;
        }

        return $this->invites()->create([
            'inviter_id' => $inviter->id,
            'invitee_id' => $invitee->id,
            'message' => $message,
            'expires_at' => now()->addDays(7), // Invite expires in 7 days
        ]);
    }

    /**
     * Check if user has invite or already participates
     */
    public function hasUserInviteOrParticipation(User $user): bool
    {
        return $this->users()->where('user_id', $user->id)->exists() ||
               $this->invites()->where('invitee_id', $user->id)
                   ->whereIn('status', ['pending', 'accepted'])
                   ->exists();
    }

    /**
     * Generate bracket for knockout tournament
     */
    public function generateBracket(): array
    {
        if ($this->type !== 'knockout') {
            return [];
        }

        $participants = $this->participants()->with('user')->get();
        $participantCount = $participants->count();

        if ($participantCount < 2 || ($participantCount & ($participantCount - 1)) !== 0) {
            throw new \Exception('Knockout tournaments require a power of 2 number of participants');
        }

        // Shuffle participants for random seeding
        $shuffledParticipants = $participants->shuffle();

        // Create bracket structure
        $bracket = $this->createBracketStructure($shuffledParticipants);

        // Update the tournament with bracket data
        $this->update(['bracket_json' => $bracket]);

        return $bracket;
    }

    /**
     * Create bracket structure for knockout tournament
     */
    private function createBracketStructure($participants): array
    {
        $participantCount = $participants->count();
        $rounds = log($participantCount, 2);
        
        $bracket = [
            'rounds' => $rounds,
            'participant_count' => $participantCount,
            'matches' => []
        ];

        // Generate first round matches
        for ($i = 0; $i < $participantCount; $i += 2) {
            $match = [
                'round' => 1,
                'match_number' => ($i / 2) + 1,
                'player_a' => [
                    'id' => $participants[$i]->user->id,
                    'username' => $participants[$i]->user->username,
                ],
                'player_b' => [
                    'id' => $participants[$i + 1]->user->id,
                    'username' => $participants[$i + 1]->user->username,
                ],
                'winner' => null,
                'status' => 'pending'
            ];
            
            $bracket['matches'][] = $match;
        }

        // Generate subsequent round placeholders
        $matchesInPreviousRound = $participantCount / 2;
        for ($round = 2; $round <= $rounds; $round++) {
            $matchesInCurrentRound = $matchesInPreviousRound / 2;
            
            for ($i = 0; $i < $matchesInCurrentRound; $i++) {
                $match = [
                    'round' => $round,
                    'match_number' => $i + 1,
                    'player_a' => null,
                    'player_b' => null,
                    'winner' => null,
                    'status' => 'waiting'
                ];
                
                $bracket['matches'][] = $match;
            }
            
            $matchesInPreviousRound = $matchesInCurrentRound;
        }

        return $bracket;
    }

    /**
     * Start the tournament
     */
    public function start(): bool
    {
        if (!$this->canStart()) {
            return false;
        }

        // Generate bracket for knockout tournaments
        if ($this->type === 'knockout') {
            $this->generateBracket();
        }

        // Create actual match records
        $this->createMatches();

        // Update tournament status
        $this->update(['status' => 'ongoing']);

        return true;
    }

    /**
     * Create match records from bracket
     */
    private function createMatches(): void
    {
        if (!$this->bracket_json) {
            return;
        }

        $bracket = $this->bracket_json;
        
        foreach ($bracket['matches'] as $matchData) {
            if ($matchData['round'] === 1 && 
                $matchData['player_a'] && 
                $matchData['player_b']) {
                
                $this->matches()->create([
                    'round' => $matchData['round'],
                    'player_a_id' => $matchData['player_a']['id'],
                    'player_b_id' => $matchData['player_b']['id'],
                    'result' => 'none'
                ]);
            }
        }
    }

    /**
     * Get tournament winner
     */
    public function getWinner(): ?User
    {
        $winner = $this->participants()
                      ->where('result', 'winner')
                      ->first();
                      
        return $winner ? $winner->user : null;
    }

    /**
     * Get tournament runner-up
     */
    public function getRunnerUp(): ?User
    {
        $runnerUp = $this->participants()
                        ->where('result', 'runner_up')
                        ->first();
                        
        return $runnerUp ? $runnerUp->user : null;
    }
}