<?php

namespace App\Models;

use App\Enums\UserRole;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Facades\Hash;
use Laravel\Sanctum\HasApiTokens;

/**
 * User Model
 * 
 * Represents a user in the gaming platform with comprehensive relationship support,
 * level management, wallet functionality, and friend system integration.
 * 
 * @property int $id
 * @property string $username
 * @property string $email
 * @property string|null $phone
 * @property string $password
 * @property string|null $transaction_pin
 * @property string|null $avatar_url
 * @property int $xp Experience points earned by the user
 * @property int $level Current level of the user
 * @property float $wallet_balance Current wallet balance in naira
 * @property float $locked_balance Locked balance during pending transactions
 * @property string $kyc_status KYC verification status (pending, verified, rejected)
 * @property float $agreement_rate User's agreement rate percentage
 * @property int $strike_count Number of strikes against the user
 * @property bool $is_organizer Whether user is an organizer/admin
 * @property string $role User role (level1, level2, highest, creator)
 * @property string $status Account status (active, suspended, banned)
 * @property string|null $first_name
 * @property string|null $last_name
 * @property string|null $phone_number
 * @property \Carbon\Carbon|null $date_of_birth
 * @property string|null $address
 * @property array|null $kyc_data Additional KYC verification data
 * @property string|null $qore_id_reference External QoreID reference
 * @property \Carbon\Carbon|null $email_verified_at
 * @property \Carbon\Carbon|null $phone_verified_at
 * @property \Carbon\Carbon|null $pin_set_at
 * @property \Carbon\Carbon|null $created_at
 * @property \Carbon\Carbon|null $updated_at
 */
class User extends Authenticatable
{
    use HasApiTokens, HasFactory, Notifiable;

    /**
     * The attributes that are mass assignable.
     */
    protected $fillable = [
        'username',
        'email',
        'phone',
        'password',
        'transaction_pin',
        'avatar_url',
        'xp',
        'games_played',
        'level',
        'wallet_balance',
        'locked_balance',
        'kyc_status',
        'agreement_rate',
        'strike_count',
        'is_organizer',
        'is_banned',
        'role',
        'creator_request_status',
        'creator_request_date',
        'creator_request_reason',
        'status',
        'first_name',
        'last_name',
        'phone_number',
        'date_of_birth',
        'address',
        'kyc_data',
        'qore_id_reference',
    ];

    /**
     * The attributes that should be hidden for serialization.
     */
    protected $hidden = [
        'password',
        'transaction_pin',
        'remember_token',
    ];

    /**
     * The attributes that should be cast.
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
        'phone_verified_at' => 'datetime',
        'pin_set_at' => 'datetime',
        'creator_request_date' => 'datetime',
        'date_of_birth' => 'date',
        'password' => 'hashed',
        'transaction_pin' => 'hashed',
        'wallet_balance' => 'decimal:2',
        'locked_balance' => 'decimal:2',
        'agreement_rate' => 'decimal:2',
        'is_organizer' => 'boolean',
        'is_banned' => 'boolean',
        'kyc_data' => 'array',
        'role' => UserRole::class,
    ];

    // Relationships

    /**
     * Challenges created by this user
     */
    public function createdChallenges(): HasMany
    {
        return $this->hasMany(Challenge::class, 'creator_id');
    }

    /**
     * Challenges accepted by this user
     */
    public function acceptedChallenges(): HasMany
    {
        return $this->hasMany(Challenge::class, 'accepter_id');
    }

    /**
     * All challenges (created or accepted)
     */
    public function challenges()
    {
        return $this->createdChallenges()->union($this->acceptedChallenges());
    }

    /**
     * Tournaments hosted by this user
     */
    public function hostedTournaments(): HasMany
    {
        return $this->hasMany(Tournament::class, 'host_id');
    }

    /**
     * Tournament participations
     */
    public function tournamentParticipations(): HasMany
    {
        return $this->hasMany(TournamentParticipant::class);
    }

    /**
     * Tournaments this user is participating in
     */
    public function tournaments()
    {
        return $this->belongsToMany(Tournament::class, 'tournament_participants')
                    ->withPivot('seed', 'result')
                    ->withTimestamps();
    }

    /**
     * Tournament invites sent by this user
     */
    public function sentTournamentInvites(): HasMany
    {
        return $this->hasMany(TournamentInvite::class, 'inviter_id');
    }

    /**
     * Tournament invites received by this user
     */
    public function receivedTournamentInvites(): HasMany
    {
        return $this->hasMany(TournamentInvite::class, 'invitee_id');
    }

    /**
     * Pending tournament invites for this user
     */
    public function pendingTournamentInvites(): HasMany
    {
        return $this->hasMany(TournamentInvite::class, 'invitee_id')
                    ->where('status', 'pending')
                    ->where('expires_at', '>', now());
    }

    /**
     * Challenge invites sent by this user
     */
    public function sentChallengeInvites(): HasMany
    {
        return $this->hasMany(ChallengeInvite::class, 'inviter_id');
    }

    /**
     * Challenge invites received by this user
     */
    public function receivedChallengeInvites(): HasMany
    {
        return $this->hasMany(ChallengeInvite::class, 'invitee_id');
    }

    /**
     * Pending challenge invites for this user
     */
    public function pendingChallengeInvites(): HasMany
    {
        return $this->hasMany(ChallengeInvite::class, 'invitee_id')
                    ->where('status', 'pending')
                    ->where('expires_at', '>', now());
    }

    /**
     * User's badges
     */
    public function badges()
    {
        return $this->belongsToMany(Badge::class, 'user_badges')->withTimestamps();
    }

    /**
     * User's wallet transactions
     */
    public function walletTransactions(): HasMany
    {
        return $this->hasMany(WalletTransaction::class);
    }

    /**
     * Match proofs uploaded by this user
     */
    public function matchProofs(): HasMany
    {
        return $this->hasMany(MatchProof::class);
    }

    /**
     * Disputes opened by this user
     */
    public function disputes(): HasMany
    {
        return $this->hasMany(Dispute::class, 'opened_by');
    }

    /**
     * Disputes resolved by this user (admin)
     */
    public function resolvedDisputes(): HasMany
    {
        return $this->hasMany(Dispute::class, 'admin_id');
    }

    /**
     * User's notifications
     */
    public function notifications(): HasMany
    {
        return $this->hasMany(Notification::class);
    }

    /**
     * Posts authored by this user
     */
    public function posts(): HasMany
    {
        return $this->hasMany(Post::class, 'author_id');
    }

    /**
     * Friend requests sent by this user
     * 
     * @return HasMany<Friend> Collection of friend requests sent by this user
     */
    public function sentFriendRequests(): HasMany
    {
        return $this->hasMany(Friend::class, 'user_id');
    }

    /**
     * Friend requests received by this user  
     * 
     * @return HasMany<Friend> Collection of friend requests received by this user
     */
    public function receivedFriendRequests(): HasMany
    {
        return $this->hasMany(Friend::class, 'friend_id');
    }

    /**
     * Accepted friends of this user
     * 
     * Uses many-to-many relationship through the friends table to get users
     * who have an accepted friendship with this user.
     * 
     * @return BelongsToMany Collection of users (User models) who are friends with this user
     */
    public function friends(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'friends', 'user_id', 'friend_id')
                    ->where('friends.status', 'accepted')
                    ->withPivot('status', 'request_sent_at', 'responded_at')
                    ->withTimestamps();
    }

    /**
     * Pending friend requests for this user (received requests only)
     * 
     * @return HasMany<Friend> Collection of pending friend requests received by this user
     */
    public function pendingFriendRequests(): HasMany
    {
        return $this->hasMany(Friend::class, 'friend_id')
                    ->where('status', 'pending');
    }

    /**
     * Popup notifications created by this user (admin)
     */
    public function createdPopupNotifications(): HasMany
    {
        return $this->hasMany(PopupNotification::class, 'created_by');
    }

    /**
     * Popup notifications seen by this user
     */
    public function seenPopupNotifications(): BelongsToMany
    {
        return $this->belongsToMany(PopupNotification::class, 'user_popup_notifications')
                    ->withPivot('seen_at')
                    ->withTimestamps();
    }

    /**
     * Credit transfers sent by this user
     */
    public function sentCreditTransfers(): HasMany
    {
        return $this->hasMany(\App\Models\CreditTransfer::class, 'sender_id');
    }

    /**
     * Credit transfers received by this user
     */
    public function receivedCreditTransfers(): HasMany
    {
        return $this->hasMany(\App\Models\CreditTransfer::class, 'recipient_id');
    }

    // Helper methods

    /**
     * Check if user can create challenges
     */
    public function canCreateChallenges(): bool
    {
        $level = Level::where('min_xp', '<=', $this->xp)
                     ->orderBy('min_xp', 'desc')
                     ->first();
        
        return $level && $level->can_create_challenges;
    }

    /**
     * Check if user can host free tournaments
     */
    public function canHostFreeTournaments(): bool
    {
        $level = Level::where('min_xp', '<=', $this->xp)
                     ->orderBy('min_xp', 'desc')
                     ->first();
        
        return $level && $level->can_host_free_tournaments;
    }

    /**
     * Check if user can host paid tournaments
     */
    public function canHostPaidTournaments(): bool
    {
        $level = Level::where('min_xp', '<=', $this->xp)
                     ->orderBy('min_xp', 'desc')
                     ->first();
        
        return $level && $level->can_host_paid_tournaments && $this->kyc_status === 'verified';
    }

    /**
     * Role-based permissions for new system
     */
    
    /**
     * Check if user can create challenges with given amount (enhanced for new system)
     */
    public function canCreateChallengeWithAmount(float $amount): bool
    {
        // Admin cannot create challenges according to requirements
        if ($this->role === UserRole::ADMIN || $this->is_organizer) {
            return false;
        }
        
        // Use new level-based system first
        if ($this->canCreateChallengeWithAmountNewSystem($amount)) {
            return true;
        }
        
        // Fall back to old role-based system for backward compatibility
        switch ($this->role->value) {
            case 'basic_user':
                return $amount <= 20000; // Max 20,000 naira
            case 'premium_user':
                return $amount <= 200000; // Max 200,000 naira
            case 'creator':
                return true; // No limit
            default:
                return false;
        }
    }

    /**
     * Check if user can create tournaments (enhanced for new system)
     */
    public function canCreateTournaments(): bool
    {
        // Use new system first, fall back to old role-based system
        if ($this->canCreateTournamentsNewSystem()) {
            return true;
        }
        
        // Old role-based system for backward compatibility
        return in_array($this->role->value, ['creator', 'admin', 'premium_user']);
    }

    /**
     * Check if user can post blogs
     */
    public function canPostBlogs(): bool
    {
        return $this->role === UserRole::CREATOR || $this->is_organizer;
    }

    /**
     * Get maximum challenge amount for this user's role
     */
    public function getMaxChallengeAmount(): ?float
    {
        switch ($this->role->value) {
            case 'basic_user':
                return 20000;
            case 'premium_user':
                return 200000;
            case 'creator':
                return null; // No limit
            default:
                return 0;
        }
    }

    /**
     * Get current level information based on XP and games played
     */
    public function getCurrentLevel(): ?Level
    {
        return Level::where('min_xp', '<=', $this->xp)
                   ->where('min_games_played', '<=', $this->games_played)
                   ->orderBy('min_xp', 'desc')
                   ->first();
    }

    /**
     * Current level relationship for eager loading
     */
    public function currentLevel()
    {
        return $this->hasOne(Level::class, 'id', 'level');
    }

    /**
     * Get next level information based on XP and games played
     */
    public function getNextLevel(): ?Level
    {
        return Level::where('min_xp', '>', $this->xp)
                     ->where('min_games_played', '>', $this->games_played)
                     ->orderBy('min_xp', 'asc')
                     ->first();
    }

    // Friend-related helper methods

    /**
     * Send a friend request to another user
     * 
     * Creates a friend request between this user and the target user.
     * Prevents self-friending and duplicate requests.
     * 
     * @param User $user The user to send a friend request to
     * @param string|null $message Optional message to include with the request
     * @return Friend|null Returns the created Friend record or null if failed/already exists
     */
    public function sendFriendRequest(User $user, string $message = null): ?Friend
    {
        // Prevent self-friending
        if ($this->id === $user->id) {
            return null;
        }

        // Check if friendship already exists
        if ($this->isFriendsWith($user) || $this->hasPendingRequestWith($user)) {
            return null;
        }

        return Friend::create([
            'user_id' => $this->id,
            'friend_id' => $user->id,
            'message' => $message,
            'request_sent_at' => now(),
        ]);
    }

    /**
     * Check if user is friends with another user
     * 
     * Checks both directions of friendship since the relationship can be stored
     * in either direction in the friends table.
     * 
     * @param User $user The user to check friendship with
     * @return bool True if users are friends (status = 'accepted')
     */
    public function isFriendsWith(User $user): bool
    {
        return Friend::where(function ($query) use ($user) {
            $query->where('user_id', $this->id)->where('friend_id', $user->id);
        })->orWhere(function ($query) use ($user) {
            $query->where('user_id', $user->id)->where('friend_id', $this->id);
        })->where('status', 'accepted')->exists();
    }

    /**
     * Check if there's a pending friend request between users
     * 
     * Checks both directions to see if there's any pending friend request
     * between this user and the target user.
     * 
     * @param User $user The user to check pending requests with
     * @return bool True if there's a pending friend request in either direction
     */
    public function hasPendingRequestWith(User $user): bool
    {
        return Friend::where(function ($query) use ($user) {
            $query->where('user_id', $this->id)->where('friend_id', $user->id);
        })->orWhere(function ($query) use ($user) {
            $query->where('user_id', $user->id)->where('friend_id', $this->id);
        })->where('status', 'pending')->exists();
    }

    /**
     * Check if user has played with another user before
     * 
     * Determines if two users have interacted in completed challenges or tournaments.
     * This is used to enforce the rule that users can only befriend players they've
     * actually played with.
     * 
     * @param User $user The user to check play history with
     * @return bool True if users have played together in challenges or tournaments
     */
    public function hasPlayedWith(User $user): bool
    {
        // Check challenges where they played against each other
        $challengePlayed = Challenge::where(function ($query) use ($user) {
            $query->where('creator_id', $this->id)->where('accepter_id', $user->id);
        })->orWhere(function ($query) use ($user) {
            $query->where('creator_id', $user->id)->where('accepter_id', $this->id);
        })->where('status', 'completed')->exists();

        // Check tournaments where they both participated
        $tournamentPlayed = Tournament::whereHas('participants', function ($query) {
            $query->where('user_id', $this->id);
        })->whereHas('participants', function ($query) use ($user) {
            $query->where('user_id', $user->id);
        })->where('status', 'completed')->exists();

        return $challengePlayed || $tournamentPlayed;
    }

    /**
     * Get mutual friends with another user
     * 
     * Finds users who are friends with both this user and the target user.
     * Useful for displaying connection information and friend suggestions.
     * 
     * @param User $user The user to find mutual friends with
     * @return \Illuminate\Database\Eloquent\Collection<User> Collection of mutual friends
     */
    public function getMutualFriends(User $user)
    {
        $myFriends = $this->friends()->pluck('users.id');
        $theirFriends = $user->friends()->pluck('users.id');
        
        $mutualFriendIds = $myFriends->intersect($theirFriends);
        
        return User::whereIn('id', $mutualFriendIds)->get();
    }

    /**
     * Get friend requests count
     * 
     * Returns the number of pending friend requests this user has received.
     * Used for displaying notification badges in the UI.
     * 
     * @return int Number of pending friend requests
     */
    public function getFriendRequestsCount(): int
    {
        return $this->pendingFriendRequests()->count();
    }

    /**
     * Get user's current rank based on XP
     * Fix: Implement getRank() method to support leaderboard display
     */
    public function getRank(): int
    {
        // If a precomputed rank attribute is set (e.g., via eager loading), use it to avoid N+1 queries
        if (array_key_exists('rank', $this->attributes)) {
            return $this->attributes['rank'];
        }
        // Otherwise, fall back to the query-based calculation
        $higherRankedCount = self::where('status', 'active')
            ->where('xp', '>', $this->xp)
            ->count();
        
        // Rank is count of users with higher XP + 1
        return $higherRankedCount + 1;
    }

    // PIN-related helper methods

    /**
     * Check if user has set a transaction PIN
     */
    public function hasTransactionPin(): bool
    {
        return !is_null($this->transaction_pin);
    }

    /**
     * Set transaction PIN
     */
    public function setTransactionPin(string $pin): void
    {
        $this->update([
            'transaction_pin' => $pin,
            'pin_set_at' => now(),
        ]);
    }

    /**
     * Verify transaction PIN
     */
    public function verifyTransactionPin(string $pin): bool
    {
        if (!$this->hasTransactionPin()) {
            return false;
        }

        return Hash::check($pin, $this->transaction_pin);
    }

    /**
     * Can transfer credits (has PIN and sufficient balance)
     */
    public function canTransferCredits(float $amount): bool
    {
        return $this->hasTransactionPin() && 
               $this->wallet_balance >= $amount && 
               $amount > 0;
    }

    // Enhanced Level System Methods

    /**
     * Add games played and potentially level up
     */
    public function addGamesPlayed(int $count = 1): void
    {
        $this->increment('games_played', $count);
        $this->checkLevelUp();
    }

    /**
     * Get maximum wager amount for user's current level
     */
    public function getMaxWagerAmount(): ?float
    {
        // Check role-based permissions first
        if ($this->role === UserRole::CREATOR || $this->role === UserRole::ADMIN || $this->is_organizer) {
            return null; // No limit
        }

        $level = $this->getCurrentLevel();
        return $level ? $level->max_wager_amount : 0;
    }

    /**
     * Check if user can create tournaments (new level system)
     */
    public function canCreateTournamentsNewSystem(): bool
    {
        // Admin/organizer can always create
        if ($this->is_organizer || $this->role === UserRole::ADMIN) {
            return true;
        }

        // Creator role can create
        if ($this->role === UserRole::CREATOR) {
            return true;
        }

        // Level 29 and 30 can create
        $level = $this->getCurrentLevel();
        if ($level) {
            // Get level number from name (e.g., "Level 29" -> 29)
            if (preg_match('/Level (\d+)/', $level->name, $matches)) {
                $levelNumber = (int)$matches[1];
                return $levelNumber >= 29;
            }
        }

        return false;
    }

    /**
     * Check if user can create challenges with given amount
     */
    public function canCreateChallengeWithAmountNewSystem(float $amount): bool
    {
        // Admin cannot create challenges according to requirements
        if ($this->role === UserRole::ADMIN || $this->is_organizer) {
            return false;
        }
        
        $maxAmount = $this->getMaxWagerAmount();
        
        // No limit for creators
        if ($maxAmount === null) {
            return true;
        }

        return $amount <= $maxAmount;
    }

    // Creator Request System Methods

    /**
     * Request Creator status
     */
    public function requestCreatorStatus(string $reason): bool
    {
        if ($this->creator_request_status !== 'none') {
            return false; // Already has a pending/approved request
        }

        $this->update([
            'creator_request_status' => 'pending',
            'creator_request_date' => now(),
            'creator_request_reason' => $reason,
        ]);

        return true;
    }

    /**
     * Approve Creator request (admin only)
     */
    public function approveCreatorRequest(): bool
    {
        if ($this->creator_request_status !== 'pending') {
            return false;
        }

        $this->update([
            'creator_request_status' => 'approved',
            'role' => 'creator',
        ]);

        return true;
    }

    /**
     * Reject Creator request (admin only)
     */
    public function rejectCreatorRequest(): bool
    {
        if ($this->creator_request_status !== 'pending') {
            return false;
        }

        $this->update([
            'creator_request_status' => 'rejected',
        ]);

        return true;
    }

    /**
     * Check if user can request Creator status
     */
    public function canRequestCreatorStatus(): bool
    {
        return $this->creator_request_status === 'none' && 
               $this->role !== 'creator' && 
               !$this->is_organizer;
    }

    /**
     * Check if user has pending Creator request
     */
    public function hasPendingCreatorRequest(): bool
    {
        return $this->creator_request_status === 'pending';
    }

    /**
     * Check if user is Creator level
     */
    public function isCreator(): bool
    {
        return $this->role === UserRole::CREATOR;
    }

    /**
     * Check if user is Admin
     */
    public function isAdmin(): bool
    {
        return $this->is_organizer;
    }

    // New Role System Methods

    /**
     * Set user role and auto-adjust level if needed
     */
    public function setRole(UserRole $role): void
    {
        $this->role = $role;
        
        // Auto-assign level based on role
        $autoLevel = $role->getAutoAssignedLevel();
        if ($autoLevel !== null) {
            $this->level = $autoLevel;
        } elseif ($role === UserRole::CREATOR) {
            // Use cached Creator level ID to avoid database query
            $this->level = static::getCreatorLevelId();
        }
        
        $this->save();
    }

    /**
     * Get cached Creator level ID to avoid repeated database queries
     */
    private static function getCreatorLevelId(): ?int
    {
        return \Illuminate\Support\Facades\Cache::remember('creator_level_id', 3600, function () {
            $creatorLevel = Level::where('name', 'Creator')->first();
            return $creatorLevel ? $creatorLevel->id : null;
        });
    }

    /**
     * Check if user has a specific role
     */
    public function hasRole(UserRole $role): bool
    {
        return $this->role === $role;
    }

    /**
     * Check if user is basic user
     */
    public function isBasicUser(): bool
    {
        return $this->role === UserRole::BASIC_USER;
    }

    /**
     * Check if user is premium user
     */
    public function isPremiumUser(): bool
    {
        return $this->role === UserRole::PREMIUM_USER;
    }

    /**
     * Check if user is moderator
     */
    public function isModerator(): bool
    {
        return $this->role === UserRole::MODERATOR;
    }

    /**
     * Check if user can manage disputes (moderator or admin)
     */
    public function canManageDisputes(): bool
    {
        return $this->role->canManageDisputes() || $this->is_organizer;
    }

    /**
     * Auto-promote user based on level progression
     */
    public function checkRolePromotion(): void
    {
        // Don't auto-promote moderators or admins
        if (in_array($this->role, [UserRole::MODERATOR, UserRole::ADMIN])) {
            return;
        }

        // Auto-promote to Premium User when reaching level 29 or 30
        if ($this->level >= 29 && $this->role !== UserRole::PREMIUM_USER && $this->role !== UserRole::CREATOR) {
            $this->role = UserRole::PREMIUM_USER;
            $this->save();
        }
    }

    /**
     * Override the addXP method to include role promotion check
     */
    public function addXP(int $amount): void
    {
        $this->increment('xp', $amount);
        
        $newLevel = $this->getCurrentLevel();
        if ($newLevel && $newLevel->id > $this->level) {
            $this->update(['level' => $newLevel->id]);
            $this->checkRolePromotion();
        }
    }

    /**
     * Override checkLevelUp to include role promotion
     */
    public function checkLevelUp(): void
    {
        $currentLevel = $this->getCurrentLevel();
        if (
            $currentLevel &&
            $currentLevel->id > $this->level &&
            isset($currentLevel->min_xp, $currentLevel->min_games_played) &&
            $this->xp >= $currentLevel->min_xp &&
            $this->games_played >= $currentLevel->min_games_played
        ) {
            $this->update(['level' => $currentLevel->id]);
            $this->checkRolePromotion();
        }
    }

    /**
     * Get disputes assigned to this moderator
     */
    public function moderatedDisputes(): HasMany
    {
        return $this->hasMany(Dispute::class, 'moderator_id');
    }

    /**
     * Get pending disputes assigned to this moderator
     */
    public function pendingModeratedDisputes(): HasMany
    {
        return $this->hasMany(Dispute::class, 'moderator_id')
                    ->where('status', 'open');
    }

    /**
     * Check if user can be assigned disputes (is moderator and active)
     */
    public function canBeAssignedDisputes(): bool
    {
        return $this->role === UserRole::MODERATOR && $this->status === 'active';
    }

    /**
     * Get moderator's current dispute load
     */
    public function getDisputeLoad(): int
    {
        return $this->pendingModeratedDisputes()->count();
    }
}