<?php

namespace App\Notifications;

use App\Models\Challenge;
use App\Models\User;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class ChallengeInvitation extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(
        public Challenge $challenge,
        public User $inviter,
        public ?string $message = null
    ) {}

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject("Challenge Invitation: {$this->challenge->title}")
            ->greeting("Hello {$notifiable->username}!")
            ->line("You've been invited to a challenge by {$this->inviter->username}.")
            ->line("Challenge Details:")
            ->line("• Game: {$this->challenge->game->name}")
            ->line("• Wager: ₦" . number_format($this->challenge->wager, 2))
            ->line("• Mode: " . ucfirst($this->challenge->mode))
            ->when($this->message, function ($mailMessage) {
                return $mailMessage->line("Message from {$this->inviter->username}: \"{$this->message}\"");
            })
            ->action('View Challenge', route('challenges.show', $this->challenge))
            ->line('Accept the challenge and prove your skills!')
            ->salutation('Best regards, The BattleMania Team');
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'challenge_invitation',
            'challenge_id' => $this->challenge->id,
            'challenge_title' => $this->challenge->title,
            'inviter_id' => $this->inviter->id,
            'inviter_username' => $this->inviter->username,
            'message' => $this->message,
            'wager' => $this->challenge->wager,
            'game_name' => $this->challenge->game->name,
        ];
    }
}
