<?php

namespace App\Notifications;

use App\Models\Tournament;
use App\Models\User;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class TournamentInvitation extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(
        public Tournament $tournament,
        public User $inviter,
        public ?string $message = null
    ) {}

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject("Tournament Invitation: {$this->tournament->name}")
            ->greeting("Hello {$notifiable->username}!")
            ->line("You've been invited to join the tournament \"{$this->tournament->name}\" by {$this->inviter->username}.")
            ->line("Tournament Details:")
            ->line("• Game: {$this->tournament->game->name}")
            ->line("• Entry Fee: ₦" . number_format($this->tournament->entry_fee, 2))
            ->line("• Prize Pool: ₦" . number_format($this->tournament->prize_pool, 2))
            ->line("• Start Date: " . $this->tournament->start_date->format('M j, Y g:i A'))
            ->when($this->message, function ($mailMessage) {
                return $mailMessage->line("Message from {$this->inviter->username}: \"{$this->message}\"");
            })
            ->action('View Tournament', route('tournaments.show', $this->tournament))
            ->line('Join now and compete for exciting prizes!')
            ->salutation('Best regards, The BattleMania Team');
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'tournament_invitation',
            'tournament_id' => $this->tournament->id,
            'tournament_name' => $this->tournament->name,
            'inviter_id' => $this->inviter->id,
            'inviter_username' => $this->inviter->username,
            'message' => $this->message,
            'entry_fee' => $this->tournament->entry_fee,
            'prize_pool' => $this->tournament->prize_pool,
            'start_date' => $this->tournament->start_date,
        ];
    }
}
