<?php

namespace App\Policies;

use App\Models\Challenge;
use App\Models\User;
use Illuminate\Auth\Access\Response;

class ChallengePolicy
{
    /**
     * Determine whether the user can view any models.
     */
    public function viewAny(User $user): bool
    {
        return true; // Anyone can view challenges
    }

    /**
     * Determine whether the user can view the model.
     */
    public function view(User $user, Challenge $challenge): bool
    {
        return true; // Anyone can view a challenge
    }

    /**
     * Determine whether the user can create models.
     * Allow all users to create challenges - level and amount limits are enforced in controller
     */
    public function create(User $user): bool
    {
        // All authenticated users can create challenges regardless of role/level
        // The specific amount limits and level requirements are handled in the controller
        // Admin users cannot create challenges as per business rules
        return !($user->role === \App\Enums\UserRole::ADMIN || $user->is_organizer);
    }

    /**
     * Determine whether the user can update the model.
     */
    public function update(User $user, Challenge $challenge): bool
    {
        // Only creator of the challenge can update it, or admins
        return $user->id === $challenge->creator_id || $user->is_organizer;
    }

    /**
     * Determine whether the user can delete the model.
     */
    public function delete(User $user, Challenge $challenge): bool
    {
        // Only creator of the challenge can delete it, or admins
        return $user->id === $challenge->creator_id || $user->is_organizer;
    }

    /**
     * Determine whether the user can restore the model.
     */
    public function restore(User $user, Challenge $challenge): bool
    {
        return $user->is_organizer;
    }

    /**
     * Determine whether the user can permanently delete the model.
     */
    public function forceDelete(User $user, Challenge $challenge): bool
    {
        return $user->is_organizer;
    }
}
