<?php

namespace App\Services;

use App\Models\Badge;
use App\Models\User;
use Illuminate\Support\Facades\Log; // Added: Import Laravel's Log facade

class BadgeService
{
    /**
     * Check and award win-related badges.
     */
    public function checkWinBadges(User $user): void
    {
        $totalWins = $this->getTotalWins($user);

        $winBadges = [
            ['name' => 'First Victory', 'wins_required' => 1],
            ['name' => 'Rising Champion', 'wins_required' => 10],
            ['name' => 'Veteran Fighter', 'wins_required' => 50],
            ['name' => 'Gaming Legend', 'wins_required' => 100],
        ];

        foreach ($winBadges as $badgeData) {
            if ($totalWins >= $badgeData['wins_required']) {
                $this->awardBadge($user, $badgeData['name']);
            }
        }
    }

    /**
     * Check and award tournament-related badges.
     */
    public function checkTournamentBadges(User $user): void
    {
        $tournamentCount = $user->tournamentParticipations()->count();

        $tournamentBadges = [
            ['name' => 'Tournament Rookie', 'tournaments_required' => 1],
            ['name' => 'Tournament Regular', 'tournaments_required' => 10],
            ['name' => 'Tournament Master', 'tournaments_required' => 25],
        ];

        foreach ($tournamentBadges as $badgeData) {
            if ($tournamentCount >= $badgeData['tournaments_required']) {
                $this->awardBadge($user, $badgeData['name']);
            }
        }
    }

    /**
     * Check and award participation badges.
     */
    public function checkParticipationBadges(User $user): void
    {
        $totalMatches = $user->createdChallenges()->count() + $user->acceptedChallenges()->count();

        $participationBadges = [
            ['name' => 'First Match', 'matches_required' => 1],
            ['name' => 'Active Player', 'matches_required' => 25],
            ['name' => 'Gaming Addict', 'matches_required' => 100],
        ];

        foreach ($participationBadges as $badgeData) {
            if ($totalMatches >= $badgeData['matches_required']) {
                $this->awardBadge($user, $badgeData['name']);
            }
        }
    }

    /**
     * Check and award special achievement badges.
     */
    public function checkSpecialBadges(User $user): void
    {
        // Perfect streak badge (10 wins without disputes)
        $recentWins = $user->createdChallenges()
            ->where('status', 'completed')
            ->where('result', 'creator_win')
            ->whereDoesntHave('dispute')
            ->latest()
            ->limit(10)
            ->count();

        $recentAcceptedWins = $user->acceptedChallenges()
            ->where('status', 'completed')
            ->where('result', 'accepter_win')
            ->whereDoesntHave('dispute')
            ->latest()
            ->limit(10)
            ->count();

        if ($recentWins >= 10 || $recentAcceptedWins >= 10) {
            $this->awardBadge($user, 'Perfect Streak');
        }

        // High Roller badge (participate in high wager challenges)
        $highWagerChallenges = $user->createdChallenges()
            ->where('wager_amount', '>=', 1000)
            ->count();

        if ($highWagerChallenges >= 5) {
            $this->awardBadge($user, 'High Roller');
        }

        // Verified badge for KYC users
        if ($user->kyc_status === 'verified') {
            $this->awardBadge($user, 'Verified Player');
        }
    }

    /**
     * Award a badge to a user if they don't already have it.
     */
    private function awardBadge(User $user, string $badgeName): void
    {
        $badge = Badge::where('name', $badgeName)->first();
        
        if (!$badge) {
            // Create badge if it doesn't exist
            $badge = Badge::create([
                'name' => $badgeName,
                'description' => $this->getBadgeDescription($badgeName),
                'icon' => $this->getBadgeIcon($badgeName),
            ]);
        }

        // Check if user already has this badge
        if (!$user->badges()->where('badge_id', $badge->id)->exists()) {
            $user->badges()->attach($badge->id);
            
            // Could fire an event here for badge notification
            // Using Laravel's Log facade for proper logging
            Log::info("User {$user->username} earned badge: {$badgeName}");
        }
    }

    /**
     * Get total wins for a user.
     */
    private function getTotalWins(User $user): int
    {
        $creatorWins = $user->createdChallenges()
            ->where('status', 'completed')
            ->where('result', 'creator_win')
            ->count();

        $accepterWins = $user->acceptedChallenges()
            ->where('status', 'completed')
            ->where('result', 'accepter_win')
            ->count();

        return $creatorWins + $accepterWins;
    }

    /**
     * Get badge description based on name.
     */
    private function getBadgeDescription(string $badgeName): string
    {
        $descriptions = [
            'First Victory' => 'Won your first challenge',
            'Rising Champion' => 'Won 10 challenges',
            'Veteran Fighter' => 'Won 50 challenges',
            'Gaming Legend' => 'Won 100 challenges',
            'Tournament Rookie' => 'Participated in your first tournament',
            'Tournament Regular' => 'Participated in 10 tournaments',
            'Tournament Master' => 'Participated in 25 tournaments',
            'First Match' => 'Played your first match',
            'Active Player' => 'Played 25 matches',
            'Gaming Addict' => 'Played 100 matches',
            'Perfect Streak' => 'Won 10 matches in a row without disputes',
            'High Roller' => 'Participated in 5 high-stakes challenges',
            'Verified Player' => 'Completed KYC verification',
        ];

        return $descriptions[$badgeName] ?? 'Achievement unlocked';
    }

    /**
     * Get badge icon based on name.
     */
    private function getBadgeIcon(string $badgeName): string
    {
        $icons = [
            'First Victory' => '🏆',
            'Rising Champion' => '⭐',
            'Veteran Fighter' => '💪',
            'Gaming Legend' => '👑',
            'Tournament Rookie' => '🎯',
            'Tournament Regular' => '🎮',
            'Tournament Master' => '🏅',
            'First Match' => '🎲',
            'Active Player' => '⚡',
            'Gaming Addict' => '🔥',
            'Perfect Streak' => '💯',
            'High Roller' => '💰',
            'Verified Player' => '✅',
        ];

        return $icons[$badgeName] ?? '🏆';
    }
}