<?php

namespace App\Services;

use App\Models\CreditTransfer;
use App\Models\User;
use App\Models\WalletTransaction;
use Illuminate\Support\Facades\DB;
use Exception;

class CreditTransferService
{
    /**
     * Initiate a credit transfer between users
     */
    public function initiateTransfer(
        User $sender,
        User $recipient,
        float $amount,
        string $pin,
        string $note = null
    ): CreditTransfer {
        // Validate inputs
        if ($sender->id === $recipient->id) {
            throw new Exception('Cannot transfer credits to yourself.');
        }

        if (!$sender->verifyTransactionPin($pin)) {
            throw new Exception('Invalid transaction PIN.');
        }

        if (!$sender->canTransferCredits($amount)) {
            throw new Exception('Insufficient balance or invalid amount.');
        }

        if ($amount < 1) {
            throw new Exception('Minimum transfer amount is ₦1.00');
        }

        if ($amount > 100000) {
            throw new Exception('Maximum transfer amount is ₦100,000.00');
        }

        return DB::transaction(function () use ($sender, $recipient, $amount, $note) {
            // Create credit transfer record
            $transfer = CreditTransfer::create([
                'sender_id' => $sender->id,
                'recipient_id' => $recipient->id,
                'amount' => $amount,
                'reference' => CreditTransfer::generateReference(),
                'note' => $note,
                'status' => 'pending',
            ]);

            // Process the transfer immediately
            $this->processTransfer($transfer);

            return $transfer;
        });
    }

    /**
     * Process a credit transfer
     */
    public function processTransfer(CreditTransfer $transfer): void
    {
        if (!$transfer->isPending()) {
            throw new Exception('Transfer is not in pending status.');
        }

        $sender = $transfer->sender;
        $recipient = $transfer->recipient;

        DB::transaction(function () use ($transfer, $sender, $recipient) {
            // Check sender still has sufficient balance
            if ($sender->wallet_balance < $transfer->amount) {
                $transfer->markAsFailed();
                throw new Exception('Insufficient balance to complete transfer.');
            }

            // Deduct from sender
            $sender->decrement('wallet_balance', $transfer->amount);

            // Add to recipient
            $recipient->increment('wallet_balance', $transfer->amount);

            // Create wallet transaction for sender (debit)
            WalletTransaction::create([
                'user_id' => $sender->id,
                'type' => 'transfer',
                'amount' => -$transfer->amount, // Negative for debit
                'status' => 'completed',
                'reference' => $transfer->reference,
                'description' => "Transfer to {$recipient->username}",
                'metadata' => [
                    'transfer_id' => $transfer->id,
                    'recipient_id' => $recipient->id,
                    'recipient_username' => $recipient->username,
                    'note' => $transfer->note,
                ],
            ]);

            // Create wallet transaction for recipient (credit)
            WalletTransaction::create([
                'user_id' => $recipient->id,
                'type' => 'transfer_received',
                'amount' => $transfer->amount, // Positive for credit
                'status' => 'completed',
                'reference' => $transfer->reference,
                'description' => "Transfer from {$sender->username}",
                'metadata' => [
                    'transfer_id' => $transfer->id,
                    'sender_id' => $sender->id,
                    'sender_username' => $sender->username,
                    'note' => $transfer->note,
                ],
            ]);

            // Mark transfer as completed
            $transfer->markAsCompleted();
        });
    }

    /**
     * Get transfer history for a user
     */
    public function getTransferHistory(User $user, int $perPage = 20)
    {
        return CreditTransfer::where(function ($query) use ($user) {
            $query->where('sender_id', $user->id)
                  ->orWhere('recipient_id', $user->id);
        })
        ->with(['sender', 'recipient'])
        ->latest()
        ->paginate($perPage);
    }

    /**
     * Get transfer statistics for a user
     */
    public function getTransferStats(User $user): array
    {
        $sent = $user->sentCreditTransfers()
                    ->where('status', 'completed')
                    ->sum('amount');

        $received = $user->receivedCreditTransfers()
                         ->where('status', 'completed')
                         ->sum('amount');

        $totalTransfers = $user->sentCreditTransfers()
                              ->where('status', 'completed')
                              ->count() +
                          $user->receivedCreditTransfers()
                              ->where('status', 'completed')
                              ->count();

        return [
            'total_sent' => $sent,
            'total_received' => $received,
            'total_transfers' => $totalTransfers,
            'net_transfer' => $received - $sent,
        ];
    }
}