<?php

namespace App\Services;

use App\Models\Level;
use Illuminate\Support\Collection;

class LevelService
{
    /**
     * Get descriptive title for a level number.
     */
    public function getLevelTitle(int $levelNumber): string
    {
        return match (true) {
            $levelNumber === 1 => 'Beginner',
            $levelNumber <= 5 => 'Novice',
            $levelNumber <= 10 => 'Intermediate',
            $levelNumber <= 15 => 'Advanced',
            $levelNumber <= 20 => 'Expert',
            $levelNumber <= 25 => 'Master',
            $levelNumber === 30 => 'Legend',
            $levelNumber > 25 => 'Elite',
            default => 'Player'
        };
    }

    /**
     * Get complete display name for a level (e.g., "Level 5 - Novice").
     */
    public function getLevelDisplayName(Level $level): string
    {
        if ($level->name === 'Creator') {
            return 'Creator - Content & Tournament Host';
        }

        $levelNumber = (int) filter_var($level->name, FILTER_SANITIZE_NUMBER_INT);
        $levelTitle = $this->getLevelTitle($levelNumber);

        return "Level {$levelNumber} - {$levelTitle}";
    }

    /**
     * Get complete display name with wager limits for a level.
     */
    public function getLevelDisplayNameWithLimits(Level $level): string
    {
        $displayName = $this->getLevelDisplayName($level);

        if ($level->max_wager_amount) {
            return "{$displayName} (₦".number_format($level->max_wager_amount).' limit)';
        }

        return "{$displayName} (No limit)";
    }

    /**
     * Process levels collection to include display names.
     */
    public function processLevelsForDropdown(Collection $levels): Collection
    {
        return $levels->map(function (Level $level) {
            $level->display_name = $this->getLevelDisplayName($level);
            $level->display_name_with_limits = $this->getLevelDisplayNameWithLimits($level);

            return $level;
        });
    }

    /**
     * Extract level number from level name.
     */
    public function extractLevelNumber(string $levelName): int
    {
        return (int) filter_var($levelName, FILTER_SANITIZE_NUMBER_INT);
    }
}
