<?php

namespace App\Services;

use App\Models\NotificationProvider;
use App\Models\NotificationSetting;
use Exception;
use Illuminate\Support\Facades\Log;

class NotificationService
{
    /**
     * Send an email notification.
     */
    public function sendEmail($to, $subject, $body, $from = null)
    {
        if (!NotificationSetting::get('email_enabled', true)) {
            Log::info('Email notifications are disabled');
            return false;
        }

        $providers = NotificationProvider::getActiveProviders('email');
        
        if ($providers->isEmpty()) {
            Log::error('No active email providers configured');
            return false;
        }

        foreach ($providers as $provider) {
            try {
                $success = $this->sendEmailViaProvider($provider, $to, $subject, $body, $from);
                
                if ($success) {
                    Log::info("Email sent successfully via {$provider->name}", [
                        'provider' => $provider->name,
                        'to' => $to,
                        'subject' => $subject,
                    ]);
                    return true;
                }
            } catch (Exception $e) {
                Log::error("Failed to send email via {$provider->name}: {$e->getMessage()}", [
                    'provider' => $provider->name,
                    'to' => $to,
                    'subject' => $subject,
                    'error' => $e->getMessage(),
                ]);
                
                // Continue to next provider
                continue;
            }
        }

        Log::error('All email providers failed', [
            'to' => $to,
            'subject' => $subject,
        ]);

        return false;
    }

    /**
     * Send an SMS notification.
     */
    public function sendSMS($to, $message, $from = null)
    {
        if (!NotificationSetting::get('sms_enabled', true)) {
            Log::info('SMS notifications are disabled');
            return false;
        }

        $providers = NotificationProvider::getActiveProviders('sms');
        
        if ($providers->isEmpty()) {
            Log::error('No active SMS providers configured');
            return false;
        }

        foreach ($providers as $provider) {
            try {
                $success = $this->sendSMSViaProvider($provider, $to, $message, $from);
                
                if ($success) {
                    Log::info("SMS sent successfully via {$provider->name}", [
                        'provider' => $provider->name,
                        'to' => $to,
                        'message_length' => strlen($message),
                    ]);
                    return true;
                }
            } catch (Exception $e) {
                Log::error("Failed to send SMS via {$provider->name}: {$e->getMessage()}", [
                    'provider' => $provider->name,
                    'to' => $to,
                    'error' => $e->getMessage(),
                ]);
                
                // Continue to next provider
                continue;
            }
        }

        Log::error('All SMS providers failed', [
            'to' => $to,
            'message_length' => strlen($message),
        ]);

        return false;
    }

    /**
     * Send email via a specific provider.
     */
    protected function sendEmailViaProvider(NotificationProvider $provider, $to, $subject, $body, $from = null)
    {
        if (!$provider->isConfigurationValid()) {
            throw new Exception("Provider {$provider->name} configuration is invalid");
        }

        return match($provider->name) {
            'resend' => $this->sendEmailViaResend($provider, $to, $subject, $body, $from),
            'mailgun' => $this->sendEmailViaMailgun($provider, $to, $subject, $body, $from),
            'sendgrid' => $this->sendEmailViaSendGrid($provider, $to, $subject, $body, $from),
            default => throw new Exception("Unsupported email provider: {$provider->name}"),
        };
    }

    /**
     * Send SMS via a specific provider.
     */
    protected function sendSMSViaProvider(NotificationProvider $provider, $to, $message, $from = null)
    {
        if (!$provider->isConfigurationValid()) {
            throw new Exception("Provider {$provider->name} configuration is invalid");
        }

        return match($provider->name) {
            'termii' => $this->sendSMSViaTermii($provider, $to, $message, $from),
            'twilio' => $this->sendSMSViaTwilio($provider, $to, $message, $from),
            'nexmo' => $this->sendSMSViaNexmo($provider, $to, $message, $from),
            default => throw new Exception("Unsupported SMS provider: {$provider->name}"),
        };
    }

    /**
     * Send email via Resend.
     */
    protected function sendEmailViaResend(NotificationProvider $provider, $to, $subject, $body, $from = null)
    {
        $client = new \GuzzleHttp\Client();
        $config = $provider->configuration;
        $from = $from ?: NotificationSetting::get('default_email_from', 'noreply@battlemania.com');

        $response = $client->post('https://api.resend.com/emails', [
            'headers' => [
                'Authorization' => 'Bearer ' . $config['api_key'],
                'Content-Type' => 'application/json',
            ],
            'json' => [
                'from' => $from,
                'to' => [$to],
                'subject' => $subject,
                'html' => $body,
            ],
        ]);

        return $response->getStatusCode() === 200;
    }

    /**
     * Send email via Mailgun.
     */
    protected function sendEmailViaMailgun(NotificationProvider $provider, $to, $subject, $body, $from = null)
    {
        $client = new \GuzzleHttp\Client();
        $config = $provider->configuration;
        $from = $from ?: NotificationSetting::get('default_email_from', 'noreply@battlemania.com');

        $response = $client->post("https://{$config['endpoint']}/v3/{$config['domain']}/messages", [
            'auth' => ['api', $config['secret']],
            'form_params' => [
                'from' => $from,
                'to' => $to,
                'subject' => $subject,
                'html' => $body,
            ],
        ]);

        return $response->getStatusCode() === 200;
    }

    /**
     * Send email via SendGrid.
     */
    protected function sendEmailViaSendGrid(NotificationProvider $provider, $to, $subject, $body, $from = null)
    {
        $client = new \GuzzleHttp\Client();
        $config = $provider->configuration;
        $from = $from ?: NotificationSetting::get('default_email_from', 'noreply@battlemania.com');

        $response = $client->post('https://api.sendgrid.com/v3/mail/send', [
            'headers' => [
                'Authorization' => 'Bearer ' . $config['api_key'],
                'Content-Type' => 'application/json',
            ],
            'json' => [
                'personalizations' => [
                    [
                        'to' => [['email' => $to]],
                        'subject' => $subject,
                    ],
                ],
                'from' => ['email' => $from],
                'content' => [
                    [
                        'type' => 'text/html',
                        'value' => $body,
                    ],
                ],
            ],
        ]);

        return $response->getStatusCode() === 202;
    }

    /**
     * Send SMS via Termii.
     */
    protected function sendSMSViaTermii(NotificationProvider $provider, $to, $message, $from = null)
    {
        $client = new \GuzzleHttp\Client();
        $config = $provider->configuration;

        $response = $client->post('https://api.ng.termii.com/api/sms/send', [
            'headers' => [
                'Content-Type' => 'application/json',
            ],
            'json' => [
                'to' => $to,
                'from' => $config['sender_id'],
                'sms' => $message,
                'type' => 'plain',
                'channel' => 'generic',
                'api_key' => $config['api_key'],
            ],
        ]);

        $data = json_decode($response->getBody(), true);
        return isset($data['message_id']);
    }

    /**
     * Send SMS via Twilio.
     */
    protected function sendSMSViaTwilio(NotificationProvider $provider, $to, $message, $from = null)
    {
        $client = new \GuzzleHttp\Client();
        $config = $provider->configuration;
        $from = $from ?: $config['from_number'];

        $response = $client->post("https://api.twilio.com/2010-04-01/Accounts/{$config['account_sid']}/Messages.json", [
            'auth' => [$config['account_sid'], $config['auth_token']],
            'form_params' => [
                'From' => $from,
                'To' => $to,
                'Body' => $message,
            ],
        ]);

        return $response->getStatusCode() === 201;
    }

    /**
     * Send SMS via Nexmo (Vonage).
     */
    protected function sendSMSViaNexmo(NotificationProvider $provider, $to, $message, $from = null)
    {
        $client = new \GuzzleHttp\Client();
        $config = $provider->configuration;
        $from = $from ?: $config['from_number'];

        $response = $client->post('https://rest.nexmo.com/sms/json', [
            'form_params' => [
                'api_key' => $config['api_key'],
                'api_secret' => $config['api_secret'],
                'from' => $from,
                'to' => $to,
                'text' => $message,
            ],
        ]);

        $data = json_decode($response->getBody(), true);
        return isset($data['messages'][0]['status']) && $data['messages'][0]['status'] === '0';
    }

    /**
     * Test a provider configuration.
     */
    public function testProvider(NotificationProvider $provider)
    {
        $testEmail = NotificationSetting::get('test_email', 'admin@battlemania.com');
        $testPhone = NotificationSetting::get('test_phone', '+1234567890');

        try {
            if ($provider->type === 'email') {
                $success = $this->sendEmailViaProvider(
                    $provider,
                    $testEmail,
                    'BattleMania Test Email',
                    '<h1>Test Email</h1><p>This is a test email from BattleMania notification system.</p>'
                );
            } else {
                $success = $this->sendSMSViaProvider(
                    $provider,
                    $testPhone,
                    'BattleMania Test SMS: Your notification system is working correctly!'
                );
            }

            $provider->update([
                'test_results' => [
                    'success' => $success,
                    'message' => $success ? 'Test successful' : 'Test failed',
                    'tested_at' => now()->toISOString(),
                ],
                'last_tested_at' => now(),
            ]);

            return $success;
        } catch (Exception $e) {
            $provider->update([
                'test_results' => [
                    'success' => false,
                    'message' => $e->getMessage(),
                    'tested_at' => now()->toISOString(),
                ],
                'last_tested_at' => now(),
            ]);

            return false;
        }
    }
}