<?php

namespace App\Services;

use App\Models\PlatformSetting;
use App\Models\WalletTransaction;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class PaymentService
{
    /**
     * Initialize a deposit transaction with the configured payment provider.
     */
    public function initiateDeposit(WalletTransaction $transaction): array
    {
        $provider = $transaction->provider;

        return match ($provider) {
            'paystack' => $this->initiatePaystackDeposit($transaction),
            'flutterwave' => $this->initiateFlutterwaveDeposit($transaction),
            'paypal' => $this->initiatePayPalDeposit($transaction),
            'stripe' => $this->initiateStripeDeposit($transaction),
            default => throw new \InvalidArgumentException("Unsupported payment provider: {$provider}"),
        };
    }

    /**
     * Process a withdrawal request with the configured payment provider.
     */
    public function processWithdrawal(WalletTransaction $transaction): array
    {
        $provider = $transaction->provider;

        return match ($provider) {
            'paystack' => $this->processPaystackWithdrawal($transaction),
            'flutterwave' => $this->processFlutterwaveWithdrawal($transaction),
            'paypal' => $this->processPayPalWithdrawal($transaction),
            'stripe' => $this->processStripeWithdrawal($transaction),
            default => throw new \InvalidArgumentException("Unsupported payment provider: {$provider}"),
        };
    }

    /**
     * Verify a payment with the configured provider.
     */
    public function verifyPayment(string $provider, string $reference): array
    {
        return match ($provider) {
            'paystack' => $this->verifyPaystackPayment($reference),
            'flutterwave' => $this->verifyFlutterwavePayment($reference),
            'paypal' => $this->verifyPayPalPayment($reference),
            'stripe' => $this->verifyStripePayment($reference),
            default => throw new \InvalidArgumentException("Unsupported payment provider: {$provider}"),
        };
    }

    /**
     * Test a payment provider configuration.
     */
    public function testProvider(string $provider): array
    {
        return match ($provider) {
            'paystack' => $this->testPaystack(),
            'flutterwave' => $this->testFlutterwave(),
            'paypal' => $this->testPayPal(),
            'stripe' => $this->testStripe(),
            default => throw new \InvalidArgumentException("Unsupported payment provider: {$provider}"),
        };
    }

    // Paystack Methods
    private function initiatePaystackDeposit(WalletTransaction $transaction): array
    {
        $publicKey = PlatformSetting::get('paystack_public_key');
        $secretKey = PlatformSetting::get('paystack_secret_key');

        if (!$publicKey || !$secretKey) {
            throw new \Exception('Paystack credentials not configured');
        }

        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $secretKey,
            'Content-Type' => 'application/json',
        ])->post('https://api.paystack.co/transaction/initialize', [
            'email' => $transaction->user->email,
            'amount' => $transaction->amount * 100, // Convert to kobo
            'reference' => $transaction->reference,
            'callback_url' => route('wallet.callback.paystack'),
            'metadata' => [
                'transaction_id' => $transaction->id,
                'user_id' => $transaction->user_id,
            ],
        ]);

        if (!$response->successful()) {
            Log::error('Paystack deposit initiation failed', [
                'response' => $response->json(),
                'transaction_id' => $transaction->id,
            ]);
            throw new \Exception('Failed to initiate deposit with Paystack');
        }

        $data = $response->json();

        return [
            'success' => true,
            'payment_url' => $data['data']['authorization_url'],
            'reference' => $data['data']['reference'],
        ];
    }

    private function verifyPaystackPayment(string $reference): array
    {
        $secretKey = PlatformSetting::get('paystack_secret_key');

        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $secretKey,
        ])->get("https://api.paystack.co/transaction/verify/{$reference}");

        if (!$response->successful()) {
            return ['success' => false, 'message' => 'Verification failed'];
        }

        $data = $response->json();

        return [
            'success' => $data['data']['status'] === 'success',
            'amount' => $data['data']['amount'] / 100, // Convert from kobo
            'reference' => $data['data']['reference'],
            'gateway_response' => $data['data']['gateway_response'],
        ];
    }

    private function testPaystack(): array
    {
        $secretKey = PlatformSetting::get('paystack_secret_key');

        if (!$secretKey) {
            return ['success' => false, 'message' => 'Paystack secret key not configured'];
        }

        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $secretKey,
        ])->get('https://api.paystack.co/bank');

        return [
            'success' => $response->successful(),
            'message' => $response->successful() 
                ? 'Paystack configuration is valid' 
                : 'Paystack configuration failed: ' . $response->body(),
        ];
    }

    // Flutterwave Methods
    private function initiateFlutterwaveDeposit(WalletTransaction $transaction): array
    {
        $publicKey = PlatformSetting::get('flutterwave_public_key');
        $secretKey = PlatformSetting::get('flutterwave_secret_key');

        if (!$publicKey || !$secretKey) {
            throw new \Exception('Flutterwave credentials not configured');
        }

        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $secretKey,
            'Content-Type' => 'application/json',
        ])->post('https://api.flutterwave.com/v3/payments', [
            'tx_ref' => $transaction->reference,
            'amount' => $transaction->amount,
            'currency' => 'NGN',
            'redirect_url' => route('wallet.callback.flutterwave'),
            'customer' => [
                'email' => $transaction->user->email,
                'name' => $transaction->user->username,
            ],
            'customizations' => [
                'title' => 'BattleMania Deposit',
                'description' => 'Wallet deposit',
                'logo' => asset('images/logo.png'),
            ],
        ]);

        if (!$response->successful()) {
            Log::error('Flutterwave deposit initiation failed', [
                'response' => $response->json(),
                'transaction_id' => $transaction->id,
            ]);
            throw new \Exception('Failed to initiate deposit with Flutterwave');
        }

        $data = $response->json();

        return [
            'success' => true,
            'payment_url' => $data['data']['link'],
            'reference' => $transaction->reference,
        ];
    }

    private function verifyFlutterwavePayment(string $reference): array
    {
        $secretKey = PlatformSetting::get('flutterwave_secret_key');

        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $secretKey,
        ])->get("https://api.flutterwave.com/v3/transactions/{$reference}/verify");

        if (!$response->successful()) {
            return ['success' => false, 'message' => 'Verification failed'];
        }

        $data = $response->json();

        return [
            'success' => $data['data']['status'] === 'successful',
            'amount' => $data['data']['amount'],
            'reference' => $data['data']['tx_ref'],
            'gateway_response' => $data['data']['processor_response'],
        ];
    }

    private function testFlutterwave(): array
    {
        $secretKey = PlatformSetting::get('flutterwave_secret_key');

        if (!$secretKey) {
            return ['success' => false, 'message' => 'Flutterwave secret key not configured'];
        }

        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $secretKey,
        ])->get('https://api.flutterwave.com/v3/banks/NG');

        return [
            'success' => $response->successful(),
            'message' => $response->successful() 
                ? 'Flutterwave configuration is valid' 
                : 'Flutterwave configuration failed: ' . $response->body(),
        ];
    }

    // PayPal Methods (simplified implementation)
    private function initiatePayPalDeposit(WalletTransaction $transaction): array
    {
        // PayPal implementation would require OAuth and more complex setup
        // For now, return a placeholder response
        return [
            'success' => false,
            'message' => 'PayPal integration requires additional setup',
        ];
    }

    private function verifyPayPalPayment(string $reference): array
    {
        $clientId = PlatformSetting::get('paypal_client_id');
        $secret = PlatformSetting::get('paypal_secret');
        $mode = PlatformSetting::get('paypal_mode', 'sandbox');

        if (!$clientId || !$secret) {
            return ['success' => false, 'message' => 'PayPal credentials not configured'];
        }

        $baseUrl = $mode === 'live' ? 'https://api-m.paypal.com' : 'https://api-m.sandbox.paypal.com';

        try {
            // Get access token
            $tokenResponse = Http::withBasicAuth($clientId, $secret)
                ->post("{$baseUrl}/v1/oauth2/token", [
                    'grant_type' => 'client_credentials',
                ]);

            if (!$tokenResponse->successful()) {
                return ['success' => false, 'message' => 'Failed to authenticate with PayPal'];
            }

            $accessToken = $tokenResponse->json('access_token');

            // Verify the payment
            $orderResponse = Http::withHeaders([
                'Authorization' => 'Bearer ' . $accessToken,
                'Content-Type' => 'application/json',
            ])->get("{$baseUrl}/v2/checkout/orders/{$reference}");

            if (!$orderResponse->successful()) {
                return ['success' => false, 'message' => 'Failed to verify payment with PayPal'];
            }

            $orderData = $orderResponse->json();

            return [
                'success' => $orderData['status'] === 'COMPLETED',
                'amount' => floatval($orderData['purchase_units'][0]['amount']['value']),
                'reference' => $reference,
                'gateway_response' => $orderData['status'],
            ];
        } catch (\Exception $e) {
            Log::error('PayPal verification error: ' . $e->getMessage());
            return ['success' => false, 'message' => 'PayPal verification failed'];
        }
    }

    private function testPayPal(): array
    {
        $clientId = PlatformSetting::get('paypal_client_id');
        $secret = PlatformSetting::get('paypal_secret');

        if (!$clientId || !$secret) {
            return ['success' => false, 'message' => 'PayPal credentials not configured'];
        }

        return ['success' => true, 'message' => 'PayPal credentials are configured (full integration pending)'];
    }

    // Stripe Methods (simplified implementation)
    private function initiateStripeDeposit(WalletTransaction $transaction): array
    {
        // Stripe implementation would require Stripe SDK
        // For now, return a placeholder response
        return [
            'success' => false,
            'message' => 'Stripe integration requires additional setup',
        ];
    }

    private function verifyStripePayment(string $reference): array
    {
        $secretKey = PlatformSetting::get('stripe_secret_key');

        if (!$secretKey) {
            return ['success' => false, 'message' => 'Stripe secret key not configured'];
        }

        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $secretKey,
            ])->get("https://api.stripe.com/v1/payment_intents/{$reference}");

            if (!$response->successful()) {
                return ['success' => false, 'message' => 'Failed to verify payment with Stripe'];
            }

            $data = $response->json();

            return [
                'success' => $data['status'] === 'succeeded',
                'amount' => $data['amount'] / 100, // Convert from cents
                'reference' => $reference,
                'gateway_response' => $data['status'],
            ];
        } catch (\Exception $e) {
            Log::error('Stripe verification error: ' . $e->getMessage());
            return ['success' => false, 'message' => 'Stripe verification failed'];
        }
    }

    private function testStripe(): array
    {
        $publishableKey = PlatformSetting::get('stripe_publishable_key');
        $secretKey = PlatformSetting::get('stripe_secret_key');

        if (!$publishableKey || !$secretKey) {
            return ['success' => false, 'message' => 'Stripe credentials not configured'];
        }

        return ['success' => true, 'message' => 'Stripe credentials are configured (full integration pending)'];
    }

    // Withdrawal methods (simplified)
    private function processPaystackWithdrawal(WalletTransaction $transaction): array
    {
        $secretKey = PlatformSetting::get('paystack_secret_key');

        if (!$secretKey) {
            return ['success' => false, 'message' => 'Paystack secret key not configured'];
        }

        try {
            // Create transfer recipient first
            $recipientResponse = Http::withHeaders([
                'Authorization' => 'Bearer ' . $secretKey,
                'Content-Type' => 'application/json',
            ])->post('https://api.paystack.co/transferrecipient', [
                'type' => 'nuban',
                'name' => $transaction->user->first_name . ' ' . $transaction->user->last_name,
                'account_number' => $transaction->metadata['account_number'] ?? '',
                'bank_code' => $transaction->metadata['bank_code'] ?? '',
                'currency' => 'NGN',
            ]);

            if (!$recipientResponse->successful()) {
                return ['success' => false, 'message' => 'Failed to create transfer recipient'];
            }

            $recipientCode = $recipientResponse->json('data.recipient_code');

            // Initiate transfer
            $transferResponse = Http::withHeaders([
                'Authorization' => 'Bearer ' . $secretKey,
                'Content-Type' => 'application/json',
            ])->post('https://api.paystack.co/transfer', [
                'source' => 'balance',
                'amount' => $transaction->amount * 100, // Convert to kobo
                'recipient' => $recipientCode,
                'reason' => 'Wallet withdrawal',
                'reference' => $transaction->reference,
            ]);

            if (!$transferResponse->successful()) {
                return ['success' => false, 'message' => 'Transfer initiation failed'];
            }

            return [
                'success' => true,
                'transfer_code' => $transferResponse->json('data.transfer_code'),
                'reference' => $transaction->reference,
            ];
        } catch (\Exception $e) {
            Log::error('Paystack withdrawal error: ' . $e->getMessage());
            return ['success' => false, 'message' => 'Withdrawal processing failed'];
        }
    }

    private function processFlutterwaveWithdrawal(WalletTransaction $transaction): array
    {
        $secretKey = PlatformSetting::get('flutterwave_secret_key');

        if (!$secretKey) {
            return ['success' => false, 'message' => 'Flutterwave secret key not configured'];
        }

        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $secretKey,
                'Content-Type' => 'application/json',
            ])->post('https://api.flutterwave.com/v3/transfers', [
                'account_bank' => $transaction->metadata['bank_code'] ?? '',
                'account_number' => $transaction->metadata['account_number'] ?? '',
                'amount' => $transaction->amount,
                'narration' => 'Wallet withdrawal',
                'currency' => 'NGN',
                'reference' => $transaction->reference,
                'callback_url' => route('wallet.callback', ['provider' => 'flutterwave']),
                'debit_currency' => 'NGN',
            ]);

            if (!$response->successful()) {
                return ['success' => false, 'message' => 'Transfer initiation failed'];
            }

            $data = $response->json();

            return [
                'success' => $data['status'] === 'success',
                'transfer_id' => $data['data']['id'] ?? null,
                'reference' => $transaction->reference,
            ];
        } catch (\Exception $e) {
            Log::error('Flutterwave withdrawal error: ' . $e->getMessage());
            return ['success' => false, 'message' => 'Withdrawal processing failed'];
        }
    }

    private function processPayPalWithdrawal(WalletTransaction $transaction): array
    {
        $clientId = PlatformSetting::get('paypal_client_id');
        $secret = PlatformSetting::get('paypal_secret');
        $mode = PlatformSetting::get('paypal_mode', 'sandbox');

        if (!$clientId || !$secret) {
            return ['success' => false, 'message' => 'PayPal credentials not configured'];
        }

        $baseUrl = $mode === 'live' ? 'https://api-m.paypal.com' : 'https://api-m.sandbox.paypal.com';

        try {
            // Get access token
            $tokenResponse = Http::withBasicAuth($clientId, $secret)
                ->post("{$baseUrl}/v1/oauth2/token", [
                    'grant_type' => 'client_credentials',
                ]);

            if (!$tokenResponse->successful()) {
                return ['success' => false, 'message' => 'Failed to authenticate with PayPal'];
            }

            $accessToken = $tokenResponse->json('access_token');

            // Create payout
            $payoutResponse = Http::withHeaders([
                'Authorization' => 'Bearer ' . $accessToken,
                'Content-Type' => 'application/json',
            ])->post("{$baseUrl}/v1/payments/payouts", [
                'sender_batch_header' => [
                    'sender_batch_id' => $transaction->reference,
                    'email_subject' => 'You have a payout!',
                    'email_message' => 'You have received a payout from BattleMania!'
                ],
                'items' => [
                    [
                        'recipient_type' => 'EMAIL',
                        'amount' => [
                            'value' => number_format($transaction->amount, 2),
                            'currency' => 'USD',
                        ],
                        'receiver' => $transaction->metadata['paypal_email'] ?? $transaction->user->email,
                        'note' => 'Wallet withdrawal',
                        'sender_item_id' => $transaction->reference,
                    ]
                ]
            ]);

            if (!$payoutResponse->successful()) {
                return ['success' => false, 'message' => 'Payout initiation failed'];
            }

            return [
                'success' => true,
                'batch_id' => $payoutResponse->json('batch_header.payout_batch_id'),
                'reference' => $transaction->reference,
            ];
        } catch (\Exception $e) {
            Log::error('PayPal withdrawal error: ' . $e->getMessage());
            return ['success' => false, 'message' => 'PayPal withdrawal failed'];
        }
    }

    private function processStripeWithdrawal(WalletTransaction $transaction): array
    {
        $secretKey = PlatformSetting::get('stripe_secret_key');

        if (!$secretKey) {
            return ['success' => false, 'message' => 'Stripe secret key not configured'];
        }

        try {
            // Create external account (bank account) if not exists
            if (!empty($transaction->metadata['bank_account'])) {
                $accountResponse = Http::withHeaders([
                    'Authorization' => 'Bearer ' . $secretKey,
                ])->post('https://api.stripe.com/v1/accounts', [
                    'type' => 'express',
                    'country' => 'US',
                    'email' => $transaction->user->email,
                ]);

                if (!$accountResponse->successful()) {
                    return ['success' => false, 'message' => 'Failed to create Stripe account'];
                }

                $accountId = $accountResponse->json('id');

                // Create transfer
                $transferResponse = Http::withHeaders([
                    'Authorization' => 'Bearer ' . $secretKey,
                ])->post('https://api.stripe.com/v1/transfers', [
                    'amount' => $transaction->amount * 100, // Convert to cents
                    'currency' => 'usd',
                    'destination' => $accountId,
                    'transfer_group' => $transaction->reference,
                ]);

                if (!$transferResponse->successful()) {
                    return ['success' => false, 'message' => 'Transfer failed'];
                }

                return [
                    'success' => true,
                    'transfer_id' => $transferResponse->json('id'),
                    'reference' => $transaction->reference,
                ];
            }

            return ['success' => false, 'message' => 'Bank account information required for Stripe withdrawals'];
        } catch (\Exception $e) {
            Log::error('Stripe withdrawal error: ' . $e->getMessage());
            return ['success' => false, 'message' => 'Stripe withdrawal failed'];
        }
    }
}