<?php

namespace App\Services;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class QoreIdService
{
    private ?string $baseUrl;
    private ?string $apiKey;
    private ?string $secretKey;

    public function __construct()
    {
        $this->baseUrl = config('services.qoreid.base_url', 'https://api.qoreid.com');
        $this->apiKey = config('services.qoreid.api_key');
        $this->secretKey = config('services.qoreid.secret_key');
    }

    /**
     * Submit KYC verification to QoreID
     */
    public function submitVerification(array $data): array
    {
        if (!$this->apiKey || !$this->secretKey) {
            throw new \Exception('QoreID API credentials not configured');
        }

        $payload = [
            'firstname' => $data['first_name'],
            'lastname' => $data['last_name'],
            'dob' => $data['date_of_birth'],
            'phone' => $data['phone_number'],
            'address' => $data['address'],
            'id_type' => $data['id_type'],
            'id_number' => $data['id_number'],
            'id_image' => $data['id_document_url'],
            'selfie' => $data['selfie_url'],
            'webhook_url' => route('kyc.webhook'),
        ];

        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type' => 'application/json',
            ])->post($this->baseUrl . '/v1/kyc/verify', $payload);

            if (!$response->successful()) {
                throw new \Exception('QoreID API request failed: ' . $response->body());
            }

            $result = $response->json();

            return [
                'success' => true,
                'reference' => $result['data']['reference'] ?? null,
                'status' => $result['data']['status'] ?? 'pending',
                'message' => $result['message'] ?? 'Verification submitted successfully'
            ];

        } catch (\Exception $e) {
            Log::error('QoreID verification submission failed: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Check verification status with QoreID
     */
    public function checkStatus(string $reference): array
    {
        if (!$this->apiKey) {
            throw new \Exception('QoreID API key not configured');
        }

        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type' => 'application/json',
            ])->get($this->baseUrl . '/v1/kyc/status/' . $reference);

            if (!$response->successful()) {
                throw new \Exception('QoreID status check failed: ' . $response->body());
            }

            $result = $response->json();

            return [
                'success' => true,
                'status' => $result['data']['status'] ?? 'pending',
                'message' => $result['message'] ?? 'Status retrieved successfully',
                'details' => $result['data'] ?? []
            ];

        } catch (\Exception $e) {
            Log::error('QoreID status check failed: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Verify webhook signature from QoreID
     */
    public function verifyWebhook(Request $request): bool
    {
        if (!$this->secretKey) {
            return false;
        }

        $signature = $request->header('X-QoreID-Signature');
        $payload = $request->getContent();
        
        $expectedSignature = hash_hmac('sha256', $payload, $this->secretKey);
        
        return hash_equals($expectedSignature, $signature);
    }

    /**
     * Test API connection
     */
    public function testConnection(): array
    {
        if (!$this->apiKey) {
            return [
                'success' => false,
                'message' => 'API key not configured'
            ];
        }

        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type' => 'application/json',
            ])->get($this->baseUrl . '/v1/health');

            if ($response->successful()) {
                return [
                    'success' => true,
                    'message' => 'QoreID API connection successful'
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'QoreID API connection failed: ' . $response->status()
                ];
            }

        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => 'QoreID API connection error: ' . $e->getMessage()
            ];
        }
    }
}