<?php

namespace App\Services;

use App\Enums\UserRole;
use App\Models\User;
use App\Models\Level;
use App\Models\Dispute;
use App\Models\PopupNotification;

class RoleManagementService
{
    /**
     * Change user role and handle level assignment
     */
    public function changeUserRole(User $user, UserRole $newRole): bool
    {
        $oldRole = $user->role;
        
        // Set the new role
        $user->setRole($newRole);
        
        // Log the role change (you might want to create a role_changes table)
        logger()->info("User role changed", [
            'user_id' => $user->id,
            'old_role' => $oldRole->value ?? 'unknown',
            'new_role' => $newRole->value,
            'level_changed_to' => $user->level,
        ]);
        
        return true;
    }

    /**
     * Auto-assign moderators to new disputes
     */
    public function assignDisputeToModerator(Dispute $dispute): bool
    {
        // Skip if already assigned
        if ($dispute->hasAssignedModerator()) {
            return false;
        }

        // Get available moderators
        $moderators = $this->getAvailableModerators();
        
        if ($moderators->isEmpty()) {
            // Log the issue
            logger()->warning("No available moderators for dispute assignment", [
                'dispute_id' => $dispute->id
            ]);
            
            // Create admin notification for urgent attention
            $this->notifyAdminsOfUnassignedDispute($dispute);
            
            // Fallback: Assign to admin if available
            return $this->assignDisputeToAdminFallback($dispute);
        }

        // Find moderator with lowest dispute load
        $selectedModerator = $this->selectBestModerator($moderators);
        
        // Assign the dispute
        $dispute->assignToModerator($selectedModerator);
        
        logger()->info("Dispute assigned to moderator", [
            'dispute_id' => $dispute->id,
            'moderator_id' => $selectedModerator->id,
            'moderator_current_load' => $selectedModerator->getDisputeLoad(),
        ]);

        return true;
    }

    /**
     * Get available moderators
     */
    private function getAvailableModerators()
    {
        return User::where('role', UserRole::MODERATOR->value)
                  ->where('status', 'active')
                  ->get();
    }

    /**
     * Select the best moderator based on current workload
     */
    private function selectBestModerator($moderators): User
    {
        // Calculate current dispute load for each moderator
        $moderatorsWithLoad = $moderators->map(function ($moderator) {
            $moderator->current_load = $moderator->getDisputeLoad();
            return $moderator;
        });

        // Find minimum load
        $minLoad = $moderatorsWithLoad->min('current_load');
        
        // Get all moderators with minimum load
        $availableModerators = $moderatorsWithLoad->where('current_load', $minLoad);
        
        // Return random moderator from those with lowest load
        return $availableModerators->random();
    }

    /**
     * Promote users based on level progression
     */
    public function checkAndPromoteUsers(): int
    {
        $promoted = 0;
        
        // Find users at level 29 or 30 who are still basic users
        $usersToPromote = User::where('level', '>=', 29)
                             ->where('role', UserRole::BASIC_USER->value)
                             ->get();

        foreach ($usersToPromote as $user) {
            $this->changeUserRole($user, UserRole::PREMIUM_USER);
            $promoted++;
        }

        return $promoted;
    }

    /**
     * Get role statistics
     */
    public function getRoleStatistics(): array
    {
        $stats = [];
        
        foreach (UserRole::cases() as $role) {
            $stats[$role->value] = [
                'name' => $role->getDisplayName(),
                'count' => User::where('role', $role->value)->count(),
            ];
        }

        return $stats;
    }

    /**
     * Get moderator workload statistics
     */
    public function getModeratorWorkloadStats(): array
    {
        $moderators = User::where('role', UserRole::MODERATOR->value)
                         ->where('status', 'active')
                         ->get();

        $stats = [
            'total_moderators' => $moderators->count(),
            'moderators' => []
        ];

        foreach ($moderators as $moderator) {
            $stats['moderators'][] = [
                'id' => $moderator->id,
                'username' => $moderator->username,
                'pending_disputes' => $moderator->getDisputeLoad(),
                'total_disputes' => $moderator->moderatedDisputes()->count(),
            ];
        }

        // Sort by pending disputes (highest load first)
        usort($stats['moderators'], function ($a, $b) {
            return $b['pending_disputes'] <=> $a['pending_disputes'];
        });

        return $stats;
    }

    /**
     * Notify admins about unassigned disputes
     */
    private function notifyAdminsOfUnassignedDispute(Dispute $dispute): void
    {
        try {
            // Create a popup notification for all admins
            PopupNotification::create([
                'title' => 'Urgent: Unassigned Dispute',
                'message' => "Dispute #{$dispute->id} could not be assigned to a moderator. No active moderators available. Please review and assign manually.",
                'type' => PopupNotification::TYPE_WARNING,
                'is_active' => true,
                'target_users' => [
                    'roles' => [UserRole::ADMIN->value]
                ],
                'created_by' => null, // System generated
            ]);

            logger()->info("Admin notification created for unassigned dispute", [
                'dispute_id' => $dispute->id
            ]);
        } catch (\Exception $e) {
            logger()->error("Failed to create admin notification for unassigned dispute", [
                'dispute_id' => $dispute->id,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Fallback: Assign dispute to admin when no moderators available
     */
    private function assignDisputeToAdminFallback(Dispute $dispute): bool
    {
        try {
            // Get available admins
            $admins = User::where('role', UserRole::ADMIN->value)
                         ->where('status', 'active')
                         ->get();

            if ($admins->isEmpty()) {
                logger()->critical("No active admins available for dispute fallback assignment", [
                    'dispute_id' => $dispute->id
                ]);
                return false;
            }

            // Assign to admin with lowest dispute load
            $selectedAdmin = $this->selectBestModerator($admins);
            $dispute->assignToModerator($selectedAdmin);

            logger()->warning("Dispute assigned to admin as fallback", [
                'dispute_id' => $dispute->id,
                'admin_id' => $selectedAdmin->id,
                'reason' => 'No moderators available'
            ]);

            return true;
        } catch (\Exception $e) {
            logger()->error("Failed to assign dispute to admin fallback", [
                'dispute_id' => $dispute->id,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }
}