<?php

namespace App\Services;

use App\Models\Challenge;
use App\Models\User;
use App\Models\WalletTransaction;
use Illuminate\Support\Facades\DB;

class WalletService
{
    public const COMMISSION_RATE = 0.05; // 5% commission

    /**
     * Lock funds for a challenge or tournament.
     */
    public function lockFunds(User $user, float $amount, string $type, int $referenceId): WalletTransaction
    {
        if ($user->wallet_balance < $amount) {
            throw new \Exception('Insufficient wallet balance.');
        }

        return DB::transaction(function () use ($user, $amount, $type, $referenceId) {
            // Decrease wallet balance and increase locked balance
            $user->decrement('wallet_balance', $amount);
            $user->increment('locked_balance', $amount);

            // Create transaction record
            return WalletTransaction::create([
                'user_id' => $user->id,
                'type' => 'wager',
                'amount' => -$amount,
                'status' => 'completed',
                'description' => "Funds locked for {$type}",
                'metadata' => [
                    'type' => $type,
                    'reference_id' => $referenceId,
                ],
            ]);
        });
    }

    /**
     * Process challenge win payout.
     */
    public function processChallengeWin(Challenge $challenge): void
    {
        $winner = $challenge->getWinner();
        
        if (!$winner) {
            throw new \Exception('No winner determined for challenge.');
        }

        $totalAmount = $challenge->getTotalEscrowAmount();
        $commission = $totalAmount * self::COMMISSION_RATE;
        $winnerAmount = $totalAmount - $commission;

        DB::transaction(function () use ($challenge, $winner, $winnerAmount, $commission) {
            // Release locked funds from both players
            $challenge->creator->decrement('locked_balance', $challenge->wager_amount);
            $challenge->accepter->decrement('locked_balance', $challenge->wager_amount);

            // Pay winner
            $winner->increment('wallet_balance', $winnerAmount);

            // Create winner transaction
            WalletTransaction::create([
                'user_id' => $winner->id,
                'type' => 'prize',
                'amount' => $winnerAmount,
                'status' => 'completed',
                'description' => 'Challenge win payout',
                'metadata' => [
                    'challenge_id' => $challenge->id,
                    'commission_taken' => $commission,
                ],
            ]);

            // Create commission transaction (assigned to platform account)
            WalletTransaction::create([
                'user_id' => self::getPlatformAccountUserId(),
                'type' => 'fee',
                'amount' => $commission,
                'status' => 'completed',
                'description' => 'Platform commission',
                'metadata' => [
                    'challenge_id' => $challenge->id,
                ],
            ]);
        });
    }

    /**
     * Refund challenge funds.
     */
    public function refundChallenge(Challenge $challenge): void
    {
        DB::transaction(function () use ($challenge) {
            // Refund creator
            $challenge->creator->decrement('locked_balance', $challenge->wager_amount);
            $challenge->creator->increment('wallet_balance', $challenge->wager_amount);

            // Refund accepter
            $challenge->accepter->decrement('locked_balance', $challenge->wager_amount);
            $challenge->accepter->increment('wallet_balance', $challenge->wager_amount);

            // Create refund transactions
            foreach ([$challenge->creator, $challenge->accepter] as $user) {
                WalletTransaction::create([
                    'user_id' => $user->id,
                    'type' => 'refund',
                    'amount' => $challenge->wager_amount,
                    'status' => 'completed',
                    'description' => 'Challenge refund due to dispute resolution',
                    'metadata' => [
                        'challenge_id' => $challenge->id,
                    ],
                ]);
            }
        });
    }

    /**
     * Initiate a deposit.
     */
    public function initiateDeposit(User $user, float $amount, string $provider): WalletTransaction
    {
        return WalletTransaction::create([
            'user_id' => $user->id,
            'type' => 'deposit',
            'amount' => $amount,
            'status' => 'pending',
            'provider' => $provider,
            'reference' => uniqid('dep_'),
            'description' => 'Wallet deposit',
        ]);
    }

    /**
     * Complete a deposit.
     */
    public function completeDeposit(WalletTransaction $transaction): void
    {
        if ($transaction->type !== 'deposit' || $transaction->status !== 'pending') {
            throw new \Exception('Invalid transaction for deposit completion.');
        }

        DB::transaction(function () use ($transaction) {
            // Update user wallet balance
            $transaction->user->increment('wallet_balance', $transaction->amount);

            // Update transaction status
            $transaction->update(['status' => 'completed']);
        });
    }

    /**
     * Initiate a withdrawal.
     */
    public function initiateWithdrawal(User $user, float $amount, string $provider): WalletTransaction
    {
        if ($user->wallet_balance < $amount) {
            throw new \Exception('Insufficient wallet balance for withdrawal.');
        }

        return DB::transaction(function () use ($user, $amount, $provider) {
            // Deduct from wallet balance
            $user->decrement('wallet_balance', $amount);

            // Create withdrawal transaction
            return WalletTransaction::create([
                'user_id' => $user->id,
                'type' => 'withdraw',
                'amount' => -$amount,
                'status' => 'pending',
                'provider' => $provider,
                'reference' => uniqid('wit_'),
                'description' => 'Wallet withdrawal',
            ]);
        });
    }

    /**
     * Get platform account user ID for commission tracking
     */
    private static function getPlatformAccountUserId(): int
    {
        // For now, return the first admin user's ID
        // In production, this should be a dedicated platform account
        return \App\Models\User::where('is_organizer', true)->first()?->id ?? 1;
    }
}