<?php

namespace App\Services;

use App\Models\User;
use App\Models\Level;
use Illuminate\Support\Facades\Log; // Added: Import Laravel's Log facade

class XPService
{
    /**
     * XP values for different actions.
     */
    public const XP_VALUES = [
        'challenge_accepted' => 10,
        'challenge_won' => 20,
        'tournament_joined' => 15,
        'tournament_created' => 15,
        'tournament_match_won' => 20,
        'tournament_won' => 50,
        'no_dispute_streak' => 5, // Bonus XP for clean matches
    ];

    /**
     * Award XP to a user for an action and track games played.
     */
    public function awardXP(User $user, int $amount, string $action): void
    {
        $oldXP = $user->xp;
        $oldLevel = $user->getCurrentLevel();

        // Add XP
        $user->increment('xp', $amount);
        
        // Track games played for relevant actions
        if ($this->isGameAction($action)) {
            $user->addGamesPlayed(1);
        }
        
        $user->refresh();

        // Check for level up (considering both XP and games played now)
        $newLevel = $user->getCurrentLevel();
        
        if ($newLevel && (!$oldLevel || $newLevel->id > $oldLevel->id)) {
            $user->update(['level' => $newLevel->id]);
            
            // Could fire an event here for level up notification
            $this->handleLevelUp($user, $oldLevel, $newLevel);
        }

        // Check for badge awards
        $this->checkBadgeAwards($user, $action);
    }

    /**
     * Check if an action represents playing a game.
     */
    private function isGameAction(string $action): bool
    {
        return in_array($action, [
            'challenge_accepted',
            'challenge_won',
            'tournament_match_won',
            'tournament_won',
        ]);
    }

    /**
     * Handle level up logic.
     */
    private function handleLevelUp(User $user, ?Level $oldLevel, Level $newLevel): void
    {
        // Create notification or send event
        // Could also award bonus items, unlock features, etc.
        
        // Using Laravel's Log facade for proper logging
        Log::info("User {$user->username} leveled up from " . 
                  ($oldLevel ? $oldLevel->name : 'Level 0') . 
                  " to {$newLevel->name}");
    }

    /**
     * Check and award badges based on user actions and achievements.
     */
    private function checkBadgeAwards(User $user, string $action): void
    {
        $badgeService = app(BadgeService::class);
        
        // Check various badge conditions
        switch ($action) {
            case 'challenge_won':
                $badgeService->checkWinBadges($user);
                break;
            case 'tournament_joined':
                $badgeService->checkTournamentBadges($user);
                break;
            case 'challenge_accepted':
                $badgeService->checkParticipationBadges($user);
                break;
        }
    }

    /**
     * Get XP required for next level.
     */
    public function getXPToNextLevel(User $user): ?int
    {
        $nextLevel = $user->getNextLevel();
        
        if (!$nextLevel) {
            return null; // Max level reached
        }
        
        return $nextLevel->min_xp - $user->xp;
    }

    /**
     * Get progress percentage to next level.
     */
    public function getLevelProgress(User $user): float
    {
        $currentLevel = $user->getCurrentLevel();
        $nextLevel = $user->getNextLevel();
        
        if (!$currentLevel || !$nextLevel) {
            return 100.0; // Max level
        }
        
        $currentLevelXP = $currentLevel->min_xp;
        $nextLevelXP = $nextLevel->min_xp;
        $userXP = $user->xp;
        
        $progressXP = $userXP - $currentLevelXP;
        $requiredXP = $nextLevelXP - $currentLevelXP;
        
        return ($progressXP / $requiredXP) * 100;
    }
}