<?php

namespace Database\Seeders;

use App\Models\Level;
use Illuminate\Database\Seeder;

class ThirtyLevelSystemSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Clear existing levels first
        Level::query()->delete();
        
        $levels = [];
        
        // Create 30 regular levels
        for ($i = 1; $i <= 30; $i++) {
            $gamesRequired = $this->calculateGamesRequired($i);
            $wagerLimit = $this->calculateWagerLimit($i);
            
            $levels[] = [
                'name' => "Level {$i}",
                'min_xp' => ($i - 1) * 100, // Basic XP progression
                'min_matches' => $i * 2, // Minimum matches progression
                'min_games_played' => $gamesRequired,
                'min_agreement_rate' => min(0.95, 0.6 + ($i * 0.01)), // Max 0.95
                'requires_kyc' => $i >= 20, // KYC required from level 20+
                'can_create_challenges' => $i >= 2, // Can create challenges from level 2
                'can_host_free_tournaments' => $i >= 5, // Free tournaments from level 5
                'can_host_paid_tournaments' => $i >= 29, // Paid tournaments only for levels 29, 30
                'max_wager_amount' => $wagerLimit,
            ];
        }
        
        // Add special Creator level (essentially level 31)
        $levels[] = [
            'name' => 'Creator',
            'min_xp' => 3000, // High XP requirement
            'min_matches' => 100,
            'min_games_played' => 1000, // Very high games requirement
            'min_agreement_rate' => 0.95,
            'requires_kyc' => true,
            'can_create_challenges' => true,
            'can_host_free_tournaments' => true,
            'can_host_paid_tournaments' => true,
            'max_wager_amount' => null, // No limit for Creator
        ];

        foreach ($levels as $levelData) {
            Level::create($levelData);
        }
    }
    
    /**
     * Calculate games required for each level with cumulative progression
     */
    private function calculateGamesRequired(int $level): int
    {
        if ($level === 1) return 0;
        
        $games = 0;
        for ($i = 2; $i <= $level; $i++) {
            if ($i <= 5) {
                $games += 10; // Levels 2-5: +10 games each
            } elseif ($i <= 10) {
                $games += 15; // Levels 6-10: +15 games each
            } elseif ($i <= 15) {
                $games += 25; // Levels 11-15: +25 games each
            } elseif ($i <= 20) {
                $games += 35; // Levels 16-20: +35 games each
            } elseif ($i <= 25) {
                $games += 50; // Levels 21-25: +50 games each
            } else {
                $games += 75; // Levels 26-30: +75 games each
            }
        }
        
        return $games;
    }
    
    /**
     * Calculate wager limits for each level (in Naira)
     */
    private function calculateWagerLimit(int $level): float
    {
        // Progressive wager limits
        $baseLimits = [
            1 => 1000,      // ₦1,000
            2 => 5000,      // ₦5,000
            3 => 20000,     // ₦20,000
            4 => 50000,     // ₦50,000
            5 => 100000,    // ₦100,000
        ];
        
        if ($level <= 5) {
            return $baseLimits[$level];
        } elseif ($level <= 10) {
            return 100000 + (($level - 5) * 50000); // ₦150K - ₦350K
        } elseif ($level <= 15) {
            return 350000 + (($level - 10) * 100000); // ₦450K - ₦850K
        } elseif ($level <= 20) {
            return 850000 + (($level - 15) * 150000); // ₦1M - ₦1.6M
        } elseif ($level <= 25) {
            return 1600000 + (($level - 20) * 200000); // ₦1.8M - ₦2.6M
        } else {
            return 2600000 + (($level - 25) * 400000); // ₦3M - ₦4.6M
        }
    }
}
