@extends('layouts.admin')

@section('title', 'Challenge Management')
@section('page-title', 'Challenge Management')
@section('page-subtitle', 'Manage 1v1 challenges and hook players together')

@section('content')
<div class="p-6">
    <div class="max-w-7xl mx-auto">
        <div class="flex justify-between items-center mb-8">
            <div>
                <h1 class="text-3xl font-gaming font-bold text-cod-green gaming-text-glow">Challenge Management</h1>
                <p class="text-gray-400 mt-2">Manage 1v1 challenges and hook players together</p>
            </div>
            <div class="flex space-x-3">
                <a href="{{ route('admin.challenges.create') }}" class="gaming-button px-6 py-3 rounded-lg">
                    Create Challenge
                </a>
                <button onclick="showHookModal()" class="btn-secondary px-6 py-3 rounded-lg">
                    Hook 2 Players
                </button>
            </div>
        </div>

        <!-- Filters -->
        <div class="gaming-card rounded-lg p-4 mb-6">
            <form method="GET" class="flex items-center space-x-4">
                <div>
                    <select name="status" class="gaming-select">
                        <option value="">All Statuses</option>
                        <option value="open" {{ request('status') == 'open' ? 'selected' : '' }}>Open</option>
                        <option value="accepted" {{ request('status') == 'accepted' ? 'selected' : '' }}>Accepted</option>
                        <option value="completed" {{ request('status') == 'completed' ? 'selected' : '' }}>Completed</option>
                        <option value="disputed" {{ request('status') == 'disputed' ? 'selected' : '' }}>Disputed</option>
                        <option value="cancelled" {{ request('status') == 'cancelled' ? 'selected' : '' }}>Cancelled</option>
                    </select>
                </div>
                <div>
                    <select name="game_id" class="gaming-select">
                        <option value="">All Games</option>
                        @foreach($games as $game)
                            <option value="{{ $game->id }}" {{ request('game_id') == $game->id ? 'selected' : '' }}>
                                {{ $game->name }}
                            </option>
                        @endforeach
                    </select>
                </div>
                <button type="submit" class="btn-secondary px-4 py-2">Filter</button>
                <a href="{{ route('admin.challenges.index') }}" class="text-gray-400 hover:text-cod-green">Clear</a>
            </form>
        </div>

        <div class="gaming-card rounded-lg p-6">
            <div class="table-container">
                <table class="gaming-table w-full">
                    <thead class="sticky top-0 z-10 bg-cod-dark">
                        <tr>
                            <th class="text-left">Creator</th>
                            <th class="text-left">Accepter</th>
                            <th class="text-left">Game</th>
                            <th class="text-center">Wager</th>
                            <th class="text-center">Status</th>
                            <th class="text-center">Created</th>
                            <th class="text-center">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        @forelse($challenges as $challenge)
                            <tr>
                                <td>
                                    <div class="flex items-center space-x-2">
                                        <div class="w-8 h-8 bg-cod-green rounded-full flex items-center justify-center">
                                            <span class="text-xs font-bold text-black">{{ substr($challenge->creator->username, 0, 1) }}</span>
                                        </div>
                                        <span class="text-white">{{ $challenge->creator->username }}</span>
                                    </div>
                                </td>
                                <td>
                                    @if($challenge->accepter)
                                        <div class="flex items-center space-x-2">
                                            <div class="w-8 h-8 bg-cod-green rounded-full flex items-center justify-center">
                                                <span class="text-xs font-bold text-black">{{ substr($challenge->accepter->username, 0, 1) }}</span>
                                            </div>
                                            <span class="text-white">{{ $challenge->accepter->username }}</span>
                                        </div>
                                    @else
                                        <span class="text-gray-400 italic">Open</span>
                                    @endif
                                </td>
                                <td>
                                    <div class="flex items-center space-x-2">
                                        <div class="w-6 h-6 bg-cod-green rounded flex items-center justify-center">
                                            <span class="text-xs font-bold text-black">{{ substr($challenge->game->name, 0, 1) }}</span>
                                        </div>
                                        <span class="text-white text-sm">{{ $challenge->game->name }}</span>
                                    </div>
                                </td>
                                <td class="text-center text-cod-green font-medium">₦{{ number_format($challenge->wager_amount, 2) }}</td>
                                <td class="text-center">
                                    <span class="status-badge status-{{ $challenge->status }}">{{ ucfirst($challenge->status) }}</span>
                                </td>
                                <td class="text-center text-gray-400 text-sm">{{ $challenge->created_at->diffForHumans() }}</td>
                                <td class="text-center">
                                    <div class="flex justify-center space-x-2">
                                        <a href="{{ route('admin.challenges.show', $challenge) }}" class="btn-secondary px-3 py-1 text-xs">
                                            View
                                        </a>
                                        @if($challenge->status == 'accepted' || $challenge->status == 'disputed')
                                            <button onclick="showResolveModal({{ $challenge->id }}, '{{ $challenge->creator->username }}', '{{ $challenge->accepter ? $challenge->accepter->username : '' }}')" 
                                                    class="gaming-button px-3 py-1 text-xs">
                                                Resolve
                                            </button>
                                        @endif
                                        @if($challenge->status != 'completed' && $challenge->status != 'cancelled')
                                            <form action="{{ route('admin.challenges.cancel', $challenge) }}" method="POST" class="inline">
                                                @csrf
                                                <button type="submit" class="btn-danger px-3 py-1 text-xs" 
                                                        onclick="return confirm('Are you sure you want to cancel this challenge?')">
                                                    Cancel
                                                </button>
                                            </form>
                                        @endif
                                    </div>
                                </td>
                            </tr>
                        @empty
                            <tr>
                                <td colspan="7" class="text-center py-8">
                                    <div class="text-gray-400">
                                        <svg class="w-12 h-12 mx-auto mb-4 opacity-50" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4M7.835 4.697a3.42 3.42 0 001.946-.806 3.42 3.42 0 014.438 0 3.42 3.42 0 001.946.806 3.42 3.42 0 013.138 3.138 3.42 3.42 0 00.806 1.946 3.42 3.42 0 010 4.438 3.42 3.42 0 00-.806 1.946 3.42 3.42 0 01-3.138 3.138 3.42 3.42 0 00-1.946.806 3.42 3.42 0 01-4.438 0 3.42 3.42 0 00-1.946-.806 3.42 3.42 0 01-3.138-3.138 3.42 3.42 0 00-.806-1.946 3.42 3.42 0 010-4.438 3.42 3.42 0 00.806-1.946 3.42 3.42 0 013.138-3.138z"></path>
                                        </svg>
                                        <p>No challenges found.</p>
                                    </div>
                                </td>
                            </tr>
                        @endforelse
                    </tbody>
                </table>
            </div>

            @if($challenges->hasPages())
                <div class="mt-6">
                    {{ $challenges->links() }}
                </div>
            @endif
        </div>
    </div>
</div>

<!-- Hook Players Modal -->
<div id="hookModal" class="fixed inset-0 bg-black bg-opacity-50 hidden z-50">
    <div class="flex items-center justify-center min-h-screen p-4">
        <div class="gaming-card rounded-lg p-6 w-full max-w-md">
            <h3 class="text-lg font-semibold text-cod-green mb-4">Hook 2 Players Together</h3>
            <form action="{{ route('admin.challenges.hook-users') }}" method="POST" class="space-y-4">
                @csrf
                <div>
                    <label class="block text-sm font-medium text-cod-green mb-2">Player 1</label>
                    <select name="player1_id" class="gaming-select w-full" required>
                        <option value="">Select Player 1</option>
                        @foreach(\App\Models\User::where('is_banned', false)->orderBy('username')->get() as $user)
                            <option value="{{ $user->id }}">{{ $user->username }} (₦{{ number_format($user->wallet_balance, 2) }})</option>
                        @endforeach
                    </select>
                </div>
                <div>
                    <label class="block text-sm font-medium text-cod-green mb-2">Player 2</label>
                    <select name="player2_id" class="gaming-select w-full" required>
                        <option value="">Select Player 2</option>
                        @foreach(\App\Models\User::where('is_banned', false)->orderBy('username')->get() as $user)
                            <option value="{{ $user->id }}">{{ $user->username }} (₦{{ number_format($user->wallet_balance, 2) }})</option>
                        @endforeach
                    </select>
                </div>
                <div>
                    <label class="block text-sm font-medium text-cod-green mb-2">Game</label>
                    <select name="game_id" class="gaming-select w-full" required>
                        <option value="">Select Game</option>
                        @foreach($games as $game)
                            <option value="{{ $game->id }}">{{ $game->name }}</option>
                        @endforeach
                    </select>
                </div>
                <div>
                    <label class="block text-sm font-medium text-cod-green mb-2">Wager Amount (₦)</label>
                    <input type="number" name="wager_amount" class="gaming-input w-full" min="0.01" step="0.01" required>
                </div>
                <div>
                    <label class="block text-sm font-medium text-cod-green mb-2">Best of</label>
                    <select name="best_of" class="gaming-select w-full" required>
                        <option value="1">Best of 1</option>
                        <option value="3">Best of 3</option>
                        <option value="5">Best of 5</option>
                        <option value="7">Best of 7</option>
                    </select>
                </div>
                <div>
                    <label class="block text-sm font-medium text-cod-green mb-2">Rules</label>
                    <textarea name="rules_text" class="gaming-textarea w-full" rows="3" required></textarea>
                </div>
                <div class="flex justify-end space-x-3 pt-4">
                    <button type="button" onclick="hideHookModal()" class="btn-secondary px-4 py-2">Cancel</button>
                    <button type="submit" class="gaming-button px-4 py-2 rounded-lg">Hook Players</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Resolve Challenge Modal -->
<div id="resolveModal" class="fixed inset-0 bg-black bg-opacity-50 hidden z-50">
    <div class="flex items-center justify-center min-h-screen p-4">
        <div class="gaming-card rounded-lg p-6 w-full max-w-md">
            <h3 class="text-lg font-semibold text-cod-green mb-4">Resolve Challenge</h3>
            <form id="resolveForm" method="POST" class="space-y-4">
                @csrf
                <div>
                    <label class="block text-sm font-medium text-cod-green mb-2">Select Winner</label>
                    <div class="space-y-2">
                        <label class="flex items-center">
                            <input type="radio" name="winner" value="creator" class="mr-2">
                            <span id="creatorName" class="text-white"></span>
                        </label>
                        <label class="flex items-center">
                            <input type="radio" name="winner" value="accepter" class="mr-2">
                            <span id="accepterName" class="text-white"></span>
                        </label>
                        <label class="flex items-center">
                            <input type="radio" name="winner" value="draw" class="mr-2">
                            <span class="text-white">Draw (Refund Both)</span>
                        </label>
                    </div>
                </div>
                <div class="flex justify-end space-x-3 pt-4">
                    <button type="button" onclick="hideResolveModal()" class="btn-secondary px-4 py-2">Cancel</button>
                    <button type="submit" class="gaming-button px-4 py-2 rounded-lg">Resolve Challenge</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
function showHookModal() {
    document.getElementById('hookModal').classList.remove('hidden');
}

function hideHookModal() {
    document.getElementById('hookModal').classList.add('hidden');
}

function showResolveModal(challengeId, creatorName, accepterName) {
    document.getElementById('resolveForm').action = `/admin/challenges/${challengeId}/resolve`;
    document.getElementById('creatorName').textContent = creatorName;
    document.getElementById('accepterName').textContent = accepterName;
    document.getElementById('resolveModal').classList.remove('hidden');
}

function hideResolveModal() {
    document.getElementById('resolveModal').classList.add('hidden');
}
</script>
@endsection