@extends('layouts.admin')

@section('title', 'Dispute Management')
@section('page-title', 'Dispute Management')
@section('page-subtitle', 'Manage and resolve user disputes')

@section('content')
<div class="p-6">
    <div class="max-w-7xl mx-auto">
        <div class="page-header">
            <h1>⚖️ Dispute Management</h1>
            <p>Review and resolve challenge disputes</p>
        </div>

        <!-- Disputes List -->
        <div class="disputes-container">
            @if($disputes->count() > 0)
                <div class="disputes-list">
                    @foreach($disputes as $dispute)
                        <div class="dispute-card">
                            <div class="dispute-header">
                                <div class="dispute-title">
                                    <h3>Challenge #{{ $dispute->challenge->id }}</h3>
                                    <span class="dispute-status {{ $dispute->status }}">{{ ucfirst($dispute->status) }}</span>
                                </div>
                                <div class="dispute-meta">
                                    <small>Opened {{ $dispute->created_at->diffForHumans() }}</small>
                                </div>
                            </div>

                            <div class="dispute-content">
                                <div class="challenge-info">
                                    <h4>⚔️ Challenge Details</h4>
                                    <div class="challenge-summary">
                                        <div class="players">
                                            <div class="player">
                                                <strong>{{ $dispute->challenge->creator->username }}</strong>
                                                <span class="role">(Creator)</span>
                                            </div>
                                            <div class="vs">VS</div>
                                            <div class="player">
                                                <strong>{{ $dispute->challenge->accepter->username }}</strong>
                                                <span class="role">(Accepter)</span>
                                            </div>
                                        </div>
                                        <div class="wager-info">
                                            <span class="wager-amount">₦{{ number_format($dispute->challenge->wager_amount, 2) }}</span>
                                            <span class="game">{{ $dispute->challenge->game->name ?? 'Unknown Game' }}</span>
                                        </div>
                                    </div>
                                </div>

                                <div class="dispute-details">
                                    <h4>🚨 Dispute Information</h4>
                                    <div class="dispute-info">
                                        <p><strong>Opened by:</strong> {{ $dispute->openedBy->username }}</p>
                                        <p><strong>Reason:</strong> {{ $dispute->reason }}</p>
                                        @if($dispute->description)
                                            <p><strong>Description:</strong> {{ $dispute->description }}</p>
                                        @endif
                                    </div>
                                </div>
                            </div>

                            <div class="dispute-actions">
                                <a href="{{ route('admin.disputes.show', $dispute) }}" class="btn-primary">
                                    🔍 Review Dispute
                                </a>
                            </div>
                        </div>
                    @endforeach
                </div>

                <!-- Pagination -->
                @if($disputes->hasPages())
                    <div class="pagination-wrapper">
                        {{ $disputes->links() }}
                    </div>
                @endif
            @else
                <div class="empty-state">
                    <div class="empty-content">
                        <h2>🎉 No Disputes!</h2>
                        <p>Great job! There are no open disputes that need your attention.</p>
                        <div class="empty-illustration">⚖️</div>
                    </div>
                </div>
            @endif
        </div>
    </div>
</div>

<style>
.admin-disputes {
    padding: 2rem 0;
    background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
    min-height: 100vh;
}

.page-header {
    text-align: center;
    margin-bottom: 3rem;
}

.page-header h1 {
    font-size: 2.5rem;
    color: #ff6600;
    margin-bottom: 0.5rem;
}

.page-header p {
    color: #94a3b8;
    font-size: 1.1rem;
}

.disputes-container {
    max-width: 1000px;
    margin: 0 auto;
}

.disputes-list {
    display: flex;
    flex-direction: column;
    gap: 1.5rem;
}

.dispute-card {
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(239, 68, 68, 0.3);
    border-radius: 12px;
    overflow: hidden;
    transition: all 0.3s ease;
}

.dispute-card:hover {
    transform: translateY(-3px);
    border-color: rgba(239, 68, 68, 0.5);
    box-shadow: 0 10px 25px rgba(239, 68, 68, 0.1);
}

.dispute-header {
    background: rgba(239, 68, 68, 0.1);
    border-bottom: 1px solid rgba(239, 68, 68, 0.2);
    padding: 1.5rem;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.dispute-title {
    display: flex;
    align-items: center;
    gap: 1rem;
}

.dispute-title h3 {
    color: #e2e8f0;
    margin: 0;
    font-size: 1.3rem;
}

.dispute-status {
    padding: 0.25rem 0.75rem;
    border-radius: 20px;
    font-size: 0.8rem;
    font-weight: 600;
    text-transform: uppercase;
}

.dispute-status.open {
    background: rgba(239, 68, 68, 0.2);
    color: #ef4444;
    border: 1px solid #ef4444;
}

.dispute-status.resolved {
    background: rgba(34, 197, 94, 0.2);
    color: #22c55e;
    border: 1px solid #22c55e;
}

.dispute-meta small {
    color: #94a3b8;
    font-size: 0.9rem;
}

.dispute-content {
    padding: 1.5rem;
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 2rem;
}

.challenge-info h4, .dispute-details h4 {
    color: #ff6600;
    margin: 0 0 1rem 0;
    font-size: 1.1rem;
}

.challenge-summary {
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 255, 255, 0.1);
    border-radius: 8px;
    padding: 1rem;
}

.players {
    display: flex;
    align-items: center;
    justify-content: space-between;
    margin-bottom: 1rem;
}

.player {
    text-align: center;
    flex: 1;
}

.player strong {
    color: #e2e8f0;
    display: block;
    font-size: 1.1rem;
    margin-bottom: 0.25rem;
}

.player .role {
    color: #94a3b8;
    font-size: 0.8rem;
}

.vs {
    color: #ff6600;
    font-weight: bold;
    font-size: 1.2rem;
    margin: 0 1rem;
}

.wager-info {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding-top: 1rem;
    border-top: 1px solid rgba(255, 255, 255, 0.1);
}

.wager-amount {
    color: #22c55e;
    font-weight: bold;
    font-size: 1.2rem;
}

.game {
    color: #94a3b8;
    font-size: 0.9rem;
}

.dispute-info {
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 255, 255, 0.1);
    border-radius: 8px;
    padding: 1rem;
}

.dispute-info p {
    color: #e2e8f0;
    margin: 0 0 0.75rem 0;
    line-height: 1.5;
}

.dispute-info p:last-child {
    margin-bottom: 0;
}

.dispute-info strong {
    color: #ff6600;
}

.dispute-actions {
    padding: 1.5rem;
    border-top: 1px solid rgba(255, 255, 255, 0.1);
    background: rgba(255, 255, 255, 0.02);
    text-align: center;
}

.btn-primary {
    background: #ff6600;
    color: #1a1a2e;
    padding: 0.75rem 2rem;
    border-radius: 8px;
    text-decoration: none;
    font-weight: 600;
    transition: all 0.3s ease;
    display: inline-block;
}

.btn-primary:hover {
    background: #e55a00;
    transform: translateY(-2px);
    box-shadow: 0 5px 15px rgba(255, 102, 0, 0.3);
}

.empty-state {
    text-align: center;
    padding: 4rem 2rem;
}

.empty-content {
    max-width: 500px;
    margin: 0 auto;
}

.empty-content h2 {
    color: #e2e8f0;
    font-size: 2rem;
    margin-bottom: 1rem;
}

.empty-content p {
    color: #94a3b8;
    font-size: 1.1rem;
    margin-bottom: 2rem;
    line-height: 1.6;
}

.empty-illustration {
    font-size: 4rem;
    margin-top: 2rem;
    opacity: 0.5;
}

.pagination-wrapper {
    display: flex;
    justify-content: center;
    margin-top: 2rem;
}

@media (max-width: 768px) {
    .dispute-header {
        flex-direction: column;
        gap: 1rem;
        align-items: flex-start;
    }
    
    .dispute-content {
        grid-template-columns: 1fr;
        gap: 1.5rem;
    }
    
    .players {
        flex-direction: column;
        gap: 1rem;
    }
    
    .vs {
        margin: 0;
    }
    
    .wager-info {
        flex-direction: column;
        gap: 0.5rem;
        align-items: flex-start;
    }
}
</style>
@endsection