@extends('layouts.app')
@php
    use Illuminate\Support\Facades\Storage;
@endphp

@section('title', 'Dispute Details')

@section('content')
<div class="admin-dispute-detail">
    <div class="container">
        <div class="page-header">
            <a href="{{ route('admin.disputes.index') }}" class="back-link">← Back to Disputes</a>
            <h1>⚖️ Dispute Resolution</h1>
        </div>

        <div class="dispute-resolution-container">
            <!-- Dispute Information -->
            <div class="dispute-info-card">
                <div class="card-header">
                    <h2>🚨 Dispute #{{ $dispute->id }}</h2>
                    <span class="dispute-status {{ $dispute->status }}">{{ ucfirst($dispute->status) }}</span>
                </div>

                <div class="dispute-details">
                    <div class="detail-row">
                        <strong>Opened by:</strong>
                        <span>{{ $dispute->openedBy->username }}</span>
                    </div>
                    <div class="detail-row">
                        <strong>Date:</strong>
                        <span>{{ $dispute->created_at->format('M d, Y H:i') }} ({{ $dispute->created_at->diffForHumans() }})</span>
                    </div>
                    <div class="detail-row">
                        <strong>Reason:</strong>
                        <span>{{ $dispute->reason }}</span>
                    </div>
                    @if($dispute->description)
                        <div class="detail-row full-width">
                            <strong>Description:</strong>
                            <p>{{ $dispute->description }}</p>
                        </div>
                    @endif
                </div>
            </div>

            <!-- Challenge Information -->
            <div class="challenge-info-card">
                <div class="card-header">
                    <h2>⚔️ Challenge #{{ $dispute->challenge->id }}</h2>
                    <span class="challenge-status {{ $dispute->challenge->status }}">{{ ucfirst($dispute->challenge->status) }}</span>
                </div>

                <div class="challenge-summary">
                    <!-- Players -->
                    <div class="players-section">
                        <div class="player-card creator">
                            <div class="player-avatar">{{ strtoupper(substr($dispute->challenge->creator->username, 0, 1)) }}</div>
                            <div class="player-info">
                                <h3>{{ $dispute->challenge->creator->username }}</h3>
                                <p>Challenge Creator</p>
                                <div class="player-stats">
                                    <span>Level {{ $dispute->challenge->creator->level }}</span>
                                    <span>{{ number_format($dispute->challenge->creator->xp) }} XP</span>
                                </div>
                            </div>
                        </div>

                        <div class="vs-divider">
                            <span>VS</span>
                            <div class="wager-amount">₦{{ number_format($dispute->challenge->wager_amount, 2) }}</div>
                        </div>

                        @if($dispute->challenge->accepter)
                            <div class="player-card accepter">
                                <div class="player-avatar">{{ strtoupper(substr($dispute->challenge->accepter->username, 0, 1)) }}</div>
                                <div class="player-info">
                                    <h3>{{ $dispute->challenge->accepter->username }}</h3>
                                    <p>Challenge Accepter</p>
                                    <div class="player-stats">
                                        <span>Level {{ $dispute->challenge->accepter->level }}</span>
                                        <span>{{ number_format($dispute->challenge->accepter->xp) }} XP</span>
                                    </div>
                                </div>
                            </div>
                        @else
                            <div class="player-card empty">
                                <div class="player-avatar">?</div>
                                <div class="player-info">
                                    <h3>No Accepter</h3>
                                    <p>Challenge not accepted</p>
                                </div>
                            </div>
                        @endif
                    </div>

                    <!-- Game and Platform -->
                    <div class="game-info">
                        <div class="info-item">
                            <strong>Game:</strong>
                            <span>{{ $dispute->challenge->game->name ?? 'Unknown Game' }}</span>
                        </div>
                        @if($dispute->challenge->game)
                            <div class="info-item">
                                <strong>Platform:</strong>
                                <span>{{ ucfirst($dispute->challenge->game->platform) }}</span>
                            </div>
                        @endif
                        <div class="info-item">
                            <strong>Created:</strong>
                            <span>{{ $dispute->challenge->created_at->format('M d, Y H:i') }}</span>
                        </div>
                        @if($dispute->challenge->expires_at)
                            <div class="info-item">
                                <strong>Expires:</strong>
                                <span>{{ $dispute->challenge->expires_at->format('M d, Y H:i') }}</span>
                            </div>
                        @endif
                    </div>
                </div>
            </div>

            <!-- Evidence Section -->
            @if($dispute->challenge->matchProofs->count() > 0 || $dispute->hasCreatorEvidence() || $dispute->hasAccepterEvidence())
                <div class="evidence-card">
                    <div class="card-header">
                        <h2>📷 Evidence</h2>
                    </div>

                    <!-- Match Proofs -->
                    @if($dispute->challenge->matchProofs->count() > 0)
                        <div class="evidence-section">
                            <h3>Match Results Evidence</h3>
                            <div class="evidence-list">
                                @foreach($dispute->challenge->matchProofs as $proof)
                                    <div class="evidence-item">
                                        <div class="evidence-info">
                                            <h4>{{ $proof->type === 'screenshot' ? '📷' : '🎥' }} {{ ucfirst($proof->type) }}</h4>
                                            <p>Submitted by {{ $proof->user->username }}</p>
                                            <small>{{ $proof->created_at->format('M d, Y H:i') }}</small>
                                        </div>
                                        <div class="evidence-actions">
                                            @if($proof->file_path)
                                                <a href="{{ Storage::url($proof->file_path) }}" target="_blank" class="btn-view">View Evidence</a>
                                            @endif
                                        </div>
                                    </div>
                                @endforeach
                            </div>
                        </div>
                    @endif

                    <!-- Dispute Evidence (for victory claims conflicts) -->
                    @if($dispute->hasCreatorEvidence() || $dispute->hasAccepterEvidence())
                        <div class="evidence-section">
                            <h3>Victory Claim Evidence</h3>
                            <div class="evidence-grid">
                                <!-- Creator Evidence -->
                                <div class="evidence-column">
                                    <h4>{{ $dispute->challenge->creator->username }} (Creator)</h4>
                                    @if($dispute->hasCreatorEvidence())
                                        <div class="evidence-item">
                                            <p>Victory claim evidence uploaded</p>
                                            <a href="{{ Storage::url($dispute->creator_evidence_path) }}" target="_blank" class="btn-view">
                                                View Creator's Evidence
                                            </a>
                                        </div>
                                    @else
                                        <div class="evidence-item pending">
                                            <p>No evidence uploaded</p>
                                        </div>
                                    @endif
                                </div>

                                <!-- Accepter Evidence -->
                                <div class="evidence-column">
                                    <h4>{{ $dispute->challenge->accepter->username ?? 'No Accepter' }} (Accepter)</h4>
                                    @if($dispute->hasAccepterEvidence())
                                        <div class="evidence-item">
                                            <p>Victory claim evidence uploaded</p>
                                            <a href="{{ Storage::url($dispute->accepter_evidence_path) }}" target="_blank" class="btn-view">
                                                View Accepter's Evidence
                                            </a>
                                        </div>
                                    @else
                                        <div class="evidence-item pending">
                                            <p>No evidence uploaded</p>
                                        </div>
                                    @endif
                                </div>
                            </div>

                            @if($dispute->hasBothEvidence())
                                <div class="evidence-summary">
                                    <div class="alert success">
                                        <h4>✅ Both parties have submitted evidence</h4>
                                        <p>Review both evidences carefully before making a decision.</p>
                                    </div>
                                </div>
                            @elseif($dispute->hasCreatorEvidence() || $dispute->hasAccepterEvidence())
                                <div class="evidence-summary">
                                    <div class="alert warning">
                                        <h4>⏳ Waiting for evidence</h4>
                                        <p>Only one party has submitted evidence. Consider waiting for both sides or making a decision based on available evidence.</p>
                                    </div>
                                </div>
                            @endif
                        </div>
                    @endif
                </div>
            @endif

            <!-- Resolution Actions -->
            @if($dispute->status === 'open')
                <div class="resolution-card">
                    <div class="card-header">
                        <h2>⚖️ Resolve Dispute</h2>
                    </div>

                    <form method="POST" action="{{ route('admin.disputes.decide', $dispute) }}" class="resolution-form">
                        @csrf
                        
                        <div class="resolution-options">
                            <div class="option-group">
                                <h3>Select Winner</h3>
                                <div class="winner-options">
                                    <label class="winner-option">
                                        <input type="radio" name="winner_id" value="{{ $dispute->challenge->creator->id }}" required>
                                        <div class="option-content">
                                            <div class="option-avatar">{{ strtoupper(substr($dispute->challenge->creator->username, 0, 1)) }}</div>
                                            <div class="option-info">
                                                <strong>{{ $dispute->challenge->creator->username }}</strong>
                                                <span>Challenge Creator</span>
                                            </div>
                                        </div>
                                    </label>

                                    @if($dispute->challenge->accepter)
                                        <label class="winner-option">
                                            <input type="radio" name="winner_id" value="{{ $dispute->challenge->accepter->id }}" required>
                                            <div class="option-content">
                                                <div class="option-avatar">{{ strtoupper(substr($dispute->challenge->accepter->username, 0, 1)) }}</div>
                                                <div class="option-info">
                                                    <strong>{{ $dispute->challenge->accepter->username }}</strong>
                                                    <span>Challenge Accepter</span>
                                                </div>
                                            </div>
                                        </label>
                                    @endif

                                    <label class="winner-option draw">
                                        <input type="radio" name="result" value="draw" required>
                                        <div class="option-content">
                                            <div class="option-avatar">🤝</div>
                                            <div class="option-info">
                                                <strong>Draw/Refund</strong>
                                                <span>Return wager to both players</span>
                                            </div>
                                        </div>
                                    </label>
                                </div>
                            </div>

                            <div class="admin-notes">
                                <label for="admin_notes">
                                    <h3>Admin Notes (Optional)</h3>
                                    <textarea name="admin_notes" id="admin_notes" rows="4" placeholder="Add any notes about your decision..."></textarea>
                                </label>
                            </div>
                        </div>

                        <div class="resolution-actions">
                            <button type="submit" class="btn-resolve">
                                ⚖️ Resolve Dispute
                            </button>
                        </div>
                    </form>
                </div>
            @else
                <!-- Resolution History -->
                <div class="resolution-history">
                    <div class="card-header">
                        <h2>✅ Dispute Resolved</h2>
                    </div>

                    <div class="resolution-info">
                        <div class="info-row">
                            <strong>Resolved by:</strong>
                            <span>{{ $dispute->admin ? $dispute->admin->username : 'System' }}</span>
                        </div>
                        <div class="info-row">
                            <strong>Resolution Date:</strong>
                            <span>{{ $dispute->resolved_at ? $dispute->resolved_at->format('M d, Y H:i') : 'N/A' }}</span>
                        </div>
                        @if($dispute->admin_notes)
                            <div class="info-row full-width">
                                <strong>Admin Notes:</strong>
                                <p>{{ $dispute->admin_notes }}</p>
                            </div>
                        @endif
                    </div>
                </div>
            @endif
        </div>
    </div>
</div>

<style>
.admin-dispute-detail {
    padding: 2rem 0;
    background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
    min-height: 100vh;
}

.page-header {
    margin-bottom: 2rem;
}

.back-link {
    color: #ff6600;
    text-decoration: none;
    font-weight: 600;
    margin-bottom: 1rem;
    display: inline-block;
    transition: color 0.3s ease;
}

.back-link:hover {
    color: #e55a00;
}

.page-header h1 {
    color: #e2e8f0;
    font-size: 2rem;
    margin: 0;
}

.dispute-resolution-container {
    max-width: 1000px;
    margin: 0 auto;
    display: flex;
    flex-direction: column;
    gap: 2rem;
}

.dispute-info-card, .challenge-info-card, .evidence-card, .resolution-card, .resolution-history {
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 102, 0, 0.2);
    border-radius: 12px;
    overflow: hidden;
}

.card-header {
    background: rgba(255, 102, 0, 0.1);
    border-bottom: 1px solid rgba(255, 102, 0, 0.2);
    padding: 1.5rem;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.card-header h2 {
    color: #ff6600;
    margin: 0;
    font-size: 1.3rem;
}

.dispute-status, .challenge-status {
    padding: 0.25rem 0.75rem;
    border-radius: 20px;
    font-size: 0.8rem;
    font-weight: 600;
    text-transform: uppercase;
}

.dispute-status.open, .challenge-status.disputed {
    background: rgba(239, 68, 68, 0.2);
    color: #ef4444;
    border: 1px solid #ef4444;
}

.dispute-status.resolved, .challenge-status.completed {
    background: rgba(34, 197, 94, 0.2);
    color: #22c55e;
    border: 1px solid #22c55e;
}

.challenge-status.open {
    background: rgba(59, 130, 246, 0.2);
    color: #3b82f6;
    border: 1px solid #3b82f6;
}

.dispute-details, .resolution-info {
    padding: 1.5rem;
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 1rem;
}

.detail-row, .info-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 0.5rem 0;
    border-bottom: 1px solid rgba(255, 255, 255, 0.1);
}

.detail-row.full-width, .info-row.full-width {
    grid-column: 1 / -1;
    flex-direction: column;
    align-items: flex-start;
}

.detail-row strong, .info-row strong {
    color: #ff6600;
    margin-right: 1rem;
}

.detail-row span, .info-row span {
    color: #e2e8f0;
}

.detail-row p, .info-row p {
    color: #94a3b8;
    margin: 0.5rem 0 0 0;
    line-height: 1.5;
}

.challenge-summary {
    padding: 1.5rem;
}

.players-section {
    display: flex;
    align-items: center;
    justify-content: space-between;
    margin-bottom: 2rem;
}

.player-card {
    display: flex;
    align-items: center;
    gap: 1rem;
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 255, 255, 0.1);
    border-radius: 12px;
    padding: 1rem;
    flex: 1;
    max-width: 250px;
}

.player-card.creator {
    border-color: rgba(59, 130, 246, 0.3);
}

.player-card.accepter {
    border-color: rgba(34, 197, 94, 0.3);
}

.player-card.empty {
    border-color: rgba(107, 114, 128, 0.3);
    opacity: 0.6;
}

.player-avatar {
    width: 50px;
    height: 50px;
    background: linear-gradient(135deg, #ff6600, #e55a00);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    color: #1a1a2e;
    font-weight: bold;
    font-size: 1.2rem;
}

.player-info h3 {
    color: #e2e8f0;
    margin: 0 0 0.25rem 0;
    font-size: 1rem;
}

.player-info p {
    color: #94a3b8;
    margin: 0 0 0.5rem 0;
    font-size: 0.9rem;
}

.player-stats {
    display: flex;
    gap: 1rem;
    font-size: 0.8rem;
}

.player-stats span {
    color: #64748b;
}

.vs-divider {
    text-align: center;
    margin: 0 1rem;
}

.vs-divider span {
    color: #ff6600;
    font-weight: bold;
    font-size: 1.5rem;
    display: block;
    margin-bottom: 0.5rem;
}

.wager-amount {
    color: #22c55e;
    font-weight: bold;
    font-size: 1.2rem;
}

.game-info {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 1rem;
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 255, 255, 0.1);
    border-radius: 8px;
    padding: 1rem;
}

.info-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.info-item strong {
    color: #ff6600;
}

.info-item span {
    color: #e2e8f0;
}

/* Evidence Section Styles */
.evidence-section {
    margin-bottom: 2rem;
}

.evidence-section h3 {
    color: #ff6600;
    margin-bottom: 1rem;
    font-size: 1.1rem;
}

.evidence-grid {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 1.5rem;
    margin-bottom: 1rem;
}

.evidence-column {
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 255, 255, 0.1);
    border-radius: 8px;
    padding: 1rem;
}

.evidence-column h4 {
    color: #e2e8f0;
    margin: 0 0 0.75rem 0;
    font-size: 1rem;
    border-bottom: 1px solid rgba(255, 255, 255, 0.1);
    padding-bottom: 0.5rem;
}

.evidence-item.pending {
    opacity: 0.6;
    font-style: italic;
}

.evidence-summary {
    margin-top: 1rem;
}

.alert {
    padding: 1rem;
    border-radius: 8px;
    border-left: 4px solid;
}

.alert.success {
    background: rgba(34, 197, 94, 0.1);
    border-color: #22c55e;
}

.alert.warning {
    background: rgba(245, 158, 11, 0.1);
    border-color: #f59e0b;
}

.alert h4 {
    color: inherit;
    margin: 0 0 0.5rem 0;
    font-size: 1rem;
}

.alert p {
    color: rgba(255, 255, 255, 0.8);
    margin: 0;
    font-size: 0.9rem;
}

/* Responsive adjustments */
@media (max-width: 768px) {
    .evidence-grid {
        grid-template-columns: 1fr;
    }
}

.evidence-list {
    padding: 1.5rem;
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.evidence-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 255, 255, 0.1);
    border-radius: 8px;
    padding: 1rem;
}

.evidence-info h4 {
    color: #e2e8f0;
    margin: 0 0 0.25rem 0;
    font-size: 1rem;
}

.evidence-info p {
    color: #94a3b8;
    margin: 0 0 0.25rem 0;
    font-size: 0.9rem;
}

.evidence-info small {
    color: #64748b;
    font-size: 0.8rem;
}

.btn-view {
    background: rgba(59, 130, 246, 0.2);
    color: #3b82f6;
    border: 1px solid #3b82f6;
    padding: 0.5rem 1rem;
    border-radius: 6px;
    text-decoration: none;
    font-size: 0.9rem;
    transition: all 0.3s ease;
}

.btn-view:hover {
    background: #3b82f6;
    color: #1a1a2e;
}

.resolution-form {
    padding: 1.5rem;
}

.resolution-options {
    display: flex;
    flex-direction: column;
    gap: 2rem;
}

.option-group h3 {
    color: #ff6600;
    margin: 0 0 1rem 0;
    font-size: 1.2rem;
}

.winner-options {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.winner-option {
    cursor: pointer;
    display: block;
}

.winner-option input {
    display: none;
}

.option-content {
    display: flex;
    align-items: center;
    gap: 1rem;
    background: rgba(255, 255, 255, 0.05);
    border: 2px solid rgba(255, 255, 255, 0.1);
    border-radius: 12px;
    padding: 1rem;
    transition: all 0.3s ease;
}

.winner-option:hover .option-content {
    border-color: rgba(255, 102, 0, 0.3);
}

.winner-option input:checked + .option-content {
    border-color: #ff6600;
    background: rgba(255, 102, 0, 0.1);
}

.option-avatar {
    width: 40px;
    height: 40px;
    background: linear-gradient(135deg, #ff6600, #e55a00);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    color: #1a1a2e;
    font-weight: bold;
    font-size: 1rem;
}

.winner-option.draw .option-avatar {
    background: linear-gradient(135deg, #6b7280, #4b5563);
    font-size: 1.2rem;
}

.option-info strong {
    color: #e2e8f0;
    display: block;
    margin-bottom: 0.25rem;
}

.option-info span {
    color: #94a3b8;
    font-size: 0.9rem;
}

.admin-notes label {
    display: block;
}

.admin-notes h3 {
    color: #ff6600;
    margin: 0 0 1rem 0;
    font-size: 1.2rem;
}

.admin-notes textarea {
    width: 100%;
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 102, 0, 0.3);
    border-radius: 8px;
    padding: 1rem;
    color: #e2e8f0;
    font-family: inherit;
    resize: vertical;
}

.admin-notes textarea:focus {
    outline: none;
    border-color: #ff6600;
    box-shadow: 0 0 0 2px rgba(255, 102, 0, 0.2);
}

.admin-notes textarea::placeholder {
    color: #94a3b8;
}

.resolution-actions {
    margin-top: 2rem;
    text-align: center;
}

.btn-resolve {
    background: #ff6600;
    color: #1a1a2e;
    padding: 1rem 3rem;
    border: none;
    border-radius: 8px;
    font-weight: 600;
    font-size: 1.1rem;
    cursor: pointer;
    transition: all 0.3s ease;
}

.btn-resolve:hover {
    background: #e55a00;
    transform: translateY(-2px);
    box-shadow: 0 5px 15px rgba(255, 102, 0, 0.3);
}

@media (max-width: 768px) {
    .players-section {
        flex-direction: column;
        gap: 1rem;
    }
    
    .vs-divider {
        margin: 0;
    }
    
    .player-card {
        max-width: 100%;
        width: 100%;
    }
    
    .game-info {
        grid-template-columns: 1fr;
    }
    
    .dispute-details, .resolution-info {
        grid-template-columns: 1fr;
    }
    
    .info-item {
        flex-direction: column;
        align-items: flex-start;
        gap: 0.5rem;
    }
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Form validation to ensure a winner is selected
    const form = document.querySelector('.resolution-form');
    if (form) {
        form.addEventListener('submit', function(e) {
            const winnerSelected = document.querySelector('input[name="winner_id"]:checked');
            const drawSelected = document.querySelector('input[name="result"]:checked');
            
            if (!winnerSelected && !drawSelected) {
                e.preventDefault();
                alert('Please select a winner or choose to resolve as a draw.');
                return false;
            }
        });
    }
});
</script>
@endsection