@extends('layouts.app')

@section('title', 'Create Popup Notification')

@section('content')
<div class="admin-create-notification">
    <div class="container">
        <div class="header-section">
            <div class="header-content">
                <h1>🔔 Create Popup Notification</h1>
                <p>Create announcements and important messages for users</p>
            </div>
            <div class="header-actions">
                <a href="{{ route('admin.popup-notifications.index') }}" class="btn-secondary">
                    ← Back to Notifications
                </a>
            </div>
        </div>

        <form method="POST" action="{{ route('admin.popup-notifications.store') }}" class="notification-form">
            @csrf
            
            <div class="form-grid">
                <!-- Main Content -->
                <div class="main-content">
                    <div class="form-section">
                        <h3>📝 Notification Content</h3>
                        
                        <div class="form-group">
                            <label for="title">Title *</label>
                            <input type="text" name="title" id="title" value="{{ old('title') }}" required maxlength="255" placeholder="Enter notification title">
                            @error('title')
                                <span class="error">{{ $message }}</span>
                            @enderror
                        </div>

                        <div class="form-group">
                            <label for="message">Message *</label>
                            <textarea name="message" id="message" rows="5" required placeholder="Enter notification message">{{ old('message') }}</textarea>
                            @error('message')
                                <span class="error">{{ $message }}</span>
                            @enderror
                        </div>

                        <div class="form-group">
                            <label for="type">Notification Type *</label>
                            <select name="type" id="type" required>
                                <option value="">Select Type</option>
                                <option value="info" {{ old('type') === 'info' ? 'selected' : '' }}>📢 Info</option>
                                <option value="warning" {{ old('type') === 'warning' ? 'selected' : '' }}>⚠️ Warning</option>
                                <option value="success" {{ old('type') === 'success' ? 'selected' : '' }}>✅ Success</option>
                                <option value="announcement" {{ old('type') === 'announcement' ? 'selected' : '' }}>🎉 Announcement</option>
                            </select>
                            @error('type')
                                <span class="error">{{ $message }}</span>
                            @enderror
                        </div>
                    </div>

                    <div class="form-section">
                        <h3>🎯 Targeting Options</h3>
                        
                        <div class="form-group">
                            <label>Target Users *</label>
                            <div class="radio-group">
                                <label class="radio-option">
                                    <input type="radio" name="target_type" value="all" {{ old('target_type', 'all') === 'all' ? 'checked' : '' }}>
                                    <span class="radio-label">👥 All Users</span>
                                    <small>Show to everyone on the platform</small>
                                </label>
                                
                                <label class="radio-option">
                                    <input type="radio" name="target_type" value="levels" {{ old('target_type') === 'levels' ? 'checked' : '' }}>
                                    <span class="radio-label">🎯 Specific Levels</span>
                                    <small>Target users by their level</small>
                                </label>
                                
                                <label class="radio-option">
                                    <input type="radio" name="target_type" value="users" {{ old('target_type') === 'users' ? 'checked' : '' }}>
                                    <span class="radio-label">👤 Specific Users</span>
                                    <small>Target specific usernames</small>
                                </label>
                            </div>
                            @error('target_type')
                                <span class="error">{{ $message }}</span>
                            @enderror
                        </div>

                        <div class="targeting-options">
                            <!-- Level targeting options -->
                            <div class="targeting-section" id="level-targeting" style="display: none;">
                                <label>Select Levels</label>
                                <div class="checkbox-group">
                                    @foreach($levels as $level)
                                        <label class="checkbox-option">
                                            <input type="checkbox" name="target_levels[]" value="{{ $level }}" {{ in_array($level, old('target_levels', [])) ? 'checked' : '' }}>
                                            <span>Level {{ $level }}</span>
                                        </label>
                                    @endforeach
                                </div>
                                
                                <div class="form-group" style="margin-top: 1rem;">
                                    <label for="min_level">Or Minimum Level</label>
                                    <select name="min_level" id="min_level">
                                        <option value="">No minimum level</option>
                                        @foreach($levels as $level)
                                            <option value="{{ $level }}" {{ old('min_level') == $level ? 'selected' : '' }}>Level {{ $level }}+</option>
                                        @endforeach
                                    </select>
                                    <small>Users at this level and above will see the notification</small>
                                </div>
                            </div>

                            <!-- User targeting options -->
                            <div class="targeting-section" id="user-targeting" style="display: none;">
                                <label for="target_users">Usernames (comma separated)</label>
                                <textarea name="target_users" id="target_users" rows="3" placeholder="Enter usernames separated by commas (e.g., gamer1, player2, admin)">{{ old('target_users') }}</textarea>
                                <small>Enter specific usernames who should see this notification</small>
                                @error('target_users')
                                    <span class="error">{{ $message }}</span>
                                @enderror
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Sidebar -->
                <div class="sidebar">
                    <div class="form-section">
                        <h3>⚙️ Settings</h3>
                        
                        <div class="form-group">
                            <label class="checkbox-label">
                                <input type="checkbox" name="is_active" value="1" {{ old('is_active', true) ? 'checked' : '' }}>
                                <span>📡 Active</span>
                            </label>
                            <small>Uncheck to create as inactive (can be activated later)</small>
                            @error('is_active')
                                <span class="error">{{ $message }}</span>
                            @enderror
                        </div>
                    </div>

                    <div class="form-section">
                        <h3>📅 Schedule</h3>
                        
                        <div class="form-group">
                            <label for="start_date">Start Date (optional)</label>
                            <input type="datetime-local" name="start_date" id="start_date" value="{{ old('start_date') }}" min="{{ now()->format('Y-m-d\TH:i') }}">
                            <small>When to start showing this notification</small>
                            @error('start_date')
                                <span class="error">{{ $message }}</span>
                            @enderror
                        </div>

                        <div class="form-group">
                            <label for="end_date">End Date (optional)</label>
                            <input type="datetime-local" name="end_date" id="end_date" value="{{ old('end_date') }}">
                            <small>When to stop showing this notification</small>
                            @error('end_date')
                                <span class="error">{{ $message }}</span>
                            @enderror
                        </div>
                    </div>

                    <div class="form-section">
                        <h3>👁️ Preview</h3>
                        <div class="notification-preview" id="preview">
                            <div class="preview-notification">
                                <div class="preview-header">
                                    <span class="preview-icon">📢</span>
                                    <span class="preview-title">Notification Title</span>
                                </div>
                                <div class="preview-message">
                                    Your notification message will appear here...
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="form-actions">
                        <button type="submit" class="btn-primary">
                            🔔 Create Notification
                        </button>
                    </div>
                </div>
            </div>
        </form>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Targeting type toggle
    const targetTypes = document.querySelectorAll('input[name="target_type"]');
    const levelTargeting = document.getElementById('level-targeting');
    const userTargeting = document.getElementById('user-targeting');

    function updateTargeting() {
        const selectedType = document.querySelector('input[name="target_type"]:checked').value;
        
        levelTargeting.style.display = selectedType === 'levels' ? 'block' : 'none';
        userTargeting.style.display = selectedType === 'users' ? 'block' : 'none';
    }

    targetTypes.forEach(radio => {
        radio.addEventListener('change', updateTargeting);
    });

    // Initialize targeting display
    updateTargeting();

    // Preview update
    const titleInput = document.getElementById('title');
    const messageInput = document.getElementById('message');
    const typeSelect = document.getElementById('type');
    const previewTitle = document.querySelector('.preview-title');
    const previewMessage = document.querySelector('.preview-message');
    const previewIcon = document.querySelector('.preview-icon');

    const typeIcons = {
        'info': '📢',
        'warning': '⚠️',
        'success': '✅',
        'announcement': '🎉'
    };

    function updatePreview() {
        previewTitle.textContent = titleInput.value || 'Notification Title';
        previewMessage.textContent = messageInput.value || 'Your notification message will appear here...';
        previewIcon.textContent = typeIcons[typeSelect.value] || '📢';
    }

    titleInput.addEventListener('input', updatePreview);
    messageInput.addEventListener('input', updatePreview);
    typeSelect.addEventListener('change', updatePreview);

    // Date validation
    const startDate = document.getElementById('start_date');
    const endDate = document.getElementById('end_date');

    startDate.addEventListener('change', function() {
        if (startDate.value) {
            endDate.min = startDate.value;
        }
    });
});
</script>

<style>
.admin-create-notification {
    padding: 2rem 0;
    background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
    min-height: 100vh;
    color: #e2e8f0;
}

.header-section {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 2rem;
    padding: 1.5rem;
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 102, 0, 0.2);
    border-radius: 12px;
}

.header-content h1 {
    color: #ff6600;
    margin: 0 0 0.5rem 0;
    font-size: 2rem;
}

.header-content p {
    color: #94a3b8;
    margin: 0;
}

.btn-secondary {
    background: rgba(255, 102, 0, 0.2);
    color: #ff6600;
    border: 1px solid #ff6600;
    padding: 0.75rem 1.5rem;
    border-radius: 8px;
    text-decoration: none;
    font-weight: 600;
    transition: all 0.3s ease;
}

.btn-secondary:hover {
    background: #ff6600;
    color: #1a1a2e;
}

.form-grid {
    display: grid;
    grid-template-columns: 2fr 1fr;
    gap: 2rem;
}

.form-section {
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 255, 255, 0.1);
    border-radius: 12px;
    padding: 1.5rem;
    margin-bottom: 2rem;
}

.form-section h3 {
    color: #ff6600;
    margin: 0 0 1.5rem 0;
    font-size: 1.2rem;
}

.form-group {
    margin-bottom: 1.5rem;
}

.form-group label {
    display: block;
    color: #e2e8f0;
    font-weight: 600;
    margin-bottom: 0.5rem;
}

.form-group input,
.form-group select,
.form-group textarea {
    width: 100%;
    background: rgba(255, 255, 255, 0.1);
    border: 1px solid rgba(255, 255, 255, 0.2);
    border-radius: 8px;
    padding: 0.75rem;
    color: #e2e8f0;
    font-size: 1rem;
    transition: all 0.3s ease;
}

.form-group input:focus,
.form-group select:focus,
.form-group textarea:focus {
    outline: none;
    border-color: #ff6600;
    box-shadow: 0 0 0 3px rgba(255, 102, 0, 0.1);
}

.form-group small {
    display: block;
    color: #94a3b8;
    font-size: 0.875rem;
    margin-top: 0.5rem;
}

.error {
    color: #ef4444;
    font-size: 0.875rem;
    margin-top: 0.5rem;
    display: block;
}

.radio-group {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.radio-option {
    display: flex;
    align-items: flex-start;
    gap: 0.75rem;
    padding: 1rem;
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 255, 255, 0.1);
    border-radius: 8px;
    cursor: pointer;
    transition: all 0.3s ease;
}

.radio-option:hover {
    border-color: rgba(255, 102, 0, 0.3);
}

.radio-option input[type="radio"] {
    width: auto;
    margin: 0;
}

.radio-label {
    font-weight: 600;
    color: #e2e8f0;
}

.radio-option small {
    display: block;
    color: #94a3b8;
    margin-top: 0.25rem;
}

.targeting-section {
    margin-top: 1rem;
    padding: 1rem;
    background: rgba(255, 102, 0, 0.05);
    border: 1px solid rgba(255, 102, 0, 0.2);
    border-radius: 8px;
}

.checkbox-group {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
    gap: 0.75rem;
    margin-top: 0.5rem;
}

.checkbox-option {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    cursor: pointer;
}

.checkbox-option input[type="checkbox"] {
    width: auto;
    margin: 0;
}

.checkbox-label {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    cursor: pointer;
}

.checkbox-label input[type="checkbox"] {
    width: auto;
    margin: 0;
}

.notification-preview {
    background: rgba(255, 255, 255, 0.1);
    border: 1px solid rgba(255, 255, 255, 0.2);
    border-radius: 8px;
    padding: 1rem;
}

.preview-notification {
    background: rgba(255, 102, 0, 0.1);
    border: 1px solid rgba(255, 102, 0, 0.3);
    border-radius: 8px;
    padding: 1rem;
}

.preview-header {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    margin-bottom: 0.75rem;
}

.preview-icon {
    font-size: 1.25rem;
}

.preview-title {
    font-weight: 600;
    color: #ff6600;
}

.preview-message {
    color: #e2e8f0;
    line-height: 1.5;
}

.form-actions {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.btn-primary {
    background: linear-gradient(135deg, #ff6600, #cc5500);
    color: white;
    border: none;
    padding: 0.75rem 1.5rem;
    border-radius: 8px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    display: inline-flex;
    align-items: center;
    justify-content: center;
    gap: 0.5rem;
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(255, 102, 0, 0.3);
}

@media (max-width: 1024px) {
    .form-grid {
        grid-template-columns: 1fr;
    }
    
    .header-section {
        flex-direction: column;
        gap: 1rem;
        text-align: center;
    }
}

@media (max-width: 768px) {
    .radio-group {
        gap: 0.75rem;
    }
    
    .checkbox-group {
        grid-template-columns: repeat(auto-fit, minmax(100px, 1fr));
    }
}
</style>
@endsection