@extends('layouts.admin')

@section('title', 'Popup Messages')
@section('page-title', 'Popup Messages')
@section('page-subtitle', 'Manage popup notifications and alerts')

@section('content')
<div class="p-6">
    <div class="max-w-7xl mx-auto">
        <div class="header-section">
            <div class="header-content">
                <h1>🔔 Popup Notifications</h1>
                <p>Manage announcements and important messages for users</p>
            </div>
            <div class="header-actions">
                <a href="{{ route('admin.popup-notifications.create') }}" class="btn-primary">
                    <span>+</span> Create Notification
                </a>
            </div>
        </div>

        <!-- Filters -->
        <div class="filters-section">
            <form method="GET" class="filters-form">
                <div class="filter-group">
                    <select name="type" onchange="this.form.submit()">
                        <option value="">All Types</option>
                        <option value="info" {{ request('type') === 'info' ? 'selected' : '' }}>📢 Info</option>
                        <option value="warning" {{ request('type') === 'warning' ? 'selected' : '' }}>⚠️ Warning</option>
                        <option value="success" {{ request('type') === 'success' ? 'selected' : '' }}>✅ Success</option>
                        <option value="announcement" {{ request('type') === 'announcement' ? 'selected' : '' }}>🎉 Announcement</option>
                    </select>
                </div>
                <div class="filter-group">
                    <select name="status" onchange="this.form.submit()">
                        <option value="">All Status</option>
                        <option value="active" {{ request('status') === 'active' ? 'selected' : '' }}>Active</option>
                        <option value="inactive" {{ request('status') === 'inactive' ? 'selected' : '' }}>Inactive</option>
                    </select>
                </div>
            </form>
        </div>

        <!-- Notifications Table -->
        <div class="notifications-table">
            @if($notifications->count() > 0)
                <div class="table-responsive">
                    <table>
                        <thead>
                            <tr>
                                <th>Notification</th>
                                <th>Type</th>
                                <th>Status</th>
                                <th>Target</th>
                                <th>Schedule</th>
                                <th>Views</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($notifications as $notification)
                                <tr>
                                    <td>
                                        <div class="notification-info">
                                            <div class="notification-title">
                                                {{ $notification->icon }} {{ $notification->title }}
                                            </div>
                                            <div class="notification-message">
                                                {{ Str::limit($notification->message, 80) }}
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <span class="type-badge type-{{ $notification->type }}">
                                            {{ ucfirst($notification->type) }}
                                        </span>
                                    </td>
                                    <td>
                                        <div class="status-info">
                                            <span class="status-badge status-{{ $notification->is_active ? 'active' : 'inactive' }}">
                                                {{ $notification->is_active ? 'Active' : 'Inactive' }}
                                            </span>
                                            @if($notification->start_date && $notification->start_date > now())
                                                <small class="scheduled">Scheduled</small>
                                            @elseif($notification->end_date && $notification->end_date < now())
                                                <small class="expired">Expired</small>
                                            @endif
                                        </div>
                                    </td>
                                    <td>
                                        @if($notification->target_users)
                                            @if(isset($notification->target_users['levels']))
                                                <span class="target-badge">Levels {{ implode(', ', $notification->target_users['levels']) }}</span>
                                            @elseif(isset($notification->target_users['user_ids']))
                                                <span class="target-badge">{{ count($notification->target_users['user_ids']) }} Users</span>
                                            @elseif(isset($notification->target_users['min_level']))
                                                <span class="target-badge">Level {{ $notification->target_users['min_level'] }}+</span>
                                            @endif
                                        @else
                                            <span class="target-badge all-users">All Users</span>
                                        @endif
                                    </td>
                                    <td>
                                        <div class="schedule-info">
                                            @if($notification->start_date)
                                                <small>Start: {{ $notification->start_date->format('M j, Y') }}</small>
                                            @endif
                                            @if($notification->end_date)
                                                <small>End: {{ $notification->end_date->format('M j, Y') }}</small>
                                            @endif
                                            @if(!$notification->start_date && !$notification->end_date)
                                                <small class="text-muted">Immediate</small>
                                            @endif
                                        </div>
                                    </td>
                                    <td>
                                        <div class="views-info">
                                            <strong>{{ $notification->seenBy()->count() }}</strong>
                                            <small>views</small>
                                        </div>
                                    </td>
                                    <td>
                                        <div class="actions">
                                            <a href="{{ route('admin.popup-notifications.show', $notification) }}" class="btn-sm">View</a>
                                            <a href="{{ route('admin.popup-notifications.edit', $notification) }}" class="btn-sm btn-warning">Edit</a>
                                            <form method="POST" action="{{ route('admin.popup-notifications.toggle', $notification) }}" style="display: inline;">
                                                @csrf
                                                <button type="submit" class="btn-sm {{ $notification->is_active ? 'btn-secondary' : 'btn-success' }}">
                                                    {{ $notification->is_active ? 'Deactivate' : 'Activate' }}
                                                </button>
                                            </form>
                                            <form method="POST" action="{{ route('admin.popup-notifications.destroy', $notification) }}" style="display: inline;" onsubmit="return confirm('Are you sure you want to delete this notification?')">
                                                @csrf
                                                @method('DELETE')
                                                <button type="submit" class="btn-sm btn-danger">Delete</button>
                                            </form>
                                        </div>
                                    </td>
                                </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>

                <!-- Pagination -->
                <div class="pagination-wrapper">
                    {{ $notifications->appends(request()->query())->links() }}
                </div>
            @else
                <div class="empty-state">
                    <div class="empty-icon">🔔</div>
                    <h3>No notifications found</h3>
                    <p>Create your first popup notification to engage users!</p>
                    <a href="{{ route('admin.popup-notifications.create') }}" class="btn-primary">Create First Notification</a>
                </div>
            @endif
        </div>
    </div>
</div>

<style>
.admin-popup-notifications {
    padding: 2rem 0;
    background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
    min-height: 100vh;
    color: #e2e8f0;
}

.header-section {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 2rem;
    padding: 1.5rem;
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 102, 0, 0.2);
    border-radius: 12px;
}

.header-content h1 {
    color: #ff6600;
    margin: 0 0 0.5rem 0;
    font-size: 2rem;
}

.header-content p {
    color: #94a3b8;
    margin: 0;
}

.btn-primary {
    background: linear-gradient(135deg, #ff6600, #cc5500);
    color: white;
    border: none;
    padding: 0.75rem 1.5rem;
    border-radius: 8px;
    text-decoration: none;
    font-weight: 600;
    transition: all 0.3s ease;
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(255, 102, 0, 0.3);
}

.filters-section {
    margin-bottom: 2rem;
    padding: 1.5rem;
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 255, 255, 0.1);
    border-radius: 12px;
}

.filters-form {
    display: flex;
    gap: 1rem;
    align-items: center;
    flex-wrap: wrap;
}

.filter-group select {
    background: rgba(255, 255, 255, 0.1);
    border: 1px solid rgba(255, 255, 255, 0.2);
    border-radius: 6px;
    padding: 0.5rem 0.75rem;
    color: #e2e8f0;
    min-width: 150px;
}

.filter-group select:focus {
    outline: none;
    border-color: #ff6600;
    box-shadow: 0 0 0 2px rgba(255, 102, 0, 0.2);
}

.notifications-table {
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 255, 255, 0.1);
    border-radius: 12px;
    overflow: hidden;
}

.table-responsive {
    overflow-x: auto;
}

table {
    width: 100%;
    border-collapse: collapse;
}

th {
    background: rgba(255, 102, 0, 0.1);
    color: #ff6600;
    padding: 1rem;
    text-align: left;
    font-weight: 600;
    border-bottom: 1px solid rgba(255, 102, 0, 0.2);
}

td {
    padding: 1rem;
    border-bottom: 1px solid rgba(255, 255, 255, 0.1);
    vertical-align: top;
}

.notification-info {
    max-width: 300px;
}

.notification-title {
    font-weight: 600;
    color: #e2e8f0;
    margin-bottom: 0.5rem;
}

.notification-message {
    color: #94a3b8;
    font-size: 0.9rem;
    line-height: 1.4;
}

.type-badge,
.status-badge,
.target-badge {
    padding: 0.25rem 0.75rem;
    border-radius: 20px;
    font-size: 0.8rem;
    font-weight: 600;
    text-transform: uppercase;
}

.type-info { background: rgba(59, 130, 246, 0.2); color: #3b82f6; }
.type-warning { background: rgba(245, 158, 11, 0.2); color: #f59e0b; }
.type-success { background: rgba(34, 197, 94, 0.2); color: #22c55e; }
.type-announcement { background: rgba(236, 72, 153, 0.2); color: #ec4899; }

.status-active { background: rgba(34, 197, 94, 0.2); color: #22c55e; }
.status-inactive { background: rgba(107, 114, 128, 0.2); color: #6b7280; }

.target-badge {
    background: rgba(139, 92, 246, 0.2);
    color: #8b5cf6;
}

.all-users {
    background: rgba(255, 102, 0, 0.2);
    color: #ff6600;
}

.status-info {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.scheduled {
    color: #f59e0b;
    font-size: 0.75rem;
}

.expired {
    color: #ef4444;
    font-size: 0.75rem;
}

.schedule-info {
    display: flex;
    flex-direction: column;
    gap: 0.25rem;
}

.schedule-info small {
    color: #94a3b8;
    font-size: 0.8rem;
}

.views-info {
    text-align: center;
}

.views-info strong {
    color: #ff6600;
    font-size: 1.2rem;
    display: block;
}

.views-info small {
    color: #94a3b8;
}

.actions {
    display: flex;
    gap: 0.5rem;
    flex-wrap: wrap;
}

.btn-sm {
    padding: 0.375rem 0.75rem;
    border-radius: 6px;
    text-decoration: none;
    font-size: 0.875rem;
    border: 1px solid;
    cursor: pointer;
    transition: all 0.3s ease;
    background: rgba(255, 102, 0, 0.2);
    color: #ff6600;
    border-color: #ff6600;
}

.btn-sm:hover {
    background: #ff6600;
    color: #1a1a2e;
}

.btn-warning {
    background: rgba(245, 158, 11, 0.2) !important;
    color: #f59e0b !important;
    border-color: #f59e0b !important;
}

.btn-warning:hover {
    background: #f59e0b !important;
    color: #1a1a2e !important;
}

.btn-success {
    background: rgba(34, 197, 94, 0.2) !important;
    color: #22c55e !important;
    border-color: #22c55e !important;
}

.btn-success:hover {
    background: #22c55e !important;
    color: #1a1a2e !important;
}

.btn-secondary {
    background: rgba(107, 114, 128, 0.2) !important;
    color: #6b7280 !important;
    border-color: #6b7280 !important;
}

.btn-secondary:hover {
    background: #6b7280 !important;
    color: white !important;
}

.btn-danger {
    background: rgba(239, 68, 68, 0.2) !important;
    color: #ef4444 !important;
    border-color: #ef4444 !important;
}

.btn-danger:hover {
    background: #ef4444 !important;
    color: white !important;
}

.text-muted {
    color: #6b7280;
}

.empty-state {
    text-align: center;
    padding: 4rem 2rem;
}

.empty-icon {
    font-size: 4rem;
    margin-bottom: 1rem;
}

.empty-state h3 {
    color: #e2e8f0;
    margin-bottom: 0.5rem;
}

.empty-state p {
    color: #94a3b8;
    margin-bottom: 2rem;
}

.pagination-wrapper {
    padding: 1rem;
    border-top: 1px solid rgba(255, 255, 255, 0.1);
}

@media (max-width: 768px) {
    .header-section {
        flex-direction: column;
        gap: 1rem;
        text-align: center;
    }
    
    .filters-form {
        flex-direction: column;
        align-items: stretch;
    }
    
    .actions {
        flex-direction: column;
    }
    
    .notification-info {
        max-width: none;
    }
}
</style>
@endsection