@extends('layouts.app')

@section('title', 'Edit Post')

@section('content')
<div class="admin-edit-post">
    <div class="container">
        <div class="header-section">
            <div class="header-content">
                <h1>✏️ Edit Post</h1>
                <p>Update your content</p>
            </div>
            <div class="header-actions">
                <a href="{{ route('admin.posts.index') }}" class="btn-secondary">
                    ← Back to Posts
                </a>
                @if($post->status === 'published')
                    <a href="{{ route('posts.show', $post->slug) }}" class="btn-info" target="_blank">
                        👁️ View Public
                    </a>
                @endif
            </div>
        </div>

        <form method="POST" action="{{ route('admin.posts.update', $post) }}" enctype="multipart/form-data" class="post-form">
            @csrf
            @method('PUT')
            
            <div class="form-grid">
                <!-- Main Content -->
                <div class="main-content">
                    <div class="form-section">
                        <h3>📝 Content Details</h3>
                        
                        <div class="form-group">
                            <label for="title">Title *</label>
                            <input type="text" name="title" id="title" value="{{ old('title', $post->title) }}" required maxlength="255">
                            @error('title')
                                <span class="error">{{ $message }}</span>
                            @enderror
                        </div>

                        <div class="form-group">
                            <label for="excerpt">Excerpt</label>
                            <textarea name="excerpt" id="excerpt" rows="3" maxlength="500" placeholder="Brief description of the post (optional)">{{ old('excerpt', $post->excerpt) }}</textarea>
                            @error('excerpt')
                                <span class="error">{{ $message }}</span>
                            @enderror
                        </div>

                        <div class="form-group">
                            <label for="content">Content *</label>
                            <textarea name="content" id="content" rows="15" required placeholder="Write your post content here...">{{ old('content', $post->content) }}</textarea>
                            @error('content')
                                <span class="error">{{ $message }}</span>
                            @enderror
                        </div>

                        <div class="form-group">
                            <label for="featured_image">Featured Image</label>
                            @if($post->featured_image)
                                <div class="current-image">
                                    <img src="{{ asset('storage/' . $post->featured_image) }}" alt="Current featured image" style="max-width: 300px; border-radius: 8px; margin-bottom: 1rem;">
                                    <p><small>Current image - upload a new one to replace</small></p>
                                </div>
                            @endif
                            <input type="file" name="featured_image" id="featured_image" accept="image/*">
                            <small>Recommended size: 1200x630px. Max size: 2MB</small>
                            @error('featured_image')
                                <span class="error">{{ $message }}</span>
                            @enderror
                        </div>
                    </div>
                </div>

                <!-- Sidebar -->
                <div class="sidebar">
                    <div class="form-section">
                        <h3>⚙️ Post Settings</h3>
                        
                        <div class="form-group">
                            <label for="type">Post Type *</label>
                            <select name="type" id="type" required>
                                <option value="">Select Type</option>
                                <option value="blog" {{ old('type', $post->type) === 'blog' ? 'selected' : '' }}>📖 Blog Post</option>
                                <option value="news" {{ old('type', $post->type) === 'news' ? 'selected' : '' }}>📰 News Update</option>
                                <option value="highlight" {{ old('type', $post->type) === 'highlight' ? 'selected' : '' }}>🏆 Tournament Highlight</option>
                                <option value="tournament" {{ old('type', $post->type) === 'tournament' ? 'selected' : '' }}>🎮 Tournament Announcement</option>
                                <option value="update" {{ old('type', $post->type) === 'update' ? 'selected' : '' }}>🔄 Platform Update</option>
                            </select>
                            @error('type')
                                <span class="error">{{ $message }}</span>
                            @enderror
                        </div>

                        <div class="form-group">
                            <label for="status">Status *</label>
                            <select name="status" id="status" required>
                                <option value="draft" {{ old('status', $post->status) === 'draft' ? 'selected' : '' }}>📝 Draft</option>
                                <option value="published" {{ old('status', $post->status) === 'published' ? 'selected' : '' }}>✅ Published</option>
                                <option value="archived" {{ old('status', $post->status) === 'archived' ? 'selected' : '' }}>📦 Archived</option>
                            </select>
                            @error('status')
                                <span class="error">{{ $message }}</span>
                            @enderror
                        </div>

                        <div class="form-group">
                            <label for="published_at">Publish Date</label>
                            <input type="datetime-local" name="published_at" id="published_at" value="{{ old('published_at', $post->published_at ? $post->published_at->format('Y-m-d\TH:i') : '') }}">
                            <small>Leave empty to publish immediately when status is "Published"</small>
                            @error('published_at')
                                <span class="error">{{ $message }}</span>
                            @enderror
                        </div>
                    </div>

                    <div class="form-section">
                        <h3>📊 Post Statistics</h3>
                        <div class="stats-grid">
                            <div class="stat-item">
                                <span class="stat-label">Views</span>
                                <span class="stat-value">{{ number_format($post->views) }}</span>
                            </div>
                            <div class="stat-item">
                                <span class="stat-label">Created</span>
                                <span class="stat-value">{{ $post->created_at->format('M j, Y') }}</span>
                            </div>
                            @if($post->published_at)
                                <div class="stat-item">
                                    <span class="stat-label">Published</span>
                                    <span class="stat-value">{{ $post->published_at->format('M j, Y') }}</span>
                                </div>
                            @endif
                        </div>
                    </div>

                    <div class="form-section">
                        <h3>🎯 Additional Metadata</h3>
                        <div id="metadata-container">
                            <div class="metadata-pair">
                                <input type="text" name="metadata[tournament_date]" placeholder="Tournament Date" value="{{ old('metadata.tournament_date', $post->metadata['tournament_date'] ?? '') }}">
                                <input type="text" name="metadata[prize_pool]" placeholder="Prize Pool" value="{{ old('metadata.prize_pool', $post->metadata['prize_pool'] ?? '') }}">
                            </div>
                            <div class="metadata-pair">
                                <input type="text" name="metadata[game_title]" placeholder="Game Title" value="{{ old('metadata.game_title', $post->metadata['game_title'] ?? '') }}">
                                <input type="text" name="metadata[participants]" placeholder="Participants" value="{{ old('metadata.participants', $post->metadata['participants'] ?? '') }}">
                            </div>
                        </div>
                        <small>Add relevant metadata like tournament details, dates, etc.</small>
                    </div>

                    <div class="form-actions">
                        <button type="submit" name="action" value="save" class="btn-primary">
                            💾 Update Post
                        </button>
                        <button type="submit" name="action" value="preview" class="btn-secondary">
                            👁️ Update & Preview
                        </button>
                    </div>
                </div>
            </div>
        </form>
    </div>
</div>

<style>
.admin-edit-post {
    padding: 2rem 0;
    background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
    min-height: 100vh;
    color: #e2e8f0;
}

.header-section {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 2rem;
    padding: 1.5rem;
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 102, 0, 0.2);
    border-radius: 12px;
}

.header-content h1 {
    color: #ff6600;
    margin: 0 0 0.5rem 0;
    font-size: 2rem;
}

.header-content p {
    color: #94a3b8;
    margin: 0;
}

.header-actions {
    display: flex;
    gap: 1rem;
}

.btn-secondary, .btn-info {
    padding: 0.75rem 1.5rem;
    border-radius: 8px;
    text-decoration: none;
    font-weight: 600;
    transition: all 0.3s ease;
}

.btn-secondary {
    background: rgba(255, 102, 0, 0.2);
    color: #ff6600;
    border: 1px solid #ff6600;
}

.btn-secondary:hover {
    background: #ff6600;
    color: #1a1a2e;
}

.btn-info {
    background: rgba(59, 130, 246, 0.2);
    color: #3b82f6;
    border: 1px solid #3b82f6;
}

.btn-info:hover {
    background: #3b82f6;
    color: white;
}

.form-grid {
    display: grid;
    grid-template-columns: 2fr 1fr;
    gap: 2rem;
}

.form-section {
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 255, 255, 0.1);
    border-radius: 12px;
    padding: 1.5rem;
    margin-bottom: 2rem;
}

.form-section h3 {
    color: #ff6600;
    margin: 0 0 1.5rem 0;
    font-size: 1.2rem;
}

.form-group {
    margin-bottom: 1.5rem;
}

.form-group label {
    display: block;
    color: #e2e8f0;
    font-weight: 600;
    margin-bottom: 0.5rem;
}

.form-group input,
.form-group select,
.form-group textarea {
    width: 100%;
    background: rgba(255, 255, 255, 0.1);
    border: 1px solid rgba(255, 255, 255, 0.2);
    border-radius: 8px;
    padding: 0.75rem;
    color: #e2e8f0;
    font-size: 1rem;
    transition: all 0.3s ease;
}

.form-group input:focus,
.form-group select:focus,
.form-group textarea:focus {
    outline: none;
    border-color: #ff6600;
    box-shadow: 0 0 0 3px rgba(255, 102, 0, 0.1);
}

.form-group small {
    display: block;
    color: #94a3b8;
    font-size: 0.875rem;
    margin-top: 0.5rem;
}

.error {
    color: #ef4444;
    font-size: 0.875rem;
    margin-top: 0.5rem;
    display: block;
}

.current-image img {
    border: 2px solid rgba(255, 102, 0, 0.3);
}

.metadata-pair {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 0.5rem;
    margin-bottom: 0.5rem;
}

.metadata-pair input {
    margin-bottom: 0;
}

.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
    gap: 1rem;
}

.stat-item {
    text-align: center;
    padding: 1rem;
    background: rgba(255, 102, 0, 0.1);
    border: 1px solid rgba(255, 102, 0, 0.2);
    border-radius: 8px;
}

.stat-label {
    display: block;
    color: #94a3b8;
    font-size: 0.875rem;
    margin-bottom: 0.5rem;
}

.stat-value {
    display: block;
    color: #ff6600;
    font-weight: 600;
    font-size: 1.1rem;
}

.form-actions {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.btn-primary {
    background: linear-gradient(135deg, #ff6600, #cc5500);
    color: white;
    border: none;
    padding: 0.75rem 1.5rem;
    border-radius: 8px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    display: inline-flex;
    align-items: center;
    justify-content: center;
    gap: 0.5rem;
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(255, 102, 0, 0.3);
}

@media (max-width: 1024px) {
    .form-grid {
        grid-template-columns: 1fr;
    }
    
    .header-section {
        flex-direction: column;
        gap: 1rem;
        text-align: center;
    }
}

@media (max-width: 768px) {
    .metadata-pair {
        grid-template-columns: 1fr;
    }
    
    .header-actions {
        flex-direction: column;
    }
}
</style>
@endsection