@extends('layouts.admin')

@section('title', 'Content Management')
@section('page-title', 'Content Management')
@section('page-subtitle', 'Manage blog posts and news articles')

@section('content')
<div class="p-6">
    <div class="max-w-7xl mx-auto">
        <div class="header-section">
            <div class="header-content">
                <h1>📝 Content Management</h1>
                <p>Manage blog posts, news, highlights, and announcements</p>
            </div>
            <div class="header-actions">
                <a href="{{ route('admin.posts.create') }}" class="btn-primary">
                    <span>+</span> Create New Post
                </a>
            </div>
        </div>

        <!-- Filters -->
        <div class="filters-section">
            <form method="GET" class="filters-form">
                <div class="filter-group">
                    <select name="type" onchange="this.form.submit()">
                        <option value="">All Types</option>
                        <option value="blog" {{ request('type') === 'blog' ? 'selected' : '' }}>Blog</option>
                        <option value="news" {{ request('type') === 'news' ? 'selected' : '' }}>News</option>
                        <option value="highlight" {{ request('type') === 'highlight' ? 'selected' : '' }}>Highlights</option>
                        <option value="tournament" {{ request('type') === 'tournament' ? 'selected' : '' }}>Tournaments</option>
                        <option value="update" {{ request('type') === 'update' ? 'selected' : '' }}>Updates</option>
                    </select>
                </div>
                <div class="filter-group">
                    <select name="status" onchange="this.form.submit()">
                        <option value="">All Status</option>
                        <option value="draft" {{ request('status') === 'draft' ? 'selected' : '' }}>Draft</option>
                        <option value="published" {{ request('status') === 'published' ? 'selected' : '' }}>Published</option>
                        <option value="archived" {{ request('status') === 'archived' ? 'selected' : '' }}>Archived</option>
                    </select>
                </div>
                <div class="filter-group">
                    <input type="text" name="search" placeholder="Search posts..." value="{{ request('search') }}">
                    <button type="submit" class="btn-secondary">Search</button>
                </div>
            </form>
        </div>

        <!-- Posts Table -->
        <div class="posts-table">
            @if($posts->count() > 0)
                <div class="table-responsive">
                    <table>
                        <thead>
                            <tr>
                                <th>Title</th>
                                <th>Type</th>
                                <th>Status</th>
                                <th>Author</th>
                                <th>Published</th>
                                <th>Views</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($posts as $post)
                                <tr>
                                    <td>
                                        <div class="post-title">
                                            <strong>{{ $post->title }}</strong>
                                            @if($post->featured_image)
                                                <span class="image-indicator">📷</span>
                                            @endif
                                        </div>
                                    </td>
                                    <td>
                                        <span class="type-badge type-{{ $post->type }}">
                                            {{ ucfirst($post->type) }}
                                        </span>
                                    </td>
                                    <td>
                                        <span class="status-badge status-{{ $post->status }}">
                                            {{ ucfirst($post->status) }}
                                        </span>
                                    </td>
                                    <td>{{ $post->author->username }}</td>
                                    <td>
                                        @if($post->published_at)
                                            {{ $post->published_at->format('M j, Y') }}
                                        @else
                                            <span class="text-muted">Not published</span>
                                        @endif
                                    </td>
                                    <td>{{ number_format($post->views) }}</td>
                                    <td>
                                        <div class="actions">
                                            <a href="{{ route('admin.posts.show', $post) }}" class="btn-sm">View</a>
                                            <a href="{{ route('admin.posts.edit', $post) }}" class="btn-sm btn-warning">Edit</a>
                                            @if($post->status === 'published')
                                                <a href="{{ route('posts.show', $post->slug) }}" class="btn-sm btn-info" target="_blank">Public</a>
                                            @endif
                                            <form method="POST" action="{{ route('admin.posts.destroy', $post) }}" style="display: inline;" onsubmit="return confirm('Are you sure you want to delete this post?')">
                                                @csrf
                                                @method('DELETE')
                                                <button type="submit" class="btn-sm btn-danger">Delete</button>
                                            </form>
                                        </div>
                                    </td>
                                </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>

                <!-- Pagination -->
                <div class="pagination-wrapper">
                    {{ $posts->appends(request()->query())->links() }}
                </div>
            @else
                <div class="empty-state">
                    <div class="empty-icon">📝</div>
                    <h3>No posts found</h3>
                    <p>Start creating content to engage your community!</p>
                    <a href="{{ route('admin.posts.create') }}" class="btn-primary">Create First Post</a>
                </div>
            @endif
        </div>
    </div>
</div>

<style>
.admin-posts {
    padding: 2rem 0;
    background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
    min-height: 100vh;
    color: #e2e8f0;
}

.header-section {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 2rem;
    padding: 1.5rem;
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 102, 0, 0.2);
    border-radius: 12px;
}

.header-content h1 {
    color: #ff6600;
    margin: 0 0 0.5rem 0;
    font-size: 2rem;
}

.header-content p {
    color: #94a3b8;
    margin: 0;
}

.btn-primary {
    background: linear-gradient(135deg, #ff6600, #cc5500);
    color: white;
    border: none;
    padding: 0.75rem 1.5rem;
    border-radius: 8px;
    text-decoration: none;
    font-weight: 600;
    transition: all 0.3s ease;
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(255, 102, 0, 0.3);
}

.filters-section {
    margin-bottom: 2rem;
    padding: 1.5rem;
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 255, 255, 0.1);
    border-radius: 12px;
}

.filters-form {
    display: flex;
    gap: 1rem;
    align-items: center;
    flex-wrap: wrap;
}

.filter-group {
    display: flex;
    gap: 0.5rem;
    align-items: center;
}

.filter-group select,
.filter-group input {
    background: rgba(255, 255, 255, 0.1);
    border: 1px solid rgba(255, 255, 255, 0.2);
    border-radius: 6px;
    padding: 0.5rem 0.75rem;
    color: #e2e8f0;
    min-width: 150px;
}

.filter-group select:focus,
.filter-group input:focus {
    outline: none;
    border-color: #ff6600;
    box-shadow: 0 0 0 2px rgba(255, 102, 0, 0.2);
}

.btn-secondary {
    background: rgba(255, 102, 0, 0.2);
    color: #ff6600;
    border: 1px solid #ff6600;
    padding: 0.5rem 1rem;
    border-radius: 6px;
    cursor: pointer;
    transition: all 0.3s ease;
}

.btn-secondary:hover {
    background: #ff6600;
    color: #1a1a2e;
}

.posts-table {
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 255, 255, 0.1);
    border-radius: 12px;
    overflow: hidden;
}

.table-responsive {
    overflow-x: auto;
}

table {
    width: 100%;
    border-collapse: collapse;
}

th {
    background: rgba(255, 102, 0, 0.1);
    color: #ff6600;
    padding: 1rem;
    text-align: left;
    font-weight: 600;
    border-bottom: 1px solid rgba(255, 102, 0, 0.2);
}

td {
    padding: 1rem;
    border-bottom: 1px solid rgba(255, 255, 255, 0.1);
}

.post-title {
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.image-indicator {
    background: rgba(255, 102, 0, 0.2);
    color: #ff6600;
    padding: 0.25rem 0.5rem;
    border-radius: 4px;
    font-size: 0.8rem;
}

.type-badge,
.status-badge {
    padding: 0.25rem 0.75rem;
    border-radius: 20px;
    font-size: 0.8rem;
    font-weight: 600;
    text-transform: uppercase;
}

.type-blog { background: rgba(59, 130, 246, 0.2); color: #3b82f6; }
.type-news { background: rgba(16, 185, 129, 0.2); color: #10b981; }
.type-highlight { background: rgba(245, 158, 11, 0.2); color: #f59e0b; }
.type-tournament { background: rgba(139, 92, 246, 0.2); color: #8b5cf6; }
.type-update { background: rgba(236, 72, 153, 0.2); color: #ec4899; }

.status-draft { background: rgba(107, 114, 128, 0.2); color: #6b7280; }
.status-published { background: rgba(34, 197, 94, 0.2); color: #22c55e; }
.status-archived { background: rgba(239, 68, 68, 0.2); color: #ef4444; }

.actions {
    display: flex;
    gap: 0.5rem;
    flex-wrap: wrap;
}

.btn-sm {
    padding: 0.375rem 0.75rem;
    border-radius: 6px;
    text-decoration: none;
    font-size: 0.875rem;
    border: 1px solid;
    cursor: pointer;
    transition: all 0.3s ease;
}

.btn-sm {
    background: rgba(255, 102, 0, 0.2);
    color: #ff6600;
    border-color: #ff6600;
}

.btn-sm:hover {
    background: #ff6600;
    color: #1a1a2e;
}

.btn-warning {
    background: rgba(245, 158, 11, 0.2) !important;
    color: #f59e0b !important;
    border-color: #f59e0b !important;
}

.btn-warning:hover {
    background: #f59e0b !important;
    color: #1a1a2e !important;
}

.btn-info {
    background: rgba(59, 130, 246, 0.2) !important;
    color: #3b82f6 !important;
    border-color: #3b82f6 !important;
}

.btn-info:hover {
    background: #3b82f6 !important;
    color: white !important;
}

.btn-danger {
    background: rgba(239, 68, 68, 0.2) !important;
    color: #ef4444 !important;
    border-color: #ef4444 !important;
}

.btn-danger:hover {
    background: #ef4444 !important;
    color: white !important;
}

.text-muted {
    color: #6b7280;
}

.empty-state {
    text-align: center;
    padding: 4rem 2rem;
}

.empty-icon {
    font-size: 4rem;
    margin-bottom: 1rem;
}

.empty-state h3 {
    color: #e2e8f0;
    margin-bottom: 0.5rem;
}

.empty-state p {
    color: #94a3b8;
    margin-bottom: 2rem;
}

.pagination-wrapper {
    padding: 1rem;
    border-top: 1px solid rgba(255, 255, 255, 0.1);
}

@media (max-width: 768px) {
    .header-section {
        flex-direction: column;
        gap: 1rem;
        text-align: center;
    }
    
    .filters-form {
        flex-direction: column;
        align-items: stretch;
    }
    
    .filter-group {
        flex-direction: column;
    }
    
    .actions {
        flex-direction: column;
    }
}
</style>
@endsection