@extends('layouts.app')

@section('title', 'Pending Posts Approval')

@section('content')
<div class="admin-pending-posts">
    <div class="container">
        <div class="page-header">
            <div class="header-content">
                <h1>📝 Pending Posts Approval</h1>
                <p>Review and approve posts submitted by creators</p>
            </div>
            <div class="header-actions">
                <a href="{{ route('admin.posts.index') }}" class="btn-secondary">All Posts</a>
                <a href="{{ route('admin.posts.create') }}" class="btn-primary">Create Post</a>
            </div>
        </div>

        @if(session('success'))
            <div class="alert alert-success">
                <i class="icon">✅</i>
                {{ session('success') }}
            </div>
        @endif

        @if(session('error'))
            <div class="alert alert-error">
                <i class="icon">❌</i>
                {{ session('error') }}
            </div>
        @endif

        <div class="posts-container">
            @if($posts->count() > 0)
                <div class="posts-grid">
                    @foreach($posts as $post)
                        <div class="post-card">
                            <div class="post-header">
                                <div class="post-meta">
                                    <span class="post-type type-{{ $post->type }}">{{ ucfirst($post->type) }}</span>
                                    <span class="post-date">{{ $post->created_at->diffForHumans() }}</span>
                                </div>
                                <div class="post-author">
                                    <div class="author-avatar">
                                        {{ strtoupper(substr($post->author->username, 0, 1)) }}
                                    </div>
                                    <div class="author-info">
                                        <h4>{{ $post->author->username }}</h4>
                                        <span class="author-role">{{ $post->author->role ? $post->author->role->getDisplayName() : 'User' }}</span>
                                    </div>
                                </div>
                            </div>

                            @if($post->featured_image)
                                <div class="post-image">
                                    <img src="{{ Storage::url($post->featured_image) }}" alt="{{ $post->title }}">
                                </div>
                            @endif

                            <div class="post-content">
                                <h3 class="post-title">{{ $post->title }}</h3>
                                
                                @if($post->excerpt)
                                    <p class="post-excerpt">{{ Str::limit($post->excerpt, 100) }}</p>
                                @else
                                    <p class="post-excerpt">{{ Str::limit(strip_tags($post->content), 150) }}</p>
                                @endif

                                <div class="post-stats">
                                    <span class="stat">
                                        <i class="icon">📝</i>
                                        {{ str_word_count(strip_tags($post->content)) }} words
                                    </span>
                                    <span class="stat">
                                        <i class="icon">🏷️</i>
                                        {{ $post->slug }}
                                    </span>
                                </div>
                            </div>

                            <div class="post-actions">
                                <div class="action-buttons">
                                    <form method="POST" action="{{ route('admin.posts.approve', $post) }}" style="display: inline;">
                                        @csrf
                                        <button type="submit" class="btn-approve" onclick="return confirm('Approve this post for publication?')">
                                            <i class="icon">✅</i>
                                            Approve
                                        </button>
                                    </form>

                                    <form method="POST" action="{{ route('admin.posts.reject', $post) }}" style="display: inline;">
                                        @csrf
                                        <button type="submit" class="btn-reject" onclick="return confirm('Reject this post? It will be returned to draft status.')">
                                            <i class="icon">❌</i>
                                            Reject
                                        </button>
                                    </form>

                                    <a href="{{ route('admin.posts.show', $post) }}" class="btn-view">
                                        <i class="icon">👁️</i>
                                        View Details
                                    </a>
                                </div>
                            </div>
                        </div>
                    @endforeach
                </div>

                <!-- Pagination -->
                @if($posts->hasPages())
                    <div class="pagination-wrapper">
                        {{ $posts->links() }}
                    </div>
                @endif
            @else
                <div class="empty-state">
                    <div class="empty-icon">📝</div>
                    <h3>No Pending Posts</h3>
                    <p>All posts have been reviewed! New creator submissions will appear here for approval.</p>
                    <a href="{{ route('admin.posts.index') }}" class="btn-primary">View All Posts</a>
                </div>
            @endif
        </div>
    </div>
</div>

<style>
.admin-pending-posts {
    padding: 2rem 0;
    background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
    min-height: 100vh;
}

.page-header {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    margin-bottom: 2rem;
    padding: 2rem;
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 102, 0, 0.2);
    border-radius: 12px;
}

.header-content h1 {
    color: #ff6600;
    font-size: 2rem;
    margin: 0 0 0.5rem 0;
}

.header-content p {
    color: #94a3b8;
    margin: 0;
    font-size: 1.1rem;
}

.header-actions {
    display: flex;
    gap: 1rem;
}

.btn-primary, .btn-secondary {
    padding: 0.75rem 1.5rem;
    border-radius: 8px;
    text-decoration: none;
    font-weight: 600;
    transition: all 0.3s ease;
    border: 1px solid;
}

.btn-primary {
    background: rgba(255, 102, 0, 0.2);
    color: #ff6600;
    border-color: #ff6600;
}

.btn-primary:hover {
    background: #ff6600;
    color: #1a1a2e;
}

.btn-secondary {
    background: rgba(255, 255, 255, 0.1);
    color: #e2e8f0;
    border-color: #e2e8f0;
}

.btn-secondary:hover {
    background: #e2e8f0;
    color: #1a1a2e;
}

.alert {
    padding: 1rem 1.5rem;
    border-radius: 8px;
    margin-bottom: 2rem;
    display: flex;
    align-items: center;
    gap: 1rem;
    border: 1px solid;
}

.alert-success {
    background: rgba(34, 197, 94, 0.1);
    color: #22c55e;
    border-color: #22c55e;
}

.alert-error {
    background: rgba(239, 68, 68, 0.1);
    color: #ef4444;
    border-color: #ef4444;
}

.posts-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(400px, 1fr));
    gap: 2rem;
    margin-bottom: 2rem;
}

.post-card {
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 102, 0, 0.2);
    border-radius: 12px;
    overflow: hidden;
    transition: all 0.3s ease;
}

.post-card:hover {
    border-color: #ff6600;
    box-shadow: 0 8px 32px rgba(255, 102, 0, 0.1);
}

.post-header {
    padding: 1.5rem;
    border-bottom: 1px solid rgba(255, 255, 255, 0.1);
}

.post-meta {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1rem;
}

.post-type {
    padding: 0.25rem 0.75rem;
    border-radius: 20px;
    font-size: 0.8rem;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.post-type.type-blog {
    background: rgba(59, 130, 246, 0.2);
    color: #3b82f6;
}

.post-type.type-news {
    background: rgba(34, 197, 94, 0.2);
    color: #22c55e;
}

.post-type.type-highlight {
    background: rgba(255, 102, 0, 0.2);
    color: #ff6600;
}

.post-type.type-tournament {
    background: rgba(147, 51, 234, 0.2);
    color: #9333ea;
}

.post-type.type-update {
    background: rgba(245, 158, 11, 0.2);
    color: #f59e0b;
}

.post-date {
    color: #64748b;
    font-size: 0.9rem;
}

.post-author {
    display: flex;
    align-items: center;
    gap: 1rem;
}

.author-avatar {
    width: 45px;
    height: 45px;
    background: linear-gradient(135deg, #ff6600, #e55a00);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    color: #1a1a2e;
    font-weight: bold;
    font-size: 1.2rem;
}

.author-info h4 {
    color: #e2e8f0;
    margin: 0 0 0.25rem 0;
    font-size: 1rem;
}

.author-role {
    color: #ff6600;
    font-size: 0.8rem;
    font-weight: 600;
    text-transform: uppercase;
}

.post-image img {
    width: 100%;
    height: 200px;
    object-fit: cover;
}

.post-content {
    padding: 1.5rem;
}

.post-title {
    color: #e2e8f0;
    margin: 0 0 1rem 0;
    font-size: 1.2rem;
    line-height: 1.4;
}

.post-excerpt {
    color: #94a3b8;
    margin: 0 0 1rem 0;
    line-height: 1.6;
}

.post-stats {
    display: flex;
    gap: 1rem;
    flex-wrap: wrap;
}

.stat {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    color: #64748b;
    font-size: 0.9rem;
}

.stat .icon {
    font-size: 1rem;
}

.post-actions {
    padding: 1.5rem;
    border-top: 1px solid rgba(255, 255, 255, 0.1);
}

.action-buttons {
    display: flex;
    gap: 0.75rem;
    flex-wrap: wrap;
}

.btn-approve, .btn-reject, .btn-view {
    padding: 0.6rem 1.2rem;
    border-radius: 6px;
    border: 1px solid;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    text-decoration: none;
    font-size: 0.9rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.btn-approve {
    background: rgba(34, 197, 94, 0.2);
    color: #22c55e;
    border-color: #22c55e;
}

.btn-approve:hover {
    background: #22c55e;
    color: #1a1a2e;
}

.btn-reject {
    background: rgba(239, 68, 68, 0.2);
    color: #ef4444;
    border-color: #ef4444;
}

.btn-reject:hover {
    background: #ef4444;
    color: #1a1a2e;
}

.btn-view {
    background: rgba(255, 255, 255, 0.1);
    color: #e2e8f0;
    border-color: #e2e8f0;
}

.btn-view:hover {
    background: #e2e8f0;
    color: #1a1a2e;
}

.empty-state {
    text-align: center;
    padding: 4rem 2rem;
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 102, 0, 0.2);
    border-radius: 12px;
}

.empty-icon {
    font-size: 4rem;
    margin-bottom: 1rem;
}

.empty-state h3 {
    color: #e2e8f0;
    margin: 0 0 1rem 0;
    font-size: 1.5rem;
}

.empty-state p {
    color: #94a3b8;
    margin: 0 0 2rem 0;
    font-size: 1.1rem;
    max-width: 500px;
    margin-left: auto;
    margin-right: auto;
}

.pagination-wrapper {
    display: flex;
    justify-content: center;
    margin-top: 2rem;
}

@media (max-width: 768px) {
    .page-header {
        flex-direction: column;
        gap: 1.5rem;
    }
    
    .posts-grid {
        grid-template-columns: 1fr;
    }
    
    .action-buttons {
        flex-direction: column;
    }
    
    .post-meta {
        flex-direction: column;
        align-items: flex-start;
        gap: 0.5rem;
    }
}
</style>
@endsection