@extends('layouts.app')

@section('title', 'View Post')

@section('content')
<div class="admin-view-post">
    <div class="container">
        <div class="header-section">
            <div class="header-content">
                <h1>👁️ View Post</h1>
                <p>Post details and management</p>
            </div>
            <div class="header-actions">
                <a href="{{ route('admin.posts.index') }}" class="btn-secondary">
                    ← Back to Posts
                </a>
                <a href="{{ route('admin.posts.edit', $post) }}" class="btn-warning">
                    ✏️ Edit Post
                </a>
                @if($post->status === 'published')
                    <a href="{{ route('posts.show', $post->slug) }}" class="btn-primary" target="_blank">
                        🌐 View Public
                    </a>
                @endif
            </div>
        </div>

        <div class="post-grid">
            <!-- Post Details -->
            <div class="main-content">
                <div class="post-section">
                    <div class="post-header">
                        <div class="post-meta">
                            <span class="post-type type-{{ $post->type }}">
                                @switch($post->type)
                                    @case('blog')
                                        📖 Blog Post
                                        @break
                                    @case('news')
                                        📰 News
                                        @break
                                    @case('highlight')
                                        🏆 Tournament Highlight
                                        @break
                                    @case('tournament')
                                        🎮 Tournament
                                        @break
                                    @case('update')
                                        🔄 Platform Update
                                        @break
                                @endswitch
                            </span>
                            <span class="post-status status-{{ $post->status }}">
                                {{ ucfirst($post->status) }}
                            </span>
                        </div>

                        <h1 class="post-title">{{ $post->title }}</h1>

                        @if($post->excerpt)
                            <p class="post-excerpt">{{ $post->excerpt }}</p>
                        @endif
                    </div>

                    @if($post->featured_image)
                        <div class="featured-image">
                            <img src="{{ asset('storage/' . $post->featured_image) }}" alt="{{ $post->title }}">
                        </div>
                    @endif

                    <div class="post-content">
                        {!! nl2br(e($post->content)) !!}
                    </div>

                    @if($post->metadata && count($post->metadata) > 0)
                        <div class="post-metadata">
                            <h3>📋 Additional Information</h3>
                            <div class="metadata-grid">
                                @foreach($post->metadata as $key => $value)
                                    @if($value)
                                        <div class="metadata-item">
                                            <span class="metadata-label">{{ ucwords(str_replace('_', ' ', $key)) }}:</span>
                                            <span class="metadata-value">{{ $value }}</span>
                                        </div>
                                    @endif
                                @endforeach
                            </div>
                        </div>
                    @endif
                </div>
            </div>

            <!-- Sidebar -->
            <div class="sidebar">
                <div class="stats-section">
                    <h3>📊 Post Statistics</h3>
                    <div class="stats-grid">
                        <div class="stat-item">
                            <span class="stat-label">Views</span>
                            <span class="stat-value">{{ number_format($post->views) }}</span>
                        </div>
                        <div class="stat-item">
                            <span class="stat-label">Author</span>
                            <span class="stat-value">{{ $post->author->username }}</span>
                        </div>
                        <div class="stat-item">
                            <span class="stat-label">Created</span>
                            <span class="stat-value">{{ $post->created_at->format('M j, Y') }}</span>
                        </div>
                        @if($post->published_at)
                            <div class="stat-item">
                                <span class="stat-label">Published</span>
                                <span class="stat-value">{{ $post->published_at->format('M j, Y') }}</span>
                            </div>
                        @endif
                        <div class="stat-item">
                            <span class="stat-label">Reading Time</span>
                            <span class="stat-value">{{ $post->reading_time }} min</span>
                        </div>
                        <div class="stat-item">
                            <span class="stat-label">Word Count</span>
                            <span class="stat-value">{{ str_word_count(strip_tags($post->content)) }}</span>
                        </div>
                    </div>
                </div>

                <div class="actions-section">
                    <h3>⚡ Quick Actions</h3>
                    <div class="action-buttons">
                        <a href="{{ route('admin.posts.edit', $post) }}" class="action-btn btn-warning">
                            ✏️ Edit Post
                        </a>

                        @if($post->status === 'published')
                            <a href="{{ route('posts.show', $post->slug) }}" class="action-btn btn-info" target="_blank">
                                🌐 View Public
                            </a>
                        @endif

                        @if($post->status === 'draft')
                            <form method="POST" action="{{ route('admin.posts.update', $post) }}" style="display: inline;">
                                @csrf
                                @method('PUT')
                                <input type="hidden" name="title" value="{{ $post->title }}">
                                <input type="hidden" name="content" value="{{ $post->content }}">
                                <input type="hidden" name="excerpt" value="{{ $post->excerpt }}">
                                <input type="hidden" name="type" value="{{ $post->type }}">
                                <input type="hidden" name="status" value="published">
                                <input type="hidden" name="published_at" value="{{ now() }}">
                                <button type="submit" class="action-btn btn-success">
                                    ✅ Publish Now
                                </button>
                            </form>
                        @endif

                        <form method="POST" action="{{ route('admin.posts.destroy', $post) }}" 
                              onsubmit="return confirm('Are you sure you want to delete this post? This action cannot be undone.')">
                            @csrf
                            @method('DELETE')
                            <button type="submit" class="action-btn btn-danger">
                                🗑️ Delete Post
                            </button>
                        </form>
                    </div>
                </div>

                <div class="seo-section">
                    <h3>🔍 SEO Information</h3>
                    <div class="seo-item">
                        <span class="seo-label">Slug:</span>
                        <span class="seo-value">{{ $post->slug }}</span>
                    </div>
                    @if($post->status === 'published')
                        <div class="seo-item">
                            <span class="seo-label">Public URL:</span>
                            <a href="{{ route('posts.show', $post->slug) }}" class="seo-link" target="_blank">
                                View Page →
                            </a>
                        </div>
                    @endif
                </div>
            </div>
        </div>
    </div>
</div>

<style>
.admin-view-post {
    padding: 2rem 0;
    background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
    min-height: 100vh;
    color: #e2e8f0;
}

.header-section {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 2rem;
    padding: 1.5rem;
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 102, 0, 0.2);
    border-radius: 12px;
}

.header-content h1 {
    color: #ff6600;
    margin: 0 0 0.5rem 0;
    font-size: 2rem;
}

.header-content p {
    color: #94a3b8;
    margin: 0;
}

.header-actions {
    display: flex;
    gap: 1rem;
    flex-wrap: wrap;
}

.btn-secondary, .btn-warning, .btn-primary {
    padding: 0.75rem 1.5rem;
    border-radius: 8px;
    text-decoration: none;
    font-weight: 600;
    transition: all 0.3s ease;
    border: 1px solid;
}

.btn-secondary {
    background: rgba(255, 102, 0, 0.2);
    color: #ff6600;
    border-color: #ff6600;
}

.btn-secondary:hover {
    background: #ff6600;
    color: #1a1a2e;
}

.btn-warning {
    background: rgba(245, 158, 11, 0.2);
    color: #f59e0b;
    border-color: #f59e0b;
}

.btn-warning:hover {
    background: #f59e0b;
    color: #1a1a2e;
}

.btn-primary {
    background: linear-gradient(135deg, #ff6600, #cc5500);
    color: white;
    border-color: #ff6600;
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(255, 102, 0, 0.3);
}

.post-grid {
    display: grid;
    grid-template-columns: 2fr 1fr;
    gap: 2rem;
}

.post-section {
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 255, 255, 0.1);
    border-radius: 12px;
    padding: 2rem;
}

.post-header {
    margin-bottom: 2rem;
}

.post-meta {
    display: flex;
    gap: 1rem;
    margin-bottom: 1.5rem;
    flex-wrap: wrap;
}

.post-type,
.post-status {
    padding: 0.5rem 1rem;
    border-radius: 20px;
    font-size: 0.9rem;
    font-weight: 600;
}

.type-blog { background: rgba(59, 130, 246, 0.2); color: #3b82f6; }
.type-news { background: rgba(16, 185, 129, 0.2); color: #10b981; }
.type-highlight { background: rgba(245, 158, 11, 0.2); color: #f59e0b; }
.type-tournament { background: rgba(139, 92, 246, 0.2); color: #8b5cf6; }
.type-update { background: rgba(236, 72, 153, 0.2); color: #ec4899; }

.status-draft { background: rgba(107, 114, 128, 0.2); color: #6b7280; }
.status-published { background: rgba(34, 197, 94, 0.2); color: #22c55e; }
.status-archived { background: rgba(239, 68, 68, 0.2); color: #ef4444; }

.post-title {
    color: #e2e8f0;
    font-size: 2rem;
    font-weight: 700;
    margin: 0 0 1rem 0;
    line-height: 1.3;
}

.post-excerpt {
    color: #94a3b8;
    font-size: 1.1rem;
    line-height: 1.6;
    font-style: italic;
    margin: 0;
}

.featured-image {
    margin: 2rem 0;
    border-radius: 12px;
    overflow: hidden;
}

.featured-image img {
    width: 100%;
    height: auto;
    display: block;
}

.post-content {
    font-size: 1.1rem;
    line-height: 1.8;
    color: #e2e8f0;
    margin-bottom: 2rem;
}

.post-metadata {
    margin-top: 2rem;
    padding: 1.5rem;
    background: rgba(255, 102, 0, 0.05);
    border: 1px solid rgba(255, 102, 0, 0.2);
    border-radius: 12px;
}

.post-metadata h3 {
    color: #ff6600;
    margin-bottom: 1rem;
    font-size: 1.2rem;
}

.metadata-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1rem;
}

.metadata-item {
    display: flex;
    justify-content: space-between;
    padding: 0.75rem;
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 255, 255, 0.1);
    border-radius: 8px;
}

.metadata-label {
    color: #94a3b8;
    font-weight: 500;
}

.metadata-value {
    color: #e2e8f0;
    font-weight: 600;
}

.sidebar > div {
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 255, 255, 0.1);
    border-radius: 12px;
    padding: 1.5rem;
    margin-bottom: 2rem;
}

.sidebar h3 {
    color: #ff6600;
    margin: 0 0 1.5rem 0;
    font-size: 1.2rem;
}

.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
    gap: 1rem;
}

.stat-item {
    text-align: center;
    padding: 1rem;
    background: rgba(255, 102, 0, 0.1);
    border: 1px solid rgba(255, 102, 0, 0.2);
    border-radius: 8px;
}

.stat-label {
    display: block;
    color: #94a3b8;
    font-size: 0.875rem;
    margin-bottom: 0.5rem;
}

.stat-value {
    display: block;
    color: #ff6600;
    font-weight: 600;
    font-size: 1.1rem;
}

.action-buttons {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.action-btn {
    padding: 0.75rem 1rem;
    border-radius: 8px;
    text-decoration: none;
    font-weight: 600;
    text-align: center;
    transition: all 0.3s ease;
    border: 1px solid;
    cursor: pointer;
    display: inline-flex;
    align-items: center;
    justify-content: center;
    gap: 0.5rem;
}

.btn-info {
    background: rgba(59, 130, 246, 0.2);
    color: #3b82f6;
    border-color: #3b82f6;
}

.btn-info:hover {
    background: #3b82f6;
    color: white;
}

.btn-success {
    background: rgba(34, 197, 94, 0.2);
    color: #22c55e;
    border-color: #22c55e;
}

.btn-success:hover {
    background: #22c55e;
    color: white;
}

.btn-danger {
    background: rgba(239, 68, 68, 0.2);
    color: #ef4444;
    border-color: #ef4444;
}

.btn-danger:hover {
    background: #ef4444;
    color: white;
}

.seo-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1rem;
    padding-bottom: 1rem;
    border-bottom: 1px solid rgba(255, 255, 255, 0.1);
}

.seo-item:last-child {
    margin-bottom: 0;
    padding-bottom: 0;
    border-bottom: none;
}

.seo-label {
    color: #94a3b8;
    font-weight: 500;
}

.seo-value {
    color: #e2e8f0;
    font-family: monospace;
    font-size: 0.9rem;
}

.seo-link {
    color: #ff6600;
    text-decoration: none;
    font-weight: 600;
    transition: color 0.3s ease;
}

.seo-link:hover {
    color: #cc5500;
}

@media (max-width: 1024px) {
    .post-grid {
        grid-template-columns: 1fr;
    }
    
    .header-section {
        flex-direction: column;
        gap: 1rem;
        text-align: center;
    }
    
    .header-actions {
        justify-content: center;
    }
}

@media (max-width: 768px) {
    .post-meta {
        flex-direction: column;
        align-items: flex-start;
    }
    
    .stats-grid {
        grid-template-columns: repeat(auto-fit, minmax(100px, 1fr));
    }
    
    .metadata-grid {
        grid-template-columns: 1fr;
    }
}
</style>
@endsection