@extends('layouts.app')

@section('title', 'Manage Users')

@section('content')
<div class="admin-users">
    <div class="container">
        <div class="page-header">
            <h1>👥 User Management</h1>
            <p>Manage user accounts and permissions</p>
        </div>

        <!-- Search and Filters -->
        <div class="filters-section">
            <form method="GET" action="{{ route('admin.users.index') }}" class="filters-form">
                <div class="filter-group">
                    <input type="text" name="search" value="{{ request('search') }}" placeholder="Search by username or email..." class="search-input">
                </div>
                <div class="filter-group">
                    <select name="status" class="status-filter">
                        <option value="">All Status</option>
                        <option value="active" {{ request('status') === 'active' ? 'selected' : '' }}>Active</option>
                        <option value="banned" {{ request('status') === 'banned' ? 'selected' : '' }}>Banned</option>
                    </select>
                </div>
                <button type="submit" class="btn-primary">Filter</button>
                @if(request()->hasAny(['search', 'status']))
                    <a href="{{ route('admin.users.index') }}" class="btn-secondary">Clear</a>
                @endif
            </form>
        </div>

        <!-- Users Table -->
        <div class="users-table-container">
            <div class="table-responsive">
                <table class="users-table">
                    <thead>
                        <tr>
                            <th>User</th>
                            <th>Level</th>
                            <th>Status</th>
                            <th>Wallet</th>
                            <th>XP</th>
                            <th>Joined</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        @forelse($users as $user)
                            <tr class="user-row">
                                <td class="user-cell">
                                    <div class="user-info">
                                        <div class="user-avatar">
                                            {{ strtoupper(substr($user->username, 0, 1)) }}
                                        </div>
                                        <div class="user-details">
                                            <h4>{{ $user->username }}</h4>
                                            <p>{{ $user->email }}</p>
                                            @if($user->is_organizer)
                                                <span class="organizer-badge">🛡️ Admin</span>
                                            @endif
                                        </div>
                                    </div>
                                </td>
                                <td>
                                    <span class="level-badge">Level {{ $user->level }}</span>
                                </td>
                                <td>
                                    <span class="status-badge {{ $user->status }}">
                                        {{ ucfirst($user->status) }}
                                    </span>
                                </td>
                                <td>
                                    <div class="wallet-info">
                                        <strong>₦{{ number_format($user->wallet_balance, 2) }}</strong>
                                        @if($user->locked_balance > 0)
                                            <small>₦{{ number_format($user->locked_balance, 2) }} locked</small>
                                        @endif
                                    </div>
                                </td>
                                <td>
                                    <span class="xp-badge">{{ number_format($user->xp) }} XP</span>
                                </td>
                                <td>
                                    <span class="date-info">{{ $user->created_at->format('M d, Y') }}</span>
                                    <small>{{ $user->created_at->diffForHumans() }}</small>
                                </td>
                                <td>
                                    <div class="action-buttons">
                                        <a href="{{ route('admin.users.show', $user) }}" class="btn-view">View</a>
                                        @if($user->status === 'active')
                                            <form method="POST" action="{{ route('admin.users.ban', $user) }}" style="display: inline;">
                                                @csrf
                                                <button type="submit" class="btn-danger" onclick="return confirm('Are you sure you want to ban this user?')">Ban</button>
                                            </form>
                                        @else
                                            <form method="POST" action="{{ route('admin.users.unban', $user) }}" style="display: inline;">
                                                @csrf
                                                <button type="submit" class="btn-success">Unban</button>
                                            </form>
                                        @endif
                                        
                                        <form method="POST" action="{{ route('admin.users.toggle-organizer', $user) }}" style="display: inline;">
                                            @csrf
                                            <button type="submit" class="btn-secondary">
                                                {{ $user->is_organizer ? 'Remove Admin' : 'Make Admin' }}
                                            </button>
                                        </form>
                                    </div>
                                </td>
                            </tr>
                        @empty
                            <tr>
                                <td colspan="7" class="empty-state">
                                    <div class="empty-content">
                                        <p>No users found matching your criteria.</p>
                                    </div>
                                </td>
                            </tr>
                        @endforelse
                    </tbody>
                </table>
            </div>
        </div>

        <!-- Pagination -->
        @if($users->hasPages())
            <div class="pagination-wrapper">
                {{ $users->appends(request()->query())->links() }}
            </div>
        @endif
    </div>
</div>

<style>
.admin-users {
    padding: 2rem 0;
    background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
    min-height: 100vh;
}

.page-header {
    text-align: center;
    margin-bottom: 3rem;
}

.page-header h1 {
    font-size: 2.5rem;
    color: #39ff14;
    margin-bottom: 0.5rem;
}

.page-header p {
    color: #94a3b8;
    font-size: 1.1rem;
}

.filters-section {
    margin-bottom: 2rem;
}

.filters-form {
    display: flex;
    gap: 1rem;
    align-items: center;
    flex-wrap: wrap;
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 102, 0, 0.2);
    border-radius: 12px;
    padding: 1.5rem;
}

.filter-group {
    flex: 1;
    min-width: 200px;
}

.search-input, .status-filter {
    width: 100%;
    padding: 0.75rem;
    background: rgba(255, 255, 255, 0.1);
    border: 1px solid rgba(255, 102, 0, 0.3);
    border-radius: 8px;
    color: #e2e8f0;
    font-size: 1rem;
}

.search-input:focus, .status-filter:focus {
    outline: none;
    border-color: #ff6600;
    box-shadow: 0 0 0 2px rgba(255, 102, 0, 0.2);
}

.search-input::placeholder {
    color: #94a3b8;
}

.btn-primary, .btn-secondary {
    padding: 0.75rem 1.5rem;
    border-radius: 8px;
    text-decoration: none;
    font-weight: 600;
    transition: all 0.3s ease;
    border: none;
    cursor: pointer;
    font-size: 1rem;
}

.btn-primary {
    background: #ff6600;
    color: #1a1a2e;
}

.btn-primary:hover {
    background: #e55a00;
    transform: translateY(-2px);
}

.btn-secondary {
    background: rgba(255, 102, 0, 0.2);
    color: #ff6600;
    border: 1px solid #ff6600;
}

.btn-secondary:hover {
    background: #ff6600;
    color: #1a1a2e;
}

.users-table-container {
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 102, 0, 0.2);
    border-radius: 12px;
    overflow: hidden;
    margin-bottom: 2rem;
}

.table-responsive {
    overflow-x: auto;
}

.users-table {
    width: 100%;
    border-collapse: collapse;
}

.users-table th {
    background: rgba(255, 102, 0, 0.1);
    color: #ff6600;
    padding: 1rem;
    text-align: left;
    font-weight: 600;
    border-bottom: 1px solid rgba(255, 102, 0, 0.2);
}

.users-table td {
    padding: 1rem;
    border-bottom: 1px solid rgba(255, 255, 255, 0.1);
    color: #e2e8f0;
}

.user-row:hover {
    background: rgba(255, 102, 0, 0.05);
}

.user-cell {
    min-width: 250px;
}

.user-info {
    display: flex;
    align-items: center;
    gap: 1rem;
}

.user-avatar {
    width: 50px;
    height: 50px;
    background: linear-gradient(135deg, #ff6600, #e55a00);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    color: #1a1a2e;
    font-weight: bold;
    font-size: 1.2rem;
}

.user-details h4 {
    color: #e2e8f0;
    margin: 0 0 0.25rem 0;
    font-size: 1rem;
}

.user-details p {
    color: #94a3b8;
    margin: 0;
    font-size: 0.9rem;
}

.organizer-badge {
    background: rgba(255, 215, 0, 0.2);
    color: #ffd700;
    border: 1px solid #ffd700;
    padding: 0.2rem 0.5rem;
    border-radius: 12px;
    font-size: 0.8rem;
    font-weight: 600;
    margin-top: 0.25rem;
    display: inline-block;
}

.level-badge {
    background: rgba(59, 130, 246, 0.2);
    color: #3b82f6;
    border: 1px solid #3b82f6;
    padding: 0.25rem 0.75rem;
    border-radius: 20px;
    font-size: 0.8rem;
    font-weight: 600;
}

.status-badge {
    padding: 0.25rem 0.75rem;
    border-radius: 20px;
    font-size: 0.8rem;
    font-weight: 600;
    text-transform: uppercase;
}

.status-badge.active {
    background: rgba(34, 197, 94, 0.2);
    color: #22c55e;
    border: 1px solid #22c55e;
}

.status-badge.banned {
    background: rgba(239, 68, 68, 0.2);
    color: #ef4444;
    border: 1px solid #ef4444;
}

.wallet-info strong {
    color: #22c55e;
    display: block;
}

.wallet-info small {
    color: #94a3b8;
    font-size: 0.8rem;
}

.xp-badge {
    background: rgba(168, 85, 247, 0.2);
    color: #a855f7;
    border: 1px solid #a855f7;
    padding: 0.25rem 0.75rem;
    border-radius: 20px;
    font-size: 0.8rem;
    font-weight: 600;
}

.date-info {
    color: #e2e8f0;
    display: block;
    margin-bottom: 0.25rem;
}

.date-info + small {
    color: #94a3b8;
    font-size: 0.8rem;
}

.action-buttons {
    display: flex;
    gap: 0.5rem;
    flex-wrap: wrap;
}

.btn-view, .btn-danger, .btn-success {
    padding: 0.4rem 0.8rem;
    border-radius: 6px;
    text-decoration: none;
    font-size: 0.8rem;
    font-weight: 600;
    border: none;
    cursor: pointer;
    transition: all 0.3s ease;
}

.btn-view {
    background: rgba(59, 130, 246, 0.2);
    color: #3b82f6;
    border: 1px solid #3b82f6;
}

.btn-view:hover {
    background: #3b82f6;
    color: #1a1a2e;
}

.btn-danger {
    background: rgba(239, 68, 68, 0.2);
    color: #ef4444;
    border: 1px solid #ef4444;
}

.btn-danger:hover {
    background: #ef4444;
    color: #1a1a2e;
}

.btn-success {
    background: rgba(34, 197, 94, 0.2);
    color: #22c55e;
    border: 1px solid #22c55e;
}

.btn-success:hover {
    background: #22c55e;
    color: #1a1a2e;
}

.empty-state {
    text-align: center;
    padding: 3rem;
}

.empty-content p {
    color: #94a3b8;
    font-size: 1.1rem;
}

.pagination-wrapper {
    display: flex;
    justify-content: center;
}

@media (max-width: 768px) {
    .filters-form {
        flex-direction: column;
        align-items: stretch;
    }
    
    .filter-group {
        min-width: auto;
    }
    
    .action-buttons {
        flex-direction: column;
    }
    
    .users-table {
        font-size: 0.9rem;
    }
    
    .users-table th,
    .users-table td {
        padding: 0.75rem 0.5rem;
    }
}
</style>
@endsection