@extends('layouts.admin')

@section('title', 'User Details')
@section('page-title', 'User Details')
@section('page-subtitle', 'View and manage user account information')

@section('content')
<div class="p-6">
    <div class="max-w-7xl mx-auto">
        <!-- Header -->
        <div class="mb-6">
            <a href="{{ route('admin.users.index') }}" class="inline-flex items-center px-4 py-2 bg-cod-dark hover:bg-cod-light text-gray-100 rounded-lg transition-all duration-200 border border-cod-light">
                ← Back to Users
            </a>
        </div>

        <!-- User Profile Header -->
        <div class="bg-cod-gray/95 backdrop-blur-lg rounded-2xl p-6 border border-cod-light gaming-glow mb-8">
            <div class="flex items-center space-x-6">
                <div class="w-20 h-20 bg-gradient-to-br from-yellow-400 to-orange-500 rounded-full flex items-center justify-center text-black font-bold text-2xl">
                    {{ strtoupper(substr($user->username, 0, 1)) }}
                </div>
                <div class="flex-1">
                    <h1 class="text-3xl font-bold text-white mb-2">{{ $user->username }}</h1>
                    <p class="text-gray-300 mb-3">{{ $user->email }}</p>
                    <div class="flex flex-wrap gap-2">
                        <span class="px-3 py-1 bg-blue-500/20 text-blue-400 rounded-full text-sm">
                            {{ $levels->find($user->level)?->name ?? "Level {$user->level}" }}
                        </span>
                        <span class="px-3 py-1 {{ $user->status === 'active' ? 'bg-green-500/20 text-green-400' : 'bg-red-500/20 text-red-400' }} rounded-full text-sm">
                            {{ ucfirst($user->status) }}
                        </span>
                        @if($user->is_organizer)
                            <span class="px-3 py-1 bg-red-500/20 text-red-400 rounded-full text-sm">🛡️ Admin</span>
                        @endif
                        @if($user->role !== 'basic_user')
                            <span class="px-3 py-1 bg-purple-500/20 text-purple-400 rounded-full text-sm">{{ $user->role->getDisplayName() }}</span>
                        @endif
                    </div>
                </div>
                <div class="text-right">
                    <div class="text-2xl font-bold text-yellow-400">₦{{ number_format($user->wallet_balance, 2) }}</div>
                    <div class="text-sm text-gray-300">Wallet Balance</div>
                    @if($user->locked_balance > 0)
                        <div class="text-yellow-400 text-sm">₦{{ number_format($user->locked_balance, 2) }} locked</div>
                    @endif
                </div>
            </div>
        </div>

        <!-- Stats Grid -->
        <div class="grid grid-cols-1 md:grid-cols-4 gap-6 mb-8">
            <div class="bg-cod-gray/95 backdrop-blur-lg rounded-xl p-6 border border-cod-light gaming-glow text-center">
                <div class="text-3xl mb-2 text-cod-green">{{ number_format($user->xp) }}</div>
                <div class="text-gray-300">Experience Points</div>
            </div>
            <div class="bg-cod-gray/95 backdrop-blur-lg rounded-xl p-6 border border-cod-light gaming-glow text-center">
                <div class="text-3xl mb-2 text-cod-green">{{ $user->games_played ?? 0 }}</div>
                <div class="text-gray-300">Games Played</div>
            </div>
            <div class="bg-cod-gray/95 backdrop-blur-lg rounded-xl p-6 border border-cod-light gaming-glow text-center">
                <div class="text-3xl mb-2 text-cod-green">{{ $user->createdChallenges->count() + $user->acceptedChallenges->count() }}</div>
                <div class="text-gray-300">Total Challenges</div>
            </div>
            <div class="bg-cod-gray/95 backdrop-blur-lg rounded-xl p-6 border border-cod-light gaming-glow text-center">
                <div class="text-3xl mb-2 text-cod-green">{{ $user->walletTransactions->count() }}</div>
                <div class="text-gray-300">Transactions</div>
            </div>
        </div>

        <!-- Admin Actions Grid -->
        <div class="grid grid-cols-1 lg:grid-cols-2 gap-8 mb-8">
            <!-- Level Assignment -->
            <div class="bg-cod-gray/95 backdrop-blur-lg rounded-xl p-6 border border-cod-light gaming-glow">
                <h3 class="text-xl font-bold text-cod-green mb-4">🎯 Level Assignment</h3>
                <p class="text-sm text-gray-300 mb-4">Select a level to assign to this user. Higher levels unlock more features and higher wager limits.</p>
                <form method="POST" action="{{ route('admin.users.assign-level', $user) }}" class="space-y-4">
                    @csrf
                    <div>
                        <label class="block text-sm font-medium text-cod-green mb-2">Assign Level</label>
                        <select name="level_id" class="w-full px-4 py-2 bg-cod-dark border border-cod-light rounded-lg text-gray-100 focus:ring-2 focus:ring-cod-green focus:border-cod-green transition-all duration-200 input-gaming">
                            @foreach($processedLevels as $level)
                                <option value="{{ $level->id }}" {{ $user->level == $level->id ? 'selected' : '' }} class="bg-cod-dark text-gray-100">
                                    {{ $level->display_name_with_limits }}
                                </option>
                            @endforeach
                        </select>
                    </div>
                    <button type="submit" class="w-full px-4 py-2 bg-cod-green hover:bg-cod-green-dark text-black font-bold rounded-lg transition-all duration-200 btn-gaming">
                        🎯 Assign Level
                    </button>
                </form>

                <div class="mt-4 p-4 bg-cod-green/10 border border-cod-green/30 rounded-lg">
                    <div class="text-sm text-cod-green">
                        <strong>Current Level Info:</strong><br>
                        {{ $levels->find($user->level)?->name ?? "Level {$user->level}" }}<br>
                        @php $currentLevel = $levels->find($user->level); @endphp
                        @if($currentLevel)
                            Min XP: {{ number_format($currentLevel->min_xp) }}<br>
                            Min Games: {{ $currentLevel->min_games_played }}<br>
                            @if($currentLevel->max_wager_amount)
                                Max Wager: ₦{{ number_format($currentLevel->max_wager_amount) }}
                            @else
                                Max Wager: Unlimited
                            @endif
                        @endif
                    </div>
                </div>
            </div>

            <!-- Wallet Management -->
            <div class="bg-cod-gray/95 backdrop-blur-lg rounded-xl p-6 border border-cod-light gaming-glow">
                <h3 class="text-xl font-bold text-cod-green mb-4">💰 Wallet Management</h3>
                
                <!-- Credit Wallet -->
                <form method="POST" action="{{ route('admin.users.credit-wallet', $user) }}" class="space-y-4 mb-6">
                    @csrf
                    <div>
                        <label class="block text-sm font-medium text-cod-green mb-2">Credit Amount (₦)</label>
                        <input type="number" name="amount" min="1" max="1000000" step="0.01" 
                               class="w-full px-4 py-2 bg-cod-dark border border-cod-light rounded-lg text-gray-100 focus:ring-2 focus:ring-cod-green focus:border-cod-green transition-all duration-200 input-gaming" 
                               placeholder="Enter amount to credit">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-cod-green mb-2">Description</label>
                        <input type="text" name="description" maxlength="255" 
                               class="w-full px-4 py-2 bg-cod-dark border border-cod-light rounded-lg text-gray-100 focus:ring-2 focus:ring-cod-green focus:border-cod-green transition-all duration-200 input-gaming" 
                               placeholder="Reason for credit">
                    </div>
                    <button type="submit" class="w-full px-4 py-2 bg-green-600 hover:bg-green-700 text-white font-bold rounded-lg transition-all duration-200">
                        💰 Credit Wallet
                    </button>
                </form>

                <!-- Debit Wallet -->
                <form method="POST" action="{{ route('admin.users.debit-wallet', $user) }}" class="space-y-4">
                    @csrf
                    <div>
                        <label class="block text-sm font-medium text-cod-green mb-2">Debit Amount (₦)</label>
                        <input type="number" name="amount" min="1" max="{{ $user->wallet_balance }}" step="0.01" 
                               class="w-full px-4 py-2 bg-cod-dark border border-cod-light rounded-lg text-gray-100 focus:ring-2 focus:ring-red-400 focus:border-red-400 transition-all duration-200 input-gaming" 
                               placeholder="Enter amount to debit">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-cod-green mb-2">Description</label>
                        <input type="text" name="description" maxlength="255" 
                               class="w-full px-4 py-2 bg-cod-dark border border-cod-light rounded-lg text-gray-100 focus:ring-2 focus:ring-red-400 focus:border-red-400 transition-all duration-200 input-gaming" 
                               placeholder="Reason for debit">
                    </div>
                    <button type="submit" class="w-full px-4 py-2 bg-red-600 hover:bg-red-700 text-white font-bold rounded-lg transition-all duration-200"
                            onclick="return confirm('Are you sure you want to debit this amount?')">
                        💸 Debit Wallet
                    </button>
                </form>
            </div>
        </div>

        <!-- Additional Admin Actions -->
        <div class="grid grid-cols-1 lg:grid-cols-2 gap-8 mb-8">
            <!-- Role & Status Management -->
            <div class="bg-cod-gray/95 backdrop-blur-lg rounded-xl p-6 border border-cod-light gaming-glow">
                <h3 class="text-xl font-bold text-cod-green mb-4">👑 Role & Status Management</h3>
                
                <!-- Role Update -->
                <form method="POST" action="{{ route('admin.users.update-role', $user) }}" class="space-y-4 mb-6">
                    @csrf
                    <div>
                        <label class="block text-sm font-medium text-cod-green mb-2">User Role</label>
                        <select name="role" class="w-full px-4 py-2 bg-cod-dark border border-cod-light rounded-lg text-gray-100 focus:ring-2 focus:ring-cod-green focus:border-cod-green transition-all duration-200 input-gaming">
                            @foreach($availableRoles as $roleValue => $roleDisplay)
                                <option value="{{ $roleValue }}" {{ $user->role->value === $roleValue ? 'selected' : '' }} class="bg-cod-dark text-gray-100">{{ $roleDisplay }}</option>
                            @endforeach
                        </select>
                        <p class="text-xs text-gray-400 mt-1">{{ $user->role->getDescription() }}</p>
                    </div>
                    <button type="submit" class="w-full px-4 py-2 bg-purple-600 hover:bg-purple-700 text-white font-bold rounded-lg transition-all duration-200">
                        👑 Update Role
                    </button>
                </form>

                <!-- KYC Status -->
                <form method="POST" action="{{ route('admin.users.update-kyc-status', $user) }}" class="space-y-4">
                    @csrf
                    <div>
                        <label class="block text-sm font-medium text-cod-green mb-2">KYC Status</label>
                        <select name="kyc_status" class="w-full px-4 py-2 bg-cod-dark border border-cod-light rounded-lg text-gray-100 focus:ring-2 focus:ring-cod-green focus:border-cod-green transition-all duration-200 input-gaming">
                            <option value="none" {{ $user->kyc_status === 'none' ? 'selected' : '' }} class="bg-cod-dark text-gray-100">None</option>
                            <option value="pending" {{ $user->kyc_status === 'pending' ? 'selected' : '' }} class="bg-cod-dark text-gray-100">Pending</option>
                            <option value="verified" {{ $user->kyc_status === 'verified' ? 'selected' : '' }} class="bg-cod-dark text-gray-100">Verified</option>
                            <option value="rejected" {{ $user->kyc_status === 'rejected' ? 'selected' : '' }} class="bg-cod-dark text-gray-100">Rejected</option>
                        </select>
                    </div>
                    <button type="submit" class="w-full px-4 py-2 bg-indigo-600 hover:bg-indigo-700 text-white font-bold rounded-lg transition-all duration-200">
                        🆔 Update KYC Status
                    </button>
                </form>
            </div>

            <!-- Quick Actions -->
            <div class="bg-cod-gray/95 backdrop-blur-lg rounded-xl p-6 border border-cod-light gaming-glow">
                <h3 class="text-xl font-bold text-cod-green mb-4">⚡ Quick Actions</h3>
                <div class="space-y-3">
                    @if($user->status === 'active')
                        <form method="POST" action="{{ route('admin.users.ban', $user) }}" class="w-full">
                            @csrf
                            <button type="submit" class="w-full px-4 py-2 bg-red-600 hover:bg-red-700 text-white font-bold rounded-lg transition-all duration-200"
                                    onclick="return confirm('Are you sure you want to ban this user?')">
                                🚫 Ban User
                            </button>
                        </form>
                    @else
                        <form method="POST" action="{{ route('admin.users.unban', $user) }}" class="w-full">
                            @csrf
                            <button type="submit" class="w-full px-4 py-2 bg-green-600 hover:bg-green-700 text-white font-bold rounded-lg transition-all duration-200">
                                ✅ Unban User
                            </button>
                        </form>
                    @endif

                    <form method="POST" action="{{ route('admin.users.toggle-organizer', $user) }}" class="w-full">
                        @csrf
                        <button type="submit" class="w-full px-4 py-2 bg-purple-600 hover:bg-purple-700 text-white font-bold rounded-lg transition-all duration-200">
                            {{ $user->is_organizer ? '⬇️ Remove Admin' : '⬆️ Make Admin' }}
                        </button>
                    </form>
                </div>
            </div>
        </div>

        <!-- User Activity Section -->
        <div class="bg-cod-gray/95 backdrop-blur-lg rounded-xl p-6 border border-cod-light gaming-glow">
            <h3 class="text-xl font-bold text-cod-green mb-6">📊 User Activity Log</h3>
            
            <!-- Activity Tabs -->
            <div class="border-b border-cod-light mb-6">
                <nav class="flex space-x-8">
                    <button onclick="showTab('recent')" id="recent-tab" class="tab-button active py-2 px-1 border-b-2 font-medium text-sm">
                        Recent Activity
                    </button>
                    <button onclick="showTab('challenges')" id="challenges-tab" class="tab-button py-2 px-1 border-b-2 font-medium text-sm">
                        Challenges
                    </button>
                    <button onclick="showTab('transactions')" id="transactions-tab" class="tab-button py-2 px-1 border-b-2 font-medium text-sm">
                        Transactions
                    </button>
                </nav>
            </div>

            <!-- Recent Activity Tab -->
            <div id="recent-content" class="tab-content">
                <div class="space-y-4">
                    @forelse($recentActivity as $activity)
                        <div class="flex items-center justify-between p-4 bg-cod-dark/50 rounded-lg border border-cod-light/30">
                            <div class="flex items-center space-x-4">
                                <div class="w-10 h-10 bg-{{ $activity['type'] === 'challenge' ? 'blue' : 'green' }}-500/20 rounded-full flex items-center justify-center">
                                    {{ $activity['type'] === 'challenge' ? '⚔️' : '💰' }}
                                </div>
                                <div>
                                    <div class="text-gray-100 font-medium">{{ $activity['description'] }}</div>
                                    <div class="text-gray-400 text-sm">{{ $activity['created_at']->diffForHumans() }}</div>
                                </div>
                            </div>
                            <div class="text-right">
                                @if($activity['amount'])
                                    <div class="text-{{ $activity['amount'] > 0 ? 'green' : 'red' }}-400 font-medium">
                                        {{ $activity['amount'] > 0 ? '+' : '' }}₦{{ number_format(abs($activity['amount']), 2) }}
                                    </div>
                                @endif
                                <div class="text-gray-400 text-sm">{{ $activity['action'] }}</div>
                            </div>
                        </div>
                    @empty
                        <div class="text-center py-8 text-gray-400">
                            <div class="text-4xl mb-4">📊</div>
                            <p>No recent activity found</p>
                        </div>
                    @endforelse
                </div>
            </div>

            <!-- Challenges Tab -->
            <div id="challenges-content" class="tab-content hidden">
                <div class="space-y-4">
                    @forelse($user->createdChallenges->concat($user->acceptedChallenges)->sortByDesc('created_at')->take(10) as $challenge)
                        <div class="flex items-center justify-between p-4 bg-cod-dark/50 rounded-lg border border-cod-light/30">
                            <div>
                                <div class="text-gray-100 font-medium">Challenge #{{ $challenge->id }}</div>
                                <div class="text-gray-400 text-sm">
                                    {{ $challenge->creator_id === $user->id ? 'Created' : 'Accepted' }} 
                                    {{ $challenge->created_at->diffForHumans() }}
                                </div>
                            </div>
                            <div class="text-right">
                                <div class="text-cod-green font-medium">₦{{ number_format($challenge->wager_amount, 2) }}</div>
                                <div class="text-gray-400 text-sm">{{ ucfirst($challenge->status) }}</div>
                            </div>
                        </div>
                    @empty
                        <div class="text-center py-8 text-gray-400">
                            <div class="text-4xl mb-4">⚔️</div>
                            <p>No challenges found</p>
                        </div>
                    @endforelse
                </div>
            </div>

            <!-- Transactions Tab -->
            <div id="transactions-content" class="tab-content hidden">
                <div class="space-y-4">
                    @forelse($user->walletTransactions->sortByDesc('created_at')->take(15) as $transaction)
                        <div class="flex items-center justify-between p-4 bg-cod-dark/50 rounded-lg border border-cod-light/30">
                            <div>
                                <div class="text-gray-100 font-medium">{{ $transaction->description }}</div>
                                <div class="text-gray-400 text-sm">{{ $transaction->created_at->diffForHumans() }}</div>
                            </div>
                            <div class="text-right">
                                <div class="text-{{ $transaction->amount > 0 ? 'green' : 'red' }}-400 font-medium">
                                    {{ $transaction->amount > 0 ? '+' : '' }}₦{{ number_format(abs($transaction->amount), 2) }}
                                </div>
                                <div class="text-gray-400 text-sm">{{ ucfirst($transaction->type) }}</div>
                            </div>
                        </div>
                    @empty
                        <div class="text-center py-8 text-gray-400">
                            <div class="text-4xl mb-4">💰</div>
                            <p>No transactions found</p>
                        </div>
                    @endforelse
                </div>
            </div>
        </div>
    </div>
</div>

<script>
function showTab(tabName) {
    // Hide all tab contents
    document.querySelectorAll('.tab-content').forEach(content => {
        content.classList.add('hidden');
    });
    
    // Remove active class from all tab buttons
    document.querySelectorAll('.tab-button').forEach(button => {
        button.classList.remove('active');
    });
    
    // Show selected tab content
    document.getElementById(tabName + '-content').classList.remove('hidden');
    
    // Add active class to selected tab button
    document.getElementById(tabName + '-tab').classList.add('active');
}
</script>

<style>
.tab-button {
    @apply text-gray-400 border-transparent hover:text-cod-green hover:border-cod-green transition-all duration-200;
}

.tab-button.active {
    @apply text-cod-green border-cod-green;
}
</style>
@endsection