@extends('layouts.app')

@section('title', 'User Details')

@section('content')
<div class="admin-user-detail">
    <div class="container">
        <div class="page-header">
            <a href="{{ route('admin.users.index') }}" class="back-link">← Back to Users</a>
            <h1>👤 User Details</h1>
        </div>

        <div class="user-profile-grid">
            <!-- User Information -->
            <div class="profile-card">
                <div class="profile-header">
                    <div class="user-avatar-large">
                        {{ strtoupper(substr($user->username, 0, 1)) }}
                    </div>
                    <div class="user-main-info">
                        <h2>{{ $user->username }}</h2>
                        <p>{{ $user->email }}</p>
                        <div class="user-badges">
                            <span class="status-badge {{ $user->status }}">{{ ucfirst($user->status) }}</span>
                            <span class="level-badge">Level {{ $user->level }}</span>
                            @if($user->role)
                                <span class="role-badge role-{{ $user->role->value }}">{{ $user->role->getDisplayName() }} Role</span>
                            @endif
                            @if($user->is_organizer)
                                <span class="organizer-badge">🛡️ Admin</span>
                            @endif
                            @if($user->kyc_status === 'verified')
                                <span class="kyc-badge">✅ KYC Verified</span>
                            @elseif($user->kyc_status === 'pending')
                                <span class="kyc-badge kyc-pending">⏳ KYC Pending</span>
                            @elseif($user->kyc_status === 'rejected')
                                <span class="kyc-badge kyc-rejected">❌ KYC Rejected</span>
                            @endif
                        </div>
                    </div>
                </div>

                <div class="profile-stats">
                    <div class="stat-item">
                        <h4>{{ number_format($user->xp) }}</h4>
                        <p>Experience Points</p>
                    </div>
                    <div class="stat-item">
                        <h4>₦{{ number_format($user->wallet_balance, 2) }}</h4>
                        <p>Wallet Balance</p>
                    </div>
                    @if($user->locked_balance > 0)
                        <div class="stat-item">
                            <h4>₦{{ number_format($user->locked_balance, 2) }}</h4>
                            <p>Locked Balance</p>
                        </div>
                    @endif
                    <div class="stat-item">
                        <h4>{{ number_format($user->agreement_rate * 100, 1) }}%</h4>
                        <p>Agreement Rate</p>
                    </div>
                </div>

                <div class="profile-actions">
                    @if($user->status === 'active')
                        <form method="POST" action="{{ route('admin.users.ban', $user) }}" style="display: inline;">
                            @csrf
                            <button type="submit" class="btn-danger" onclick="return confirm('Are you sure you want to ban this user?')">
                                🚫 Ban User
                            </button>
                        </form>
                    @else
                        <form method="POST" action="{{ route('admin.users.unban', $user) }}" style="display: inline;">
                            @csrf
                            <button type="submit" class="btn-success">
                                ✅ Unban User
                            </button>
                        </form>
                    @endif

                    <form method="POST" action="{{ route('admin.users.toggle-organizer', $user) }}" style="display: inline;">
                        @csrf
                        <button type="submit" class="btn-secondary">
                            {{ $user->is_organizer ? '⬇️ Remove Admin' : '⬆️ Make Admin' }}
                        </button>
                    </form>
                </div>

                <!-- Role and KYC Management -->
                <div class="management-section">
                    <h4>👑 Role Management</h4>
                    <form method="POST" action="{{ route('admin.users.update-role', $user) }}" class="role-form">
                        @csrf
                        <div class="form-group">
                            <label for="role">User Role:</label>
                            <select name="role" id="role" class="form-select">
                                <option value="level1" {{ $user->role === 'level1' ? 'selected' : '' }}>Level 1 - Basic (₦20k challenges)</option>
                                <option value="level2" {{ $user->role === 'level2' ? 'selected' : '' }}>Level 2 - Advanced (₦200k challenges)</option>
                                <option value="highest" {{ $user->role === 'highest' ? 'selected' : '' }}>Highest - Premium (Unlimited + Tournaments)</option>
                                <option value="creator" {{ $user->role === 'creator' ? 'selected' : '' }}>Creator - Content (All permissions + Blog)</option>
                            </select>
                        </div>
                        <button type="submit" class="btn-primary">Update Role</button>
                    </form>

                    <h4>🆔 KYC Management</h4>
                    <form method="POST" action="{{ route('admin.users.update-kyc-status', $user) }}" class="kyc-form">
                        @csrf
                        <div class="form-group">
                            <label for="kyc_status">KYC Status:</label>
                            <select name="kyc_status" id="kyc_status" class="form-select">
                                <option value="unverified" {{ $user->kyc_status === 'unverified' ? 'selected' : '' }}>Unverified</option>
                                <option value="pending" {{ $user->kyc_status === 'pending' ? 'selected' : '' }}>Pending Review</option>
                                <option value="verified" {{ $user->kyc_status === 'verified' ? 'selected' : '' }}>Verified</option>
                                <option value="rejected" {{ $user->kyc_status === 'rejected' ? 'selected' : '' }}>Rejected</option>
                            </select>
                        </div>
                        <button type="submit" class="btn-primary">Update KYC Status</button>
                    </form>
                </div>
            </div>

            <!-- Activity Overview -->
            <div class="activity-card">
                <h3>📊 Activity Overview</h3>
                <div class="activity-stats">
                    <div class="activity-item">
                        <div class="activity-icon">⚔️</div>
                        <div class="activity-info">
                            <h4>{{ $user->createdChallenges->count() + $user->acceptedChallenges->count() }}</h4>
                            <p>Total Challenges</p>
                            <small>{{ $user->createdChallenges->count() }} created, {{ $user->acceptedChallenges->count() }} accepted</small>
                        </div>
                    </div>

                    <div class="activity-item">
                        <div class="activity-icon">🏆</div>
                        <div class="activity-info">
                            <h4>{{ $user->badges->count() }}</h4>
                            <p>Badges Earned</p>
                            <small>Achievements unlocked</small>
                        </div>
                    </div>

                    <div class="activity-item">
                        <div class="activity-icon">💰</div>
                        <div class="activity-info">
                            <h4>{{ $user->walletTransactions->count() }}</h4>
                            <p>Transactions</p>
                            <small>Wallet activity</small>
                        </div>
                    </div>

                    <div class="activity-item">
                        <div class="activity-icon">📅</div>
                        <div class="activity-info">
                            <h4>{{ $user->created_at->diffInDays() }}</h4>
                            <p>Days Active</p>
                            <small>Since {{ $user->created_at->format('M d, Y') }}</small>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Recent Activity Tabs -->
        <div class="activity-tabs">
            <div class="tab-navigation">
                <button class="tab-btn active" data-tab="challenges">Recent Challenges</button>
                <button class="tab-btn" data-tab="transactions">Wallet Transactions</button>
                <button class="tab-btn" data-tab="badges">Badges</button>
            </div>

            <!-- Challenges Tab -->
            <div id="challenges" class="tab-content active">
                <div class="content-header">
                    <h3>⚔️ Recent Challenges</h3>
                </div>
                @if($user->createdChallenges->count() > 0 || $user->acceptedChallenges->count() > 0)
                    <div class="challenges-list">
                        @foreach($user->createdChallenges->take(5) as $challenge)
                            <div class="challenge-item">
                                <div class="challenge-info">
                                    <h4>Challenge #{{ $challenge->id }}</h4>
                                    <p>vs {{ $challenge->accepter ? $challenge->accepter->username : 'Open' }}</p>
                                    <small>Created {{ $challenge->created_at->diffForHumans() }}</small>
                                </div>
                                <div class="challenge-meta">
                                    <span class="amount">₦{{ number_format($challenge->wager_amount, 2) }}</span>
                                    <span class="status-badge {{ $challenge->status }}">{{ ucfirst($challenge->status) }}</span>
                                </div>
                            </div>
                        @endforeach
                        @foreach($user->acceptedChallenges->take(5) as $challenge)
                            <div class="challenge-item">
                                <div class="challenge-info">
                                    <h4>Challenge #{{ $challenge->id }}</h4>
                                    <p>vs {{ $challenge->creator->username }} (Accepted)</p>
                                    <small>Accepted {{ $challenge->updated_at->diffForHumans() }}</small>
                                </div>
                                <div class="challenge-meta">
                                    <span class="amount">₦{{ number_format($challenge->wager_amount, 2) }}</span>
                                    <span class="status-badge {{ $challenge->status }}">{{ ucfirst($challenge->status) }}</span>
                                </div>
                            </div>
                        @endforeach
                    </div>
                @else
                    <div class="empty-state">
                        <p>No challenges found for this user.</p>
                    </div>
                @endif
            </div>

            <!-- Transactions Tab -->
            <div id="transactions" class="tab-content">
                <div class="content-header">
                    <h3>💰 Wallet Transactions</h3>
                </div>
                @if($user->walletTransactions->count() > 0)
                    <div class="transactions-list">
                        @foreach($user->walletTransactions->take(10) as $transaction)
                            <div class="transaction-item">
                                <div class="transaction-info">
                                    <h4>{{ ucfirst($transaction->type) }}</h4>
                                    <p>{{ $transaction->description }}</p>
                                    <small>{{ $transaction->created_at->format('M d, Y H:i') }}</small>
                                </div>
                                <div class="transaction-meta">
                                    <span class="amount {{ $transaction->amount > 0 ? 'positive' : 'negative' }}">
                                        {{ $transaction->amount > 0 ? '+' : '' }}₦{{ number_format($transaction->amount, 2) }}
                                    </span>
                                    <span class="status-badge {{ $transaction->status }}">{{ ucfirst($transaction->status) }}</span>
                                </div>
                            </div>
                        @endforeach
                    </div>
                @else
                    <div class="empty-state">
                        <p>No transactions found for this user.</p>
                    </div>
                @endif
            </div>

            <!-- Badges Tab -->
            <div id="badges" class="tab-content">
                <div class="content-header">
                    <h3>🏆 Earned Badges</h3>
                </div>
                @if($user->badges->count() > 0)
                    <div class="badges-grid">
                        @foreach($user->badges as $badge)
                            <div class="badge-item">
                                <div class="badge-icon">{{ $badge->icon }}</div>
                                <div class="badge-info">
                                    <h4>{{ $badge->name }}</h4>
                                    <p>{{ $badge->description }}</p>
                                    <small>Earned {{ $badge->pivot->created_at->diffForHumans() }}</small>
                                </div>
                            </div>
                        @endforeach
                    </div>
                @else
                    <div class="empty-state">
                        <p>No badges earned yet.</p>
                    </div>
                @endif
            </div>
        </div>
    </div>
</div>

<style>
.admin-user-detail {
    padding: 2rem 0;
    background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
    min-height: 100vh;
}

.page-header {
    margin-bottom: 2rem;
}

.back-link {
    color: #ff6600;
    text-decoration: none;
    font-weight: 600;
    margin-bottom: 1rem;
    display: inline-block;
    transition: color 0.3s ease;
}

.back-link:hover {
    color: #e55a00;
}

.page-header h1 {
    color: #e2e8f0;
    font-size: 2rem;
    margin: 0;
}

.user-profile-grid {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 2rem;
    margin-bottom: 3rem;
}

.profile-card, .activity-card {
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 102, 0, 0.2);
    border-radius: 12px;
    padding: 2rem;
}

.profile-header {
    display: flex;
    align-items: center;
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.user-avatar-large {
    width: 80px;
    height: 80px;
    background: linear-gradient(135deg, #ff6600, #e55a00);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    color: #1a1a2e;
    font-weight: bold;
    font-size: 2rem;
}

.user-main-info h2 {
    color: #e2e8f0;
    margin: 0 0 0.5rem 0;
    font-size: 1.8rem;
}

.user-main-info p {
    color: #94a3b8;
    margin: 0 0 1rem 0;
    font-size: 1.1rem;
}

.user-badges {
    display: flex;
    gap: 0.5rem;
    flex-wrap: wrap;
}

.status-badge, .level-badge, .organizer-badge, .kyc-badge, .role-badge {
    padding: 0.25rem 0.75rem;
    border-radius: 20px;
    font-size: 0.8rem;
    font-weight: 600;
    border: 1px solid;
}

.status-badge.active {
    background: rgba(34, 197, 94, 0.2);
    color: #22c55e;
    border-color: #22c55e;
}

.status-badge.banned {
    background: rgba(239, 68, 68, 0.2);
    color: #ef4444;
    border-color: #ef4444;
}

.level-badge {
    background: rgba(59, 130, 246, 0.2);
    color: #3b82f6;
    border-color: #3b82f6;
}

.role-badge {
    background: rgba(147, 51, 234, 0.2);
    color: #9333ea;
    border-color: #9333ea;
}

.role-badge.role-level1 {
    background: rgba(156, 163, 175, 0.2);
    color: #9ca3af;
    border-color: #9ca3af;
}

.role-badge.role-level2 {
    background: rgba(59, 130, 246, 0.2);
    color: #3b82f6;
    border-color: #3b82f6;
}

.role-badge.role-highest {
    background: rgba(255, 215, 0, 0.2);
    color: #ffd700;
    border-color: #ffd700;
}

.role-badge.role-creator {
    background: rgba(147, 51, 234, 0.2);
    color: #9333ea;
    border-color: #9333ea;
}

.organizer-badge {
    background: rgba(255, 215, 0, 0.2);
    color: #ffd700;
    border-color: #ffd700;
}

.kyc-badge {
    background: rgba(34, 197, 94, 0.2);
    color: #22c55e;
    border-color: #22c55e;
}

.kyc-badge.kyc-pending {
    background: rgba(245, 158, 11, 0.2);
    color: #f59e0b;
    border-color: #f59e0b;
}

.kyc-badge.kyc-rejected {
    background: rgba(239, 68, 68, 0.2);
    color: #ef4444;
    border-color: #ef4444;
}

.profile-stats {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
    gap: 1rem;
    margin-bottom: 2rem;
}

.stat-item {
    text-align: center;
    background: rgba(255, 255, 255, 0.05);
    border-radius: 8px;
    padding: 1rem;
}

.stat-item h4 {
    color: #ff6600;
    font-size: 1.5rem;
    margin: 0 0 0.5rem 0;
}

.stat-item p {
    color: #94a3b8;
    margin: 0;
    font-size: 0.9rem;
}

.profile-actions {
    display: flex;
    gap: 1rem;
    flex-wrap: wrap;
}

.btn-danger, .btn-success, .btn-secondary {
    padding: 0.75rem 1.5rem;
    border-radius: 8px;
    border: none;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    text-decoration: none;
    font-size: 0.9rem;
}

.btn-danger {
    background: rgba(239, 68, 68, 0.2);
    color: #ef4444;
    border: 1px solid #ef4444;
}

.btn-danger:hover {
    background: #ef4444;
    color: #1a1a2e;
}

.btn-success {
    background: rgba(34, 197, 94, 0.2);
    color: #22c55e;
    border: 1px solid #22c55e;
}

.btn-success:hover {
    background: #22c55e;
    color: #1a1a2e;
}

.btn-secondary {
    background: rgba(255, 102, 0, 0.2);
    color: #ff6600;
    border: 1px solid #ff6600;
}

.btn-secondary:hover {
    background: #ff6600;
    color: #1a1a2e;
}

.activity-card h3 {
    color: #ff6600;
    margin: 0 0 1.5rem 0;
    font-size: 1.5rem;
}

.activity-stats {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.activity-item {
    display: flex;
    align-items: center;
    gap: 1rem;
    background: rgba(255, 255, 255, 0.05);
    border-radius: 8px;
    padding: 1rem;
}

.activity-icon {
    font-size: 2rem;
    width: 50px;
    text-align: center;
}

.activity-info h4 {
    color: #e2e8f0;
    margin: 0 0 0.25rem 0;
    font-size: 1.5rem;
}

.activity-info p {
    color: #94a3b8;
    margin: 0 0 0.25rem 0;
    font-weight: 600;
}

.activity-info small {
    color: #64748b;
    font-size: 0.8rem;
}

.activity-tabs {
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 102, 0, 0.2);
    border-radius: 12px;
    overflow: hidden;
}

.tab-navigation {
    display: flex;
    background: rgba(255, 102, 0, 0.1);
    border-bottom: 1px solid rgba(255, 102, 0, 0.2);
}

.tab-btn {
    flex: 1;
    padding: 1rem;
    background: none;
    border: none;
    color: #94a3b8;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
}

.tab-btn.active, .tab-btn:hover {
    color: #ff6600;
    background: rgba(255, 102, 0, 0.1);
}

.tab-content {
    display: none;
    padding: 2rem;
}

.tab-content.active {
    display: block;
}

.content-header h3 {
    color: #e2e8f0;
    margin: 0 0 1.5rem 0;
    font-size: 1.3rem;
}

.challenges-list, .transactions-list {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.challenge-item, .transaction-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 255, 255, 0.1);
    border-radius: 8px;
    padding: 1rem;
}

.challenge-info h4, .transaction-info h4 {
    color: #e2e8f0;
    margin: 0 0 0.25rem 0;
    font-size: 1rem;
}

.challenge-info p, .transaction-info p {
    color: #94a3b8;
    margin: 0 0 0.25rem 0;
    font-size: 0.9rem;
}

.challenge-info small, .transaction-info small {
    color: #64748b;
    font-size: 0.8rem;
}

.challenge-meta, .transaction-meta {
    text-align: right;
}

.amount {
    display: block;
    font-weight: 600;
    margin-bottom: 0.5rem;
    color: #e2e8f0;
}

.amount.positive {
    color: #22c55e;
}

.amount.negative {
    color: #ef4444;
}

.badges-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(250px, 1fr));
    gap: 1rem;
}

.badge-item {
    display: flex;
    align-items: center;
    gap: 1rem;
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 255, 255, 0.1);
    border-radius: 8px;
    padding: 1rem;
}

.badge-icon {
    font-size: 2rem;
    width: 50px;
    text-align: center;
}

.badge-info h4 {
    color: #e2e8f0;
    margin: 0 0 0.25rem 0;
    font-size: 1rem;
}

.badge-info p {
    color: #94a3b8;
    margin: 0 0 0.25rem 0;
    font-size: 0.9rem;
}

.badge-info small {
    color: #64748b;
    font-size: 0.8rem;
}

.empty-state {
    text-align: center;
    padding: 3rem;
    color: #94a3b8;
}

.management-section {
    margin-top: 2rem;
    padding-top: 2rem;
    border-top: 1px solid rgba(255, 255, 255, 0.1);
}

.management-section h4 {
    color: #ff6600;
    margin: 0 0 1rem 0;
    font-size: 1.2rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.role-form, .kyc-form {
    margin-bottom: 2rem;
    background: rgba(255, 255, 255, 0.05);
    padding: 1.5rem;
    border-radius: 8px;
    border: 1px solid rgba(255, 255, 255, 0.1);
}

.form-group {
    margin-bottom: 1rem;
}

.form-group label {
    display: block;
    color: #e2e8f0;
    font-weight: 600;
    margin-bottom: 0.5rem;
}

.form-select {
    width: 100%;
    padding: 0.75rem;
    background: rgba(255, 255, 255, 0.1);
    border: 1px solid rgba(255, 255, 255, 0.2);
    border-radius: 6px;
    color: #e2e8f0;
    font-size: 1rem;
}

.form-select:focus {
    outline: none;
    border-color: #ff6600;
    box-shadow: 0 0 0 2px rgba(255, 102, 0, 0.2);
}

.form-select option {
    background: #1a1a2e;
    color: #e2e8f0;
}

.btn-primary {
    background: rgba(255, 102, 0, 0.2);
    color: #ff6600;
    border: 1px solid #ff6600;
    padding: 0.75rem 1.5rem;
    border-radius: 6px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
}

.btn-primary:hover {
    background: #ff6600;
    color: #1a1a2e;
}

@media (max-width: 768px) {
    .user-profile-grid {
        grid-template-columns: 1fr;
    }
    
    .profile-header {
        flex-direction: column;
        text-align: center;
    }
    
    .profile-stats {
        grid-template-columns: repeat(2, 1fr);
    }
    
    .profile-actions {
        flex-direction: column;
    }
    
    .tab-navigation {
        flex-direction: column;
    }
    
    .challenge-item, .transaction-item {
        flex-direction: column;
        align-items: flex-start;
        gap: 1rem;
    }
    
    .challenge-meta, .transaction-meta {
        text-align: left;
        width: 100%;
    }
    
    .badges-grid {
        grid-template-columns: 1fr;
    }
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const tabBtns = document.querySelectorAll('.tab-btn');
    const tabContents = document.querySelectorAll('.tab-content');
    
    tabBtns.forEach(btn => {
        btn.addEventListener('click', function() {
            const targetTab = this.dataset.tab;
            
            // Remove active class from all tabs and contents
            tabBtns.forEach(b => b.classList.remove('active'));
            tabContents.forEach(c => c.classList.remove('active'));
            
            // Add active class to clicked tab and corresponding content
            this.classList.add('active');
            document.getElementById(targetTab).classList.add('active');
        });
    });
});
</script>
@endsection