@extends('layouts.app')

@section('title', 'Create Challenge')

@section('content')
<div class="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
    <div class="text-center mb-8">
        <a href="{{ route('challenges.index') }}" class="inline-flex items-center text-cod-green hover:text-green-400 transition-colors mb-4">
            <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"></path>
            </svg>
            Back to Challenges
        </a>
        <h1 class="text-4xl font-gaming font-bold text-cod-green gaming-text-glow mb-4">Create Challenge</h1>
        <p class="text-gray-400">Set up your gaming challenge and let others accept it</p>
    </div>

    <div class="bg-cod-gray rounded-xl p-8 gaming-glow border border-cod-green/20">
        <form method="POST" action="{{ route('challenges.store') }}" class="space-y-8">
            @csrf

            <!-- Challenge Details Section -->
            <div class="space-y-6">
                <h3 class="text-xl font-bold text-cod-green mb-4">Challenge Details</h3>
                
                <!-- Game Selection -->
                <div class="space-y-2">
                    <label for="game_id" class="block text-sm font-medium text-cod-green">Game *</label>
                    <select id="game_id" 
                            name="game_id" 
                            required
                            class="gaming-select focus-ring {{ $errors->has('game_id') ? 'form-error' : '' }}">
                        <option value="">Select a game</option>
                        @foreach($games as $game)
                            <option value="{{ $game->id }}" {{ old('game_id') == $game->id ? 'selected' : '' }}>
                                {{ $game->name }} ({{ ucfirst($game->platform) }})
                            </option>
                        @endforeach
                    </select>
                    @error('game_id')
                        <div class="error-message">
                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                            </svg>
                            <span>{{ $message }}</span>
                        </div>
                    @enderror
                </div>

                <!-- Wager Amount with Enhanced UI -->
                <div class="space-y-2">
                    <label for="wager_amount" class="block text-sm font-medium text-cod-green">Wager Amount *</label>
                    <div class="input-with-icon">
                        <div class="input-icon">
                            <span class="text-lg">₦</span>
                        </div>
                        <input type="number" 
                               id="wager_amount" 
                               name="wager_amount" 
                               value="{{ old('wager_amount') }}"
                               min="1" 
                               max="{{ auth()->user()->wallet_balance }}"
                               required
                               class="gaming-input focus-ring {{ $errors->has('wager_amount') ? 'form-error' : '' }}"
                               placeholder="Enter wager amount">
                    </div>
                    <div class="flex items-center justify-between text-sm">
                        <span class="text-gray-400">Your wallet balance: ₦{{ number_format(auth()->user()->wallet_balance) }}</span>
                        <div class="flex space-x-2">
                            <button type="button" onclick="setWagerAmount(100)" class="text-xs px-2 py-1 bg-cod-light text-gray-300 rounded hover:bg-cod-green hover:text-black transition-colors">₦100</button>
                            <button type="button" onclick="setWagerAmount(500)" class="text-xs px-2 py-1 bg-cod-light text-gray-300 rounded hover:bg-cod-green hover:text-black transition-colors">₦500</button>
                            <button type="button" onclick="setWagerAmount(1000)" class="text-xs px-2 py-1 bg-cod-light text-gray-300 rounded hover:bg-cod-green hover:text-black transition-colors">₦1000</button>
                        </div>
                    </div>
                    @error('wager_amount')
                        <div class="error-message">
                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                            </svg>
                            <span>{{ $message }}</span>
                        </div>
                    @enderror
                </div>
            </div>

            <!-- Game Details Section -->
            <div class="space-y-6 border-t border-cod-green/20 pt-6">
                <h3 class="text-xl font-bold text-cod-green mb-4">Game Setup</h3>
                
                <!-- Description with Floating Label -->
                <div class="floating-label-group">
                    <textarea id="description" 
                              name="description" 
                              rows="4"
                              class="floating-label-input resize-none {{ $errors->has('description') ? 'form-error' : '' }}"
                              placeholder="Describe your challenge rules and requirements...">{{ old('description') }}</textarea>
                    <label for="description" class="floating-label">Challenge Description</label>
                    @error('description')
                        <div class="error-message">
                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                            </svg>
                            <span>{{ $message }}</span>
                        </div>
                    @enderror
                </div>

                <!-- Platform Details -->
                <div class="floating-label-group">
                    <input type="text" 
                           id="platform_details" 
                           name="platform_details" 
                           value="{{ old('platform_details') }}"
                           class="floating-label-input {{ $errors->has('platform_details') ? 'form-error' : '' }}"
                           placeholder="e.g., PlayStation Network, Steam, Xbox Live">
                    <label for="platform_details" class="floating-label">Platform Details</label>
                    @error('platform_details')
                        <div class="error-message">
                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                            </svg>
                            <span>{{ $message }}</span>
                        </div>
                    @enderror
                </div>
            </div>

            <!-- Best Of -->
            <div>
                <label for="best_of" class="block text-sm font-medium text-gray-300 mb-2">Best Of</label>
                <select id="best_of" 
                        name="best_of" 
                        required
                        class="w-full px-4 py-3 bg-cod-dark border border-cod-light rounded-lg text-gray-100 focus:outline-none focus:ring-2 focus:ring-cod-green focus:border-transparent">
                    <option value="1" {{ old('best_of', 1) == 1 ? 'selected' : '' }}>Best of 1</option>
                    <option value="3" {{ old('best_of') == 3 ? 'selected' : '' }}>Best of 3</option>
                    <option value="5" {{ old('best_of') == 5 ? 'selected' : '' }}>Best of 5</option>
                    <option value="7" {{ old('best_of') == 7 ? 'selected' : '' }}>Best of 7</option>
                </select>
                @error('best_of')
                    <p class="mt-1 text-sm text-red-400">{{ $message }}</p>
                @enderror
            </div>

            <!-- Rules -->
            <div>
                <label for="rules_text" class="block text-sm font-medium text-gray-300 mb-2">Challenge Rules (Optional)</label>
                <textarea id="rules_text" 
                          name="rules_text" 
                          rows="4" 
                          maxlength="1000"
                          class="w-full px-4 py-3 bg-cod-dark border border-cod-light rounded-lg text-gray-100 focus:outline-none focus:ring-2 focus:ring-cod-green focus:border-transparent"
                          placeholder="Describe any specific rules for this challenge...">{{ old('rules_text') }}</textarea>
                <div class="mt-1 text-sm text-gray-400">
                    <span id="rules-counter">{{ strlen(old('rules_text', '')) }}</span>/1000 characters
                </div>
                @error('rules_text')
                    <p class="mt-1 text-sm text-red-400">{{ $message }}</p>
                @enderror
            </div>

            <!-- Game Join Information -->
            <div class="bg-cod-dark rounded-lg p-6 border-l-4 border-cod-green">
                <h3 class="text-lg font-semibold text-cod-green mb-4">Game Join Information</h3>
                <p class="text-sm text-gray-400 mb-4">Provide your in-game username and group/lobby information so opponents know how to join your match.</p>
                
                <!-- In-Game Username -->
                <div class="mb-4">
                    <label for="creator_game_username" class="block text-sm font-medium text-gray-300 mb-2">Your In-Game Username <span class="text-red-400">*</span></label>
                    <input type="text" 
                           id="creator_game_username" 
                           name="creator_game_username" 
                           value="{{ old('creator_game_username') }}"
                           required
                           class="w-full px-4 py-3 bg-cod-gray border border-cod-light rounded-lg text-gray-100 focus:outline-none focus:ring-2 focus:ring-cod-green focus:border-transparent"
                           placeholder="Enter your in-game username/gamertag">
                    @error('creator_game_username')
                        <p class="mt-1 text-sm text-red-400">{{ $message }}</p>
                    @enderror
                </div>

                <!-- Group Link -->
                <div class="mb-4">
                    <label for="group_link" class="block text-sm font-medium text-gray-300 mb-2">Group/Lobby Link (Optional)</label>
                    <input type="url" 
                           id="group_link" 
                           name="group_link" 
                           value="{{ old('group_link') }}"
                           class="w-full px-4 py-3 bg-cod-gray border border-cod-light rounded-lg text-gray-100 focus:outline-none focus:ring-2 focus:ring-cod-green focus:border-transparent"
                           placeholder="https://example.com/join/lobby">
                    <div class="mt-1 text-sm text-gray-400">Provide a direct link to join your game lobby/room if applicable</div>
                    @error('group_link')
                        <p class="mt-1 text-sm text-red-400">{{ $message }}</p>
                    @enderror
                </div>

                <!-- Group ID -->
                <div>
                    <label for="group_id" class="block text-sm font-medium text-gray-300 mb-2">Group/Room ID (Optional)</label>
                    <input type="text" 
                           id="group_id" 
                           name="group_id" 
                           value="{{ old('group_id') }}"
                           class="w-full px-4 py-3 bg-cod-gray border border-cod-light rounded-lg text-gray-100 focus:outline-none focus:ring-2 focus:ring-cod-green focus:border-transparent"
                           placeholder="Enter group/room/lobby ID">
                    <div class="mt-1 text-sm text-gray-400">Provide group/room ID, lobby code, or match ID if applicable</div>
                    @error('group_id')
                        <p class="mt-1 text-sm text-red-400">{{ $message }}</p>
                    @enderror
                </div>
            </div>

            <!-- Prize Information -->
            <div class="bg-cod-dark rounded-lg p-6">
                <h3 class="text-lg font-semibold text-cod-green mb-4">Prize Information</h3>
                <div class="grid grid-cols-1 md:grid-cols-3 gap-4 text-center">
                    <div>
                        <div class="text-sm text-gray-400">Your Stake</div>
                        <div class="text-xl font-bold text-cod-green" id="your-stake">₦0</div>
                    </div>
                    <div>
                        <div class="text-sm text-gray-400">Opponent Stake</div>
                        <div class="text-xl font-bold text-cod-green" id="opponent-stake">₦0</div>
                    </div>
                    <div>
                        <div class="text-sm text-gray-400">Winner Takes</div>
                        <div class="text-xl font-bold text-green-400" id="total-prize">₦0</div>
                    </div>
                </div>
                <div class="mt-4 text-xs text-gray-500 text-center">
                    *5% platform fee will be deducted from the prize
                </div>
            </div>

            <!-- Submit Button -->
            <div class="flex space-x-4">
                <button type="submit" 
                        class="flex-1 bg-cod-green text-black px-6 py-3 rounded-lg font-semibold hover:bg-green-600 transition-all transform hover:scale-105 gaming-glow">
                    Create Challenge
                </button>
                <a href="{{ route('challenges.index') }}" 
                   class="flex-1 bg-cod-light text-cod-green text-center px-6 py-3 rounded-lg font-semibold hover:bg-cod-gray transition-colors">
                    Cancel
                </a>
            </div>
        </form>
    </div>

    <!-- Quick Tips Section -->
    <div class="mt-8 bg-cod-dark/50 rounded-xl p-6 border border-cod-green/10">
        <h4 class="text-lg font-bold text-cod-green mb-4">💡 Quick Tips</h4>
        <div class="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm text-gray-400">
            <div class="flex items-start space-x-2">
                <span class="text-cod-green">✓</span>
                <span>Set a fair wager amount that attracts opponents</span>
            </div>
            <div class="flex items-start space-x-2">
                <span class="text-cod-green">✓</span>
                <span>Be specific about game rules and victory conditions</span>
            </div>
            <div class="flex items-start space-x-2">
                <span class="text-cod-green">✓</span>
                <span>Include your platform username for easy matchmaking</span>
            </div>
            <div class="flex items-start space-x-2">
                <span class="text-cod-green">✓</span>
                <span>Check your wallet balance before creating challenges</span>
            </div>
        </div>
    </div>
</div>

<script>
    function setWagerAmount(amount) {
        const input = document.getElementById('wager_amount');
        const maxAmount = {{ auth()->user()->wallet_balance }};
        
        if (amount <= maxAmount) {
            input.value = amount;
            input.focus();
        } else {
            alert('Amount exceeds your wallet balance!');
        }
    }

    // Form validation enhancements
    document.addEventListener('DOMContentLoaded', function() {
        const form = document.querySelector('form');
        const inputs = form.querySelectorAll('input, select, textarea');

        inputs.forEach(input => {
            input.addEventListener('blur', function() {
                validateField(this);
            });

            input.addEventListener('input', function() {
                if (this.classList.contains('form-error')) {
                    validateField(this);
                }
            });
        });

        function validateField(field) {
            const value = field.value.trim();
            const isRequired = field.hasAttribute('required');

            // Remove existing validation classes
            field.classList.remove('form-error', 'form-success');

            if (isRequired && !value) {
                field.classList.add('form-error');
            } else if (value) {
                field.classList.add('form-success');
            }
        }
    });
</script>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const wagerInput = document.getElementById('wager_amount');
    const yourStake = document.getElementById('your-stake');
    const opponentStake = document.getElementById('opponent-stake');
    const totalPrize = document.getElementById('total-prize');
    const rulesTextarea = document.getElementById('rules_text');
    const rulesCounter = document.getElementById('rules-counter');

    function updatePrizeCalculation() {
        const amount = parseFloat(wagerInput.value) || 0;
        const total = amount * 2;
        const fee = total * 0.05;
        const winnerTakes = total - fee;

        yourStake.textContent = '₦' + amount.toLocaleString();
        opponentStake.textContent = '₦' + amount.toLocaleString();
        totalPrize.textContent = '₦' + winnerTakes.toLocaleString();
    }

    function updateRulesCounter() {
        rulesCounter.textContent = rulesTextarea.value.length;
    }

    wagerInput.addEventListener('input', updatePrizeCalculation);
    rulesTextarea.addEventListener('input', updateRulesCounter);

    // Initialize
    updatePrizeCalculation();
    updateRulesCounter();
});
</script>
@endsection