@extends('layouts.app')

@section('title', 'Challenges')

@section('content')
<div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
    <!-- Header -->
    <div class="flex justify-between items-center mb-8">
        <h1 class="text-3xl font-gaming font-bold text-cod-green">Gaming Challenges</h1>
        @auth
            @if(auth()->user()->canCreateChallenges())
                <a href="{{ route('challenges.create') }}" class="bg-cod-green text-black px-6 py-3 rounded-lg font-semibold hover:bg-green-600 transition-colors">
                    Create Challenge
                </a>
            @else
                <span class="text-gray-400 text-sm">Reach Level 2 to create challenges</span>
            @endif
        @else
            <a href="{{ route('login') }}" class="bg-cod-green text-black px-6 py-3 rounded-lg font-semibold hover:bg-green-600 transition-colors">
                Login to Create
            </a>
        @endauth
    </div>

    <!-- Global/Friends Tabs -->
    @auth
    <div class="flex space-x-1 mb-6">
        <a href="{{ route('challenges.index', ['filter' => 'global'] + request()->except('filter')) }}" 
           class="px-6 py-3 rounded-lg font-semibold transition-colors {{ ($filter ?? 'global') === 'global' ? 'bg-cod-green text-black' : 'bg-cod-gray text-gray-300 hover:bg-cod-light' }}">
            🌍 Global Challenges
        </a>
        <a href="{{ route('challenges.index', ['filter' => 'friends'] + request()->except('filter')) }}" 
           class="px-6 py-3 rounded-lg font-semibold transition-colors {{ ($filter ?? 'global') === 'friends' ? 'bg-cod-green text-black' : 'bg-cod-gray text-gray-300 hover:bg-cod-light' }}">
            👥 Friends Only
        </a>
    </div>
    @endauth

    <!-- Enhanced Filters -->
    <div class="bg-cod-gray rounded-lg p-6 mb-8">
        <form method="GET" action="{{ route('challenges.index') }}">
            <input type="hidden" name="filter" value="{{ $filter ?? 'global' }}">
            <div class="grid grid-cols-1 md:grid-cols-4 gap-4">
                <select name="game_id" class="bg-cod-dark text-gray-100 border border-cod-light rounded-lg px-4 py-2">
                    <option value="">All Games</option>
                    @foreach($games as $game)
                        <option value="{{ $game->id }}" {{ request('game_id') == $game->id ? 'selected' : '' }}>
                            {{ $game->name }}
                        </option>
                    @endforeach
                </select>
                <select name="wager_range" class="bg-cod-dark text-gray-100 border border-cod-light rounded-lg px-4 py-2">
                    <option value="">All Wager Amounts</option>
                    <option value="100-500" {{ request('wager_range') === '100-500' ? 'selected' : '' }}>₦100 - ₦500</option>
                    <option value="500-1000" {{ request('wager_range') === '500-1000' ? 'selected' : '' }}>₦500 - ₦1,000</option>
                    <option value="1000+" {{ request('wager_range') === '1000+' ? 'selected' : '' }}>₦1,000+</option>
                </select>
                <select name="best_of" class="bg-cod-dark text-gray-100 border border-cod-light rounded-lg px-4 py-2">
                    <option value="">Best of Any</option>
                    <option value="1" {{ request('best_of') == '1' ? 'selected' : '' }}>Best of 1</option>
                    <option value="3" {{ request('best_of') == '3' ? 'selected' : '' }}>Best of 3</option>
                    <option value="5" {{ request('best_of') == '5' ? 'selected' : '' }}>Best of 5</option>
                </select>
                <button type="submit" class="bg-cod-green text-black px-4 py-2 rounded-lg font-semibold hover:bg-green-600 transition-colors">
                    Apply Filters
                </button>
            </div>
        </form>
    </div>

    <!-- Results Info -->
    @auth
        @if(($filter ?? 'global') === 'friends')
            <div class="bg-blue-900/20 border border-blue-500/30 rounded-lg p-4 mb-6">
                <div class="flex items-center space-x-2">
                    <span class="text-blue-400">👥</span>
                    <span class="text-blue-300 font-semibold">Showing challenges from your friends only</span>
                </div>
                <p class="text-blue-200 text-sm mt-1">Switch to Global to see all challenges, or <a href="{{ route('friends.suggestions') }}" class="text-blue-400 hover:text-blue-300 underline">find more friends</a>.</p>
            </div>
        @endif
    @endauth

    <!-- Challenges Grid -->
    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
        @forelse($challenges as $challenge)
            <div class="bg-cod-gray rounded-lg overflow-hidden hover:gaming-glow transition-all duration-300">
                <div class="p-6">
                    <!-- Challenge Header -->
                    <div class="flex justify-between items-start mb-4">
                        <div>
                            <h3 class="text-xl font-semibold text-cod-green">{{ $challenge->game->name }}</h3>
                            <div class="flex items-center space-x-2 mt-1">
                                <p class="text-gray-400 text-sm">by {{ $challenge->creator->username }}</p>
                                @auth
                                    @if(auth()->user()->isFriendsWith($challenge->creator))
                                        <span class="bg-blue-500/20 text-blue-400 px-2 py-1 rounded text-xs">Friend</span>
                                    @endif
                                @endauth
                            </div>
                        </div>
                        <div class="text-right">
                            <div class="text-2xl font-bold text-green-400">₦{{ number_format($challenge->wager_amount) }}</div>
                            <div class="text-xs text-gray-400">per player</div>
                        </div>
                    </div>

                    <!-- Challenge Details -->
                    <div class="space-y-2 mb-4">
                        <div class="flex justify-between">
                            <span class="text-gray-400">Format:</span>
                            <span class="text-gray-100">Best of {{ $challenge->best_of }}</span>
                        </div>
                        <div class="flex justify-between">
                            <span class="text-gray-400">Platform:</span>
                            <span class="text-gray-100">{{ ucfirst($challenge->game->platform) }}</span>
                        </div>
                        <div class="flex justify-between">
                            <span class="text-gray-400">Total Prize:</span>
                            <span class="text-green-400 font-bold">₦{{ number_format($challenge->wager_amount * 2) }}</span>
                        </div>
                        <div class="flex justify-between">
                            <span class="text-gray-400">Created:</span>
                            <span class="text-gray-300">{{ $challenge->created_at->diffForHumans() }}</span>
                        </div>
                    </div>

                    <!-- Rules -->
                    @if($challenge->rules_text)
                        <div class="mb-4">
                            <h4 class="text-sm font-semibold text-gray-300 mb-2">Rules:</h4>
                            <p class="text-gray-400 text-sm">{{ Str::limit($challenge->rules_text, 100) }}</p>
                        </div>
                    @endif

                    <!-- Actions -->
                    <div class="flex space-x-2">
                        <a href="{{ route('challenges.show', $challenge) }}" class="flex-1 bg-cod-light text-center text-cod-green px-4 py-2 rounded hover:bg-cod-green hover:text-black transition-colors">
                            View Details
                        </a>
                        @auth
                            @if($challenge->canBeAcceptedBy(auth()->user()))
                                <form method="POST" action="{{ route('challenges.accept', $challenge) }}" class="flex-1">
                                    @csrf
                                    <button type="submit" class="w-full bg-cod-green text-black px-4 py-2 rounded font-semibold hover:bg-green-600 transition-colors">
                                        Accept (₦{{ number_format($challenge->wager_amount) }})
                                    </button>
                                </form>
                            @endif
                        @endauth
                    </div>

                    <!-- Friend Action -->
                    @auth
                        @if($challenge->creator_id !== auth()->id() && auth()->user()->hasPlayedWith($challenge->creator) && !auth()->user()->isFriendsWith($challenge->creator) && !auth()->user()->hasPendingRequestWith($challenge->creator))
                            <div class="mt-3 pt-3 border-t border-cod-light">
                                <form method="POST" action="{{ route('friends.send-request', $challenge->creator) }}">
                                    @csrf
                                    <button type="submit" class="w-full bg-blue-600/20 text-blue-400 px-3 py-2 rounded text-sm hover:bg-blue-600 hover:text-white transition-colors">
                                        Add {{ $challenge->creator->username }} as Friend
                                    </button>
                                </form>
                            </div>
                        @endif
                    @endauth
                </div>
            </div>
        @empty
            <div class="col-span-full text-center py-16">
                <div class="text-6xl mb-4">🎮</div>
                @auth
                    @if(($filter ?? 'global') === 'friends')
                        <h3 class="text-xl font-semibold text-gray-400 mb-2">No Friend Challenges</h3>
                        <p class="text-gray-500 mb-4">Your friends haven't created any challenges yet.</p>
                        <div class="space-x-4">
                            <a href="{{ route('challenges.index', ['filter' => 'global']) }}" class="inline-block bg-cod-green text-black px-6 py-3 rounded-lg font-semibold hover:bg-green-600 transition-colors">
                                View Global Challenges
                            </a>
                            <a href="{{ route('friends.suggestions') }}" class="inline-block bg-cod-light text-cod-green px-6 py-3 rounded-lg font-semibold hover:bg-cod-green hover:text-black transition-colors">
                                Find Friends
                            </a>
                        </div>
                    @else
                        <h3 class="text-xl font-semibold text-gray-400 mb-2">No Open Challenges</h3>
                        <p class="text-gray-500 mb-4">Be the first to create a challenge!</p>
                        @if(auth()->user()->canCreateChallenges())
                            <a href="{{ route('challenges.create') }}" class="inline-block bg-cod-green text-black px-6 py-3 rounded-lg font-semibold hover:bg-green-600 transition-colors">
                                Create First Challenge
                            </a>
                        @endif
                    @endif
                @else
                    <h3 class="text-xl font-semibold text-gray-400 mb-2">No Open Challenges</h3>
                    <p class="text-gray-500">Check back later or create your own challenge!</p>
                @endauth
            </div>
        @endforelse
    </div>

    <!-- Pagination -->
    @if($challenges->hasPages())
        <div class="mt-8">
            {{ $challenges->appends(request()->query())->links() }}
        </div>
    @endif
</div>
@endsection