@extends('layouts.app')

@section('title', 'KYC Verification')

@section('content')
<div class="kyc-verification">
    <div class="container">
        <div class="kyc-header">
            <div class="kyc-icon">🆔</div>
            <h1>KYC Verification</h1>
            <p>Complete your identity verification to access all platform features</p>
        </div>

        @if(session('success'))
            <div class="alert alert-success">
                <i class="icon">✅</i>
                {{ session('success') }}
            </div>
        @endif

        @if(session('error'))
            <div class="alert alert-error">
                <i class="icon">❌</i>
                {{ session('error') }}
            </div>
        @endif

        @if($user->kyc_status === 'verified')
            <div class="kyc-status-card verified">
                <div class="status-icon">✅</div>
                <h3>Verification Complete</h3>
                <p>Your identity has been successfully verified. You now have access to all platform features including withdrawals.</p>
                <a href="{{ route('dashboard') }}" class="btn-primary">Go to Dashboard</a>
            </div>
        @elseif($user->kyc_status === 'pending')
            <div class="kyc-status-card pending">
                <div class="status-icon">⏳</div>
                <h3>Verification Pending</h3>
                <p>Your documents are being reviewed. This usually takes 24-48 hours. We'll notify you once the review is complete.</p>
                <a href="{{ route('kyc.status') }}" class="btn-secondary">Check Status</a>
            </div>
        @elseif($user->kyc_status === 'rejected')
            <div class="kyc-status-card rejected">
                <div class="status-icon">❌</div>
                <h3>Verification Rejected</h3>
                <p>Your previous verification was rejected. Please review the requirements and submit new documents.</p>
                <a href="#kyc-form" class="btn-primary">Submit New Documents</a>
            </div>
        @endif

        @if(in_array($user->kyc_status, ['unverified', 'rejected']))
            <div class="kyc-benefits">
                <h3>🔑 Benefits of KYC Verification</h3>
                <div class="benefits-grid">
                    <div class="benefit-item">
                        <div class="benefit-icon">💰</div>
                        <h4>Withdraw Funds</h4>
                        <p>Access your earnings and withdraw funds to your bank account</p>
                    </div>
                    <div class="benefit-item">
                        <div class="benefit-icon">🔒</div>
                        <h4>Enhanced Security</h4>
                        <p>Protect your account with verified identity</p>
                    </div>
                    <div class="benefit-item">
                        <div class="benefit-icon">🏆</div>
                        <h4>Higher Limits</h4>
                        <p>Access higher challenge amounts and premium features</p>
                    </div>
                    <div class="benefit-item">
                        <div class="benefit-icon">⚡</div>
                        <h4>Fast Transactions</h4>
                        <p>Faster processing of deposits and withdrawals</p>
                    </div>
                </div>
            </div>

            <div id="kyc-form" class="kyc-form-container">
                <h3>📄 Submit Verification Documents</h3>
                <form method="POST" action="{{ route('kyc.submit') }}" enctype="multipart/form-data" class="kyc-form">
                    @csrf
                    
                    <div class="form-section">
                        <h4>Personal Information</h4>
                        <div class="form-row">
                            <div class="form-group">
                                <label for="first_name">First Name *</label>
                                <input type="text" name="first_name" id="first_name" 
                                       value="{{ old('first_name', $user->first_name) }}" required>
                                @error('first_name')
                                    <span class="error-text">{{ $message }}</span>
                                @enderror
                            </div>
                            <div class="form-group">
                                <label for="last_name">Last Name *</label>
                                <input type="text" name="last_name" id="last_name" 
                                       value="{{ old('last_name', $user->last_name) }}" required>
                                @error('last_name')
                                    <span class="error-text">{{ $message }}</span>
                                @enderror
                            </div>
                        </div>

                        <div class="form-row">
                            <div class="form-group">
                                <label for="date_of_birth">Date of Birth *</label>
                                <input type="date" name="date_of_birth" id="date_of_birth" 
                                       value="{{ old('date_of_birth', $user->date_of_birth) }}" required>
                                @error('date_of_birth')
                                    <span class="error-text">{{ $message }}</span>
                                @enderror
                            </div>
                            <div class="form-group">
                                <label for="phone_number">Phone Number *</label>
                                <input type="tel" name="phone_number" id="phone_number" 
                                       value="{{ old('phone_number', $user->phone_number) }}" required>
                                @error('phone_number')
                                    <span class="error-text">{{ $message }}</span>
                                @enderror
                            </div>
                        </div>

                        <div class="form-group">
                            <label for="address">Address *</label>
                            <textarea name="address" id="address" rows="3" required>{{ old('address', $user->address) }}</textarea>
                            @error('address')
                                <span class="error-text">{{ $message }}</span>
                            @enderror
                        </div>
                    </div>

                    <div class="form-section">
                        <h4>Identity Verification</h4>
                        <div class="form-row">
                            <div class="form-group">
                                <label for="id_type">ID Type *</label>
                                <select name="id_type" id="id_type" required>
                                    <option value="">Select ID Type</option>
                                    <option value="nin" {{ old('id_type') === 'nin' ? 'selected' : '' }}>National Identification Number (NIN)</option>
                                    <option value="bvn" {{ old('id_type') === 'bvn' ? 'selected' : '' }}>Bank Verification Number (BVN)</option>
                                    <option value="drivers_license" {{ old('id_type') === 'drivers_license' ? 'selected' : '' }}>Driver's License</option>
                                    <option value="international_passport" {{ old('id_type') === 'international_passport' ? 'selected' : '' }}>International Passport</option>
                                </select>
                                @error('id_type')
                                    <span class="error-text">{{ $message }}</span>
                                @enderror
                            </div>
                            <div class="form-group">
                                <label for="id_number">ID Number *</label>
                                <input type="text" name="id_number" id="id_number" 
                                       value="{{ old('id_number') }}" required>
                                @error('id_number')
                                    <span class="error-text">{{ $message }}</span>
                                @enderror
                            </div>
                        </div>
                    </div>

                    <div class="form-section">
                        <h4>Document Upload</h4>
                        <div class="upload-grid">
                            <div class="upload-item">
                                <label for="id_document">ID Document *</label>
                                <div class="upload-zone" onclick="document.getElementById('id_document').click()">
                                    <div class="upload-icon">📄</div>
                                    <p>Click to upload ID document</p>
                                    <small>JPG, PNG (Max: 2MB)</small>
                                </div>
                                <input type="file" name="id_document" id="id_document" accept="image/*" required hidden>
                                @error('id_document')
                                    <span class="error-text">{{ $message }}</span>
                                @enderror
                            </div>

                            <div class="upload-item">
                                <label for="selfie">Selfie Photo *</label>
                                <div class="upload-zone" onclick="document.getElementById('selfie').click()">
                                    <div class="upload-icon">🤳</div>
                                    <p>Click to upload selfie</p>
                                    <small>JPG, PNG (Max: 2MB)</small>
                                </div>
                                <input type="file" name="selfie" id="selfie" accept="image/*" required hidden>
                                @error('selfie')
                                    <span class="error-text">{{ $message }}</span>
                                @enderror
                            </div>
                        </div>
                    </div>

                    <div class="form-actions">
                        <button type="submit" class="btn-submit">
                            <i class="icon">🔒</i>
                            Submit for Verification
                        </button>
                        <p class="terms-text">
                            By submitting, you agree to our <a href="{{ route('terms-of-service') }}">Terms of Service</a> 
                            and <a href="{{ route('privacy-policy') }}">Privacy Policy</a>
                        </p>
                    </div>
                </form>
            </div>
        @endif
    </div>
</div>

<style>
.kyc-verification {
    padding: 2rem 0;
    background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
    min-height: 100vh;
}

.kyc-header {
    text-align: center;
    margin-bottom: 3rem;
}

.kyc-icon {
    font-size: 4rem;
    margin-bottom: 1rem;
}

.kyc-header h1 {
    color: #ff6600;
    font-size: 2.5rem;
    margin: 0 0 1rem 0;
}

.kyc-header p {
    color: #94a3b8;
    font-size: 1.2rem;
    margin: 0;
}

.alert {
    padding: 1rem 1.5rem;
    border-radius: 8px;
    margin-bottom: 2rem;
    display: flex;
    align-items: center;
    gap: 1rem;
    border: 1px solid;
}

.alert-success {
    background: rgba(34, 197, 94, 0.1);
    color: #22c55e;
    border-color: #22c55e;
}

.alert-error {
    background: rgba(239, 68, 68, 0.1);
    color: #ef4444;
    border-color: #ef4444;
}

.kyc-status-card {
    text-align: center;
    padding: 3rem;
    border-radius: 12px;
    margin-bottom: 3rem;
    border: 2px solid;
}

.kyc-status-card.verified {
    background: rgba(34, 197, 94, 0.1);
    border-color: #22c55e;
}

.kyc-status-card.pending {
    background: rgba(245, 158, 11, 0.1);
    border-color: #f59e0b;
}

.kyc-status-card.rejected {
    background: rgba(239, 68, 68, 0.1);
    border-color: #ef4444;
}

.status-icon {
    font-size: 4rem;
    margin-bottom: 1rem;
}

.kyc-status-card h3 {
    color: #e2e8f0;
    margin: 0 0 1rem 0;
    font-size: 1.8rem;
}

.kyc-status-card p {
    color: #94a3b8;
    margin: 0 0 2rem 0;
    font-size: 1.1rem;
    max-width: 500px;
    margin-left: auto;
    margin-right: auto;
}

.kyc-benefits {
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 102, 0, 0.2);
    border-radius: 12px;
    padding: 2rem;
    margin-bottom: 3rem;
}

.kyc-benefits h3 {
    color: #ff6600;
    margin: 0 0 2rem 0;
    font-size: 1.5rem;
    text-align: center;
}

.benefits-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 2rem;
}

.benefit-item {
    text-align: center;
    background: rgba(255, 255, 255, 0.05);
    border-radius: 8px;
    padding: 1.5rem;
}

.benefit-icon {
    font-size: 2.5rem;
    margin-bottom: 1rem;
}

.benefit-item h4 {
    color: #e2e8f0;
    margin: 0 0 0.5rem 0;
    font-size: 1.2rem;
}

.benefit-item p {
    color: #94a3b8;
    margin: 0;
    font-size: 0.9rem;
}

.kyc-form-container {
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 102, 0, 0.2);
    border-radius: 12px;
    padding: 2rem;
}

.kyc-form-container h3 {
    color: #ff6600;
    margin: 0 0 2rem 0;
    font-size: 1.5rem;
    text-align: center;
}

.form-section {
    margin-bottom: 2rem;
    padding-bottom: 2rem;
    border-bottom: 1px solid rgba(255, 255, 255, 0.1);
}

.form-section:last-child {
    border-bottom: none;
    margin-bottom: 0;
    padding-bottom: 0;
}

.form-section h4 {
    color: #e2e8f0;
    margin: 0 0 1.5rem 0;
    font-size: 1.2rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.form-row {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 1rem;
}

.form-group {
    margin-bottom: 1rem;
}

.form-group label {
    display: block;
    color: #e2e8f0;
    font-weight: 600;
    margin-bottom: 0.5rem;
}

.form-group input,
.form-group select,
.form-group textarea {
    width: 100%;
    padding: 0.75rem;
    background: rgba(255, 255, 255, 0.1);
    border: 1px solid rgba(255, 255, 255, 0.2);
    border-radius: 6px;
    color: #e2e8f0;
    font-size: 1rem;
    transition: border-color 0.3s ease;
}

.form-group input:focus,
.form-group select:focus,
.form-group textarea:focus {
    outline: none;
    border-color: #ff6600;
    box-shadow: 0 0 0 2px rgba(255, 102, 0, 0.2);
}

.form-group select option {
    background: #1a1a2e;
    color: #e2e8f0;
}

.error-text {
    color: #ef4444;
    font-size: 0.8rem;
    display: block;
    margin-top: 0.25rem;
}

.upload-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 2rem;
}

.upload-item label {
    display: block;
    color: #e2e8f0;
    font-weight: 600;
    margin-bottom: 1rem;
}

.upload-zone {
    border: 2px dashed rgba(255, 102, 0, 0.3);
    border-radius: 8px;
    padding: 2rem;
    text-align: center;
    cursor: pointer;
    transition: all 0.3s ease;
}

.upload-zone:hover {
    border-color: #ff6600;
    background: rgba(255, 102, 0, 0.05);
}

.upload-icon {
    font-size: 3rem;
    margin-bottom: 1rem;
}

.upload-zone p {
    color: #e2e8f0;
    margin: 0 0 0.5rem 0;
    font-weight: 600;
}

.upload-zone small {
    color: #94a3b8;
    font-size: 0.8rem;
}

.form-actions {
    text-align: center;
    margin-top: 2rem;
}

.btn-submit {
    background: linear-gradient(135deg, #ff6600, #e55a00);
    color: #ffffff;
    border: none;
    padding: 1rem 3rem;
    border-radius: 8px;
    font-size: 1.1rem;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    margin: 0 auto 1rem auto;
}

.btn-submit:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 32px rgba(255, 102, 0, 0.3);
}

.btn-primary, .btn-secondary {
    padding: 0.75rem 2rem;
    border-radius: 8px;
    text-decoration: none;
    font-weight: 600;
    transition: all 0.3s ease;
    display: inline-block;
    border: 1px solid;
}

.btn-primary {
    background: rgba(255, 102, 0, 0.2);
    color: #ff6600;
    border-color: #ff6600;
}

.btn-primary:hover {
    background: #ff6600;
    color: #1a1a2e;
}

.btn-secondary {
    background: rgba(255, 255, 255, 0.1);
    color: #e2e8f0;
    border-color: #e2e8f0;
}

.btn-secondary:hover {
    background: #e2e8f0;
    color: #1a1a2e;
}

.terms-text {
    color: #94a3b8;
    font-size: 0.9rem;
    margin: 0;
}

.terms-text a {
    color: #ff6600;
    text-decoration: none;
}

.terms-text a:hover {
    text-decoration: underline;
}

@media (max-width: 768px) {
    .form-row {
        grid-template-columns: 1fr;
    }
    
    .upload-grid {
        grid-template-columns: 1fr;
    }
    
    .benefits-grid {
        grid-template-columns: 1fr;
    }
    
    .kyc-header h1 {
        font-size: 2rem;
    }
    
    .btn-submit {
        width: 100%;
        justify-content: center;
    }
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Handle file uploads
    const fileInputs = document.querySelectorAll('input[type="file"]');
    fileInputs.forEach(input => {
        input.addEventListener('change', function() {
            const fileName = this.files[0]?.name;
            const uploadZone = this.parentElement.querySelector('.upload-zone p');
            if (fileName) {
                uploadZone.textContent = `Selected: ${fileName}`;
                uploadZone.style.color = '#22c55e';
            }
        });
    });
});
</script>
@endsection