@extends('layouts.app')

@section('title', 'Leaderboards')

@section('content')
<div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
    <!-- Header -->
    <div class="text-center mb-8">
        <h1 class="text-4xl font-gaming font-bold text-cod-green gaming-text-glow mb-4">Leaderboards</h1>
        <p class="text-gray-400">See where you rank among the gaming elite</p>
    </div>

    <!-- Time Period Tabs -->
    <div class="bg-cod-gray rounded-lg p-6 mb-8">
        <h2 class="text-lg font-semibold text-gray-100 mb-4">Time Period</h2>
        <div class="flex flex-wrap gap-2 mb-6">
            <a href="{{ route('leaderboards.index', array_merge(request()->query(), ['period' => 'daily'])) }}" 
               class="px-4 py-2 rounded-lg font-semibold transition-colors {{ $period === 'daily' ? 'bg-cod-green text-black' : 'bg-cod-dark text-gray-300 hover:bg-cod-light' }}">
                Daily
            </a>
            <a href="{{ route('leaderboards.index', array_merge(request()->query(), ['period' => 'weekly'])) }}" 
               class="px-4 py-2 rounded-lg font-semibold transition-colors {{ $period === 'weekly' ? 'bg-cod-green text-black' : 'bg-cod-dark text-gray-300 hover:bg-cod-light' }}">
                Weekly
            </a>
            <a href="{{ route('leaderboards.index', array_merge(request()->query(), ['period' => 'monthly'])) }}" 
               class="px-4 py-2 rounded-lg font-semibold transition-colors {{ $period === 'monthly' ? 'bg-cod-green text-black' : 'bg-cod-dark text-gray-300 hover:bg-cod-light' }}">
                Monthly
            </a>
            <a href="{{ route('leaderboards.index', array_merge(request()->query(), ['period' => 'all_time'])) }}" 
               class="px-4 py-2 rounded-lg font-semibold transition-colors {{ $period === 'all_time' ? 'bg-cod-green text-black' : 'bg-cod-dark text-gray-300 hover:bg-cod-light' }}">
                All Time
            </a>
        </div>
        
        <!-- Leaderboard Type Tabs -->
        <h2 class="text-lg font-semibold text-gray-100 mb-4">Ranking Type</h2>
        <div class="flex flex-wrap gap-2">
            <a href="{{ route('leaderboards.index', array_merge(request()->query(), ['type' => 'wins'])) }}" 
               class="px-4 py-2 rounded-lg font-semibold transition-colors {{ $type === 'wins' ? 'bg-cod-green text-black' : 'bg-cod-dark text-gray-300 hover:bg-cod-light' }}">
                Most Wins
            </a>
            <a href="{{ route('leaderboards.index', array_merge(request()->query(), ['type' => 'winnings'])) }}" 
               class="px-4 py-2 rounded-lg font-semibold transition-colors {{ $type === 'winnings' ? 'bg-cod-green text-black' : 'bg-cod-dark text-gray-300 hover:bg-cod-light' }}">
                Amount Won
            </a>
            <a href="{{ route('leaderboards.index', array_merge(request()->query(), ['type' => 'xp'])) }}" 
               class="px-4 py-2 rounded-lg font-semibold transition-colors {{ $type === 'xp' ? 'bg-cod-green text-black' : 'bg-cod-dark text-gray-300 hover:bg-cod-light' }}">
                Experience Points
            </a>
        </div>
    </div>

    <!-- Current Period Display -->
    <div class="bg-cod-gray rounded-lg p-4 mb-6">
        <div class="text-center">
            <h3 class="text-xl font-semibold text-cod-green">
                @if($period === 'daily')
                    Today's Leaderboard
                    <span class="text-sm text-gray-400 block">{{ now()->format('F j, Y') }}</span>
                @elseif($period === 'weekly')
                    This Week's Leaderboard
                    <span class="text-sm text-gray-400 block">{{ now()->startOfWeek()->format('M j') }} - {{ now()->endOfWeek()->format('M j, Y') }}</span>
                @elseif($period === 'monthly')
                    This Month's Leaderboard
                    <span class="text-sm text-gray-400 block">{{ now()->format('F Y') }}</span>
                @else
                    All-Time Leaderboard
                    <span class="text-sm text-gray-400 block">Since the beginning</span>
                @endif
            </h3>
        </div>
    </div>

    <!-- Leaderboard -->
    <div class="bg-cod-gray rounded-lg overflow-hidden gaming-glow">
        @if($users->count() > 0)
            <div class="p-6">
                <h2 class="text-2xl font-gaming font-bold text-cod-green mb-6">
                    Top Players
                    @if($type === 'wins')
                        by Wins
                    @elseif($type === 'winnings')
                        by Amount Won
                    @else
                        by Experience Points
                    @endif
                </h2>
                <div class="space-y-4">
                    @foreach($users as $index => $user)
                        <div class="flex items-center justify-between p-4 bg-cod-dark rounded-lg hover:bg-cod-light transition-colors">
                            <div class="flex items-center space-x-4">
                                <!-- Rank -->
                                <div class="flex items-center justify-center w-12 h-12 
                                    @if($index === 0) bg-yellow-500 
                                    @elseif($index === 1) bg-gray-400 
                                    @elseif($index === 2) bg-green-600 
                                    @else bg-cod-green @endif 
                                    text-black font-bold rounded-full text-lg">
                                    {{ $index + 1 }}
                                </div>
                                
                                <!-- User Info -->
                                <div class="flex items-center space-x-3">
                                    @if($user->avatar_url)
                                        <img src="{{ $user->avatar_url }}" alt="{{ $user->username }}" class="w-10 h-10 rounded-full">
                                    @else
                                        <div class="w-10 h-10 bg-cod-green rounded-full flex items-center justify-center text-black font-bold">
                                            {{ strtoupper(substr($user->username, 0, 1)) }}
                                        </div>
                                    @endif
                                    <div>
                                        <h3 class="font-semibold text-gray-100 text-lg">{{ $user->username }}</h3>
                                        <div class="flex items-center space-x-4 text-sm text-gray-400">
                                            <span>Level {{ $user->level }}</span>
                                            <span>{{ number_format($user->xp) }} XP</span>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Score based on type -->
                            <div class="text-right">
                                @if($type === 'xp')
                                    <div class="text-2xl font-bold text-cod-green">{{ number_format($user->xp) }}</div>
                                    <div class="text-xs text-gray-400">Experience Points</div>
                                @elseif($type === 'winnings')
                                    <div class="text-2xl font-bold text-green-400">₦{{ number_format($user->total_winnings ?? 0, 2) }}</div>
                                    <div class="text-xs text-gray-400">Total Won</div>
                                @else
                                    <div class="text-2xl font-bold text-cod-green">{{ $user->total_wins ?? 0 }}</div>
                                    <div class="text-xs text-gray-400">{{ $user->total_wins == 1 ? 'Win' : 'Wins' }}</div>
                                @endif
                            </div>
                        </div>
                    @endforeach
                </div>
            </div>
        @else
            <div class="text-center py-16">
                <div class="text-6xl mb-4">🏆</div>
                <h3 class="text-xl font-semibold text-gray-400 mb-2">
                    @if($period === 'daily')
                        No Winners Today Yet
                    @elseif($period === 'weekly')
                        No Winners This Week Yet
                    @elseif($period === 'monthly')
                        No Winners This Month Yet
                    @else
                        No Rankings Yet
                    @endif
                </h3>
                <p class="text-gray-500 mb-4">
                    @if($type === 'wins')
                        Be the first to win a challenge
                        @if($period !== 'all_time')
                            {{ $period === 'daily' ? 'today' : ($period === 'weekly' ? 'this week' : 'this month') }}!
                        @else
                            and claim your spot!
                        @endif
                    @elseif($type === 'winnings')
                        Be the first to earn winnings
                        @if($period !== 'all_time')
                            {{ $period === 'daily' ? 'today' : ($period === 'weekly' ? 'this week' : 'this month') }}!
                        @else
                            and claim your spot!
                        @endif
                    @else
                        Start playing to appear on the leaderboard!
                    @endif
                </p>
                @guest
                    <a href="{{ route('register') }}" class="inline-block mt-4 bg-cod-green text-black px-6 py-3 rounded-lg font-semibold hover:bg-green-600 transition-colors">
                        Join the Competition
                    </a>
                @else
                    <a href="{{ route('challenges.index') }}" class="inline-block mt-4 bg-cod-green text-black px-6 py-3 rounded-lg font-semibold hover:bg-green-600 transition-colors">
                        Browse Challenges
                    </a>
                @endguest
            </div>
        @endif
    </div>

    @auth
        <!-- User's Current Rank -->
        <div class="mt-8 bg-cod-gray rounded-lg p-6">
            <h3 class="text-lg font-semibold text-cod-green mb-4">Your Current Standing</h3>
            <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
                <div class="text-center">
                    <div class="text-sm text-gray-400">Your Rank</div>
                    <div class="text-2xl font-bold text-gray-100">#{{ auth()->user()->getRank() ?? 'Unranked' }}</div>
                </div>
                
                @if($type === 'xp')
                    <div class="text-center">
                        <div class="text-sm text-gray-400">Experience Points</div>
                        <div class="text-2xl font-bold text-cod-green">{{ number_format(auth()->user()->xp) }}</div>
                    </div>
                    <div class="text-center">
                        <div class="text-sm text-gray-400">Level</div>
                        <div class="text-2xl font-bold text-cod-green">{{ auth()->user()->level }}</div>
                    </div>
                @elseif($type === 'wins')
                    @php
                        $userWins = auth()->user()->createdChallenges()
                            ->where('result', 'creator_win')
                            ->where('status', 'completed')
                            ->count() +
                        auth()->user()->acceptedChallenges()
                            ->where('result', 'accepter_win')
                            ->where('status', 'completed')
                            ->count();
                    @endphp
                    <div class="text-center">
                        <div class="text-sm text-gray-400">Total Wins</div>
                        <div class="text-2xl font-bold text-cod-green">{{ $userWins }}</div>
                    </div>
                    <div class="text-center">
                        <div class="text-sm text-gray-400">Level</div>
                        <div class="text-2xl font-bold text-cod-green">{{ auth()->user()->level }}</div>
                    </div>
                @else
                    @php
                        $userWinnings = auth()->user()->walletTransactions()
                            ->where('type', 'prize')
                            ->where('status', 'completed')
                            ->sum('amount');
                    @endphp
                    <div class="text-center">
                        <div class="text-sm text-gray-400">Total Winnings</div>
                        <div class="text-2xl font-bold text-green-400">₦{{ number_format($userWinnings, 2) }}</div>
                    </div>
                    <div class="text-center">
                        <div class="text-sm text-gray-400">Level</div>
                        <div class="text-2xl font-bold text-cod-green">{{ auth()->user()->level }}</div>
                    </div>
                @endif
            </div>
        </div>
    @endauth

    <!-- Auto-refresh notice -->
    <div class="mt-6 text-center">
        <p class="text-sm text-gray-500">
            Leaderboards update automatically as new games are completed.
            <span id="last-update" class="text-cod-green ml-2"></span>
            <a href="{{ request()->fullUrl() }}" class="text-cod-green hover:underline ml-2">Refresh now</a>
        </p>
    </div>
</div>

<!-- Efficient auto-refresh script using AJAX -->
<script>
    let isUpdating = false;
    
    // Function to update leaderboard via AJAX
    function updateLeaderboard() {
        if (isUpdating || document.hidden) return;
        
        isUpdating = true;
        
        fetch(window.location.href, {
            headers: {
                'X-Requested-With': 'XMLHttpRequest',
                'Accept': 'application/json',
            }
        })
        .then(response => response.json())
        .then(data => {
            updateLeaderboardData(data);
            updateLastUpdateTime(data.timestamp);
        })
        .catch(error => {
            console.error('Error updating leaderboard:', error);
            // Fallback to page reload only on error
            if (!document.hidden) {
                window.location.reload();
            }
        })
        .finally(() => {
            isUpdating = false;
        });
    }
    
    // Function to update the leaderboard display
    function updateLeaderboardData(data) {
        const leaderboardContainer = document.querySelector('.space-y-4');
        if (!leaderboardContainer || !data.users.length) return;
        
        let newContent = '';
        data.users.forEach((user, index) => {
            const rankColors = index === 0 ? 'bg-yellow-500' : 
                              index === 1 ? 'bg-gray-400' : 
                              index === 2 ? 'bg-green-600' : 'bg-cod-green';
            
            let scoreContent = '';
            if (data.type === 'xp') {
                scoreContent = `
                    <div class="text-2xl font-bold text-cod-green">${parseInt(user.xp).toLocaleString()}</div>
                    <div class="text-xs text-gray-400">Experience Points</div>
                `;
            } else if (data.type === 'winnings') {
                scoreContent = `
                    <div class="text-2xl font-bold text-green-400">₦${parseFloat(user.total_winnings || 0).toLocaleString('en-US', {minimumFractionDigits: 2})}</div>
                    <div class="text-xs text-gray-400">Total Won</div>
                `;
            } else {
                const wins = user.total_wins || 0;
                scoreContent = `
                    <div class="text-2xl font-bold text-cod-green">${wins}</div>
                    <div class="text-xs text-gray-400">${wins === 1 ? 'Win' : 'Wins'}</div>
                `;
            }
            
            const avatarHtml = user.avatar_url ? 
                `<img src="${user.avatar_url}" alt="${user.username}" class="w-10 h-10 rounded-full">` :
                `<div class="w-10 h-10 bg-cod-green rounded-full flex items-center justify-center text-black font-bold">
                    ${user.username.charAt(0).toUpperCase()}
                </div>`;
            
            newContent += `
                <div class="flex items-center justify-between p-4 bg-cod-dark rounded-lg hover:bg-cod-light transition-colors">
                    <div class="flex items-center space-x-4">
                        <div class="flex items-center justify-center w-12 h-12 ${rankColors} text-black font-bold rounded-full text-lg">
                            ${index + 1}
                        </div>
                        <div class="flex items-center space-x-3">
                            ${avatarHtml}
                            <div>
                                <h3 class="font-semibold text-gray-100 text-lg">${user.username}</h3>
                                <div class="flex items-center space-x-4 text-sm text-gray-400">
                                    <span>Level ${user.level}</span>
                                    <span>${parseInt(user.xp).toLocaleString()} XP</span>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="text-right">
                        ${scoreContent}
                    </div>
                </div>
            `;
        });
        
        leaderboardContainer.innerHTML = newContent;
    }
    
    // Function to update last update time
    function updateLastUpdateTime(timestamp) {
        const lastUpdateElement = document.getElementById('last-update');
        if (lastUpdateElement) {
            lastUpdateElement.textContent = `Last updated: ${timestamp}`;
        }
    }
    
    // Auto-refresh every 30 seconds using AJAX
    setInterval(updateLeaderboard, 30000);
    
    // Update when page becomes visible again
    document.addEventListener('visibilitychange', function() {
        if (!document.hidden) {
            setTimeout(updateLeaderboard, 1000);
        }
    });
    
    // Initial timestamp display
    updateLastUpdateTime(new Date().toLocaleTimeString('en-US', {hour12: false}));
</script>
@endsection