@extends('layouts.app')

@section('title', 'Notifications')

@section('content')
<div class="notification-center">
    <div class="container">
        <!-- Header -->
        <div class="notification-header">
            <div class="header-content">
                <h1>🔔 Notification Center</h1>
                <p>Stay updated with the latest news and announcements</p>
            </div>
            <div class="header-actions">
                @if($notifications->where('read_at', null)->count() > 0)
                    <form method="POST" action="{{ route('notifications.mark-all-read') }}" style="display: inline;">
                        @csrf
                        <button type="submit" class="btn-secondary">
                            ✅ Mark All as Read
                        </button>
                    </form>
                @endif
            </div>
        </div>

        <!-- Notifications List -->
        @if($notifications->count() > 0)
            <div class="notifications-list">
                @foreach($notifications as $notification)
                    <div class="notification-item {{ $notification->read_at ? 'read' : 'unread' }}">
                        <div class="notification-icon">
                            @switch($notification->type)
                                @case('challenge_accepted')
                                    ⚔️
                                    @break
                                @case('challenge_result')
                                    🏆
                                    @break
                                @case('tournament_invite')
                                    🎮
                                    @break
                                @case('wallet_deposit')
                                    💰
                                    @break
                                @case('wallet_withdrawal')
                                    💸
                                    @break
                                @case('dispute_resolved')
                                    ⚖️
                                    @break
                                @case('level_up')
                                    📈
                                    @break
                                @case('badge_earned')
                                    🏅
                                    @break
                                @default
                                    📢
                            @endswitch
                        </div>

                        <div class="notification-content">
                            <div class="notification-header-row">
                                <h3 class="notification-title">{{ $notification->title }}</h3>
                                <div class="notification-meta">
                                    <span class="notification-time">{{ $notification->created_at->diffForHumans() }}</span>
                                    @if(!$notification->read_at)
                                        <span class="unread-badge">New</span>
                                    @endif
                                </div>
                            </div>

                            <p class="notification-body">{{ $notification->body }}</p>

                            @if($notification->data && isset($notification->data['action_url']))
                                <div class="notification-actions">
                                    <a href="{{ $notification->data['action_url'] }}" class="action-link">
                                        {{ $notification->data['action_text'] ?? 'View Details' }} →
                                    </a>
                                </div>
                            @endif
                        </div>

                        @if(!$notification->read_at)
                            <div class="notification-controls">
                                <button onclick="markAsRead({{ $notification->id }})" class="mark-read-btn" title="Mark as read">
                                    ✓
                                </button>
                            </div>
                        @endif
                    </div>
                @endforeach
            </div>

            <!-- Pagination -->
            <div class="pagination-wrapper">
                {{ $notifications->links() }}
            </div>
        @else
            <div class="empty-state">
                <div class="empty-icon">🔔</div>
                <h3>No notifications yet</h3>
                <p>When you receive notifications about challenges, tournaments, or other activities, they'll appear here.</p>
                <a href="{{ route('challenges.index') }}" class="btn-primary">Browse Challenges</a>
            </div>
        @endif
    </div>
</div>

<!-- Popup Notification Modal -->
<div id="popup-modal" class="popup-modal" style="display: none;">
    <div class="popup-overlay"></div>
    <div class="popup-content">
        <div class="popup-header">
            <span class="popup-icon"></span>
            <h3 class="popup-title"></h3>
            <button class="popup-close" onclick="closePopup()">&times;</button>
        </div>
        <div class="popup-body">
            <p class="popup-message"></p>
        </div>
        <div class="popup-actions">
            <button onclick="closePopup()" class="btn-primary">Got it!</button>
        </div>
    </div>
</div>

<script>
// Mark notification as read
async function markAsRead(notificationId) {
    try {
        const response = await fetch(`/notifications/${notificationId}/read`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            }
        });

        if (response.ok) {
            location.reload();
        }
    } catch (error) {
        console.error('Error marking notification as read:', error);
    }
}

// Popup notification system
let currentPopupId = null;

async function checkForPopups() {
    try {
        const response = await fetch('/notifications/popups');
        const popups = await response.json();

        if (popups.length > 0) {
            showPopup(popups[0]);
        }
    } catch (error) {
        console.error('Error checking for popups:', error);
    }
}

function showPopup(popup) {
    currentPopupId = popup.id;
    
    document.querySelector('.popup-icon').textContent = getPopupIcon(popup.type);
    document.querySelector('.popup-title').textContent = popup.title;
    document.querySelector('.popup-message').textContent = popup.message;
    
    const modal = document.getElementById('popup-modal');
    modal.style.display = 'flex';
    
    // Add animation
    setTimeout(() => {
        modal.classList.add('show');
    }, 10);
}

function closePopup() {
    const modal = document.getElementById('popup-modal');
    modal.classList.remove('show');
    
    setTimeout(() => {
        modal.style.display = 'none';
    }, 300);

    // Mark popup as seen
    if (currentPopupId) {
        markPopupSeen(currentPopupId);
    }
}

async function markPopupSeen(popupId) {
    try {
        await fetch(`/notifications/popups/${popupId}/seen`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            }
        });
    } catch (error) {
        console.error('Error marking popup as seen:', error);
    }
}

function getPopupIcon(type) {
    const icons = {
        'info': '📢',
        'warning': '⚠️',
        'success': '✅',
        'announcement': '🎉'
    };
    return icons[type] || '📢';
}

// Check for popups when page loads
document.addEventListener('DOMContentLoaded', function() {
    checkForPopups();
});
</script>

<style>
.notification-center {
    padding: 2rem 0 4rem;
    background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
    min-height: 100vh;
    color: #e2e8f0;
}

.notification-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 2rem;
    padding: 1.5rem;
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 102, 0, 0.2);
    border-radius: 12px;
}

.header-content h1 {
    border: 1px solid rgba(57, 255, 20, 0.2);
    border-radius: 12px;
}

.header-content h1 {
    color: #39ff14;
    margin: 0 0 0.5rem 0;
    font-size: 2rem;
}

.header-content p {
    color: #94a3b8;
    margin: 0;
}

.btn-secondary {
    background: rgba(255, 102, 0, 0.2);
    color: #ff6600;
    border: 1px solid #ff6600;
    padding: 0.75rem 1.5rem;
    border-radius: 8px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
}

.btn-secondary:hover {
    background: #ff6600;
    color: #1a1a2e;
}

.notifications-list {
    display: flex;
    flex-direction: column;
    gap: 1rem;
    margin-bottom: 3rem;
}

.notification-item {
    display: flex;
    align-items: flex-start;
    gap: 1rem;
    padding: 1.5rem;
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 255, 255, 0.1);
    border-radius: 12px;
    transition: all 0.3s ease;
    position: relative;
}

.notification-item.unread {
    border-color: rgba(255, 102, 0, 0.3);
    background: rgba(255, 102, 0, 0.05);
}

.notification-item:hover {
    border-color: rgba(255, 102, 0, 0.4);
    transform: translateY(-2px);
}

.notification-icon {
    font-size: 2rem;
    flex-shrink: 0;
    width: 60px;
    height: 60px;
    background: rgba(255, 102, 0, 0.1);
    border: 1px solid rgba(255, 102, 0, 0.3);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
}

.notification-content {
    flex: 1;
}

.notification-header-row {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    margin-bottom: 0.75rem;
    gap: 1rem;
}

.notification-title {
    color: #e2e8f0;
    margin: 0;
    font-size: 1.1rem;
    font-weight: 600;
}

.notification-meta {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    flex-shrink: 0;
}

.notification-time {
    color: #94a3b8;
    font-size: 0.9rem;
}

.unread-badge {
    background: linear-gradient(135deg, #ff6600, #cc5500);
    color: white;
    padding: 0.25rem 0.75rem;
    border-radius: 15px;
    font-size: 0.8rem;
    font-weight: 600;
}

.notification-body {
    color: #94a3b8;
    line-height: 1.6;
    margin: 0 0 1rem 0;
}

.notification-actions {
    margin-top: 1rem;
}

.action-link {
    color: #ff6600;
    text-decoration: none;
    font-weight: 600;
    transition: color 0.3s ease;
}

.action-link:hover {
    color: #cc5500;
}

.notification-controls {
    position: absolute;
    top: 1rem;
    right: 1rem;
}

.mark-read-btn {
    width: 32px;
    height: 32px;
    background: rgba(34, 197, 94, 0.2);
    color: #22c55e;
    border: 1px solid #22c55e;
    border-radius: 50%;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: 700;
    transition: all 0.3s ease;
}

.mark-read-btn:hover {
    background: #22c55e;
    color: white;
}

.empty-state {
    text-align: center;
    padding: 4rem 2rem;
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 255, 255, 0.1);
    border-radius: 16px;
}

.empty-icon {
    font-size: 4rem;
    margin-bottom: 1.5rem;
    opacity: 0.7;
}

.empty-state h3 {
    color: #e2e8f0;
    margin-bottom: 1rem;
    font-size: 1.5rem;
}

.empty-state p {
    color: #94a3b8;
    margin-bottom: 2rem;
    font-size: 1.1rem;
}

.btn-primary {
    background: linear-gradient(135deg, #ff6600, #cc5500);
    color: white;
    border: none;
    padding: 0.75rem 2rem;
    border-radius: 25px;
    text-decoration: none;
    font-weight: 600;
    transition: all 0.3s ease;
    display: inline-block;
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(255, 102, 0, 0.3);
}

.pagination-wrapper {
    display: flex;
    justify-content: center;
}

/* Popup Modal Styles */
.popup-modal {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    z-index: 9999;
    display: flex;
    align-items: center;
    justify-content: center;
    opacity: 0;
    transition: opacity 0.3s ease;
}

.popup-modal.show {
    opacity: 1;
}

.popup-overlay {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.8);
    backdrop-filter: blur(5px);
}

.popup-content {
    position: relative;
    background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
    border: 2px solid rgba(255, 102, 0, 0.3);
    border-radius: 16px;
    max-width: 500px;
    width: 90%;
    max-height: 80vh;
    overflow: hidden;
    transform: scale(0.8);
    transition: transform 0.3s ease;
}

.popup-modal.show .popup-content {
    transform: scale(1);
}

.popup-header {
    display: flex;
    align-items: center;
    gap: 1rem;
    padding: 1.5rem;
    background: rgba(255, 102, 0, 0.1);
    border-bottom: 1px solid rgba(255, 102, 0, 0.2);
}

.popup-icon {
    font-size: 1.5rem;
}

.popup-title {
    color: #ff6600;
    margin: 0;
    font-size: 1.3rem;
    font-weight: 600;
    flex: 1;
}

.popup-close {
    background: none;
    border: none;
    color: #94a3b8;
    font-size: 1.5rem;
    cursor: pointer;
    padding: 0;
    width: 32px;
    height: 32px;
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: 50%;
    transition: all 0.3s ease;
}

.popup-close:hover {
    background: rgba(255, 255, 255, 0.1);
    color: #e2e8f0;
}

.popup-body {
    padding: 1.5rem;
}

.popup-message {
    color: #e2e8f0;
    line-height: 1.6;
    margin: 0;
    font-size: 1.1rem;
}

.popup-actions {
    padding: 1.5rem;
    border-top: 1px solid rgba(255, 255, 255, 0.1);
    text-align: center;
}

@media (max-width: 768px) {
    .notification-header {
        flex-direction: column;
        gap: 1rem;
        text-align: center;
    }
    
    .notification-header-row {
        flex-direction: column;
        align-items: flex-start;
        gap: 0.5rem;
    }
    
    .notification-controls {
        position: static;
        margin-top: 1rem;
    }
    
    .popup-content {
        width: 95%;
        margin: 1rem;
    }
}
</style>
@endsection