@extends('layouts.app')

@section('title', 'News & Updates')

@section('content')
<div class="blog-index">
    <div class="container">
        <!-- Header Section -->
        <div class="blog-header">
            <div class="header-content">
                <h1>📰 News & Updates</h1>
                <p>Stay updated with the latest news, tournament highlights, and platform updates</p>
            </div>
        </div>

        <!-- Filters -->
        <div class="blog-filters">
            <div class="filter-tabs">
                <a href="{{ route('posts.index') }}" class="filter-tab {{ !request('type') ? 'active' : '' }}">
                    📋 All Posts
                </a>
                @foreach($postTypes as $typeKey => $typeLabel)
                    <a href="{{ route('posts.index', ['type' => $typeKey]) }}" class="filter-tab {{ request('type') === $typeKey ? 'active' : '' }}">
                        @switch($typeKey)
                            @case('blog')
                                📖 {{ $typeLabel }}
                                @break
                            @case('news')
                                📰 {{ $typeLabel }}
                                @break
                            @case('highlight')
                                🏆 {{ $typeLabel }}
                                @break
                            @case('tournament')
                                🎮 {{ $typeLabel }}
                                @break
                            @case('update')
                                🔄 {{ $typeLabel }}
                                @break
                        @endswitch
                    </a>
                @endforeach
            </div>

            <div class="search-box">
                <form method="GET" action="{{ route('posts.index') }}">
                    @if(request('type'))
                        <input type="hidden" name="type" value="{{ request('type') }}">
                    @endif
                    <input type="text" name="search" placeholder="Search posts..." value="{{ request('search') }}">
                    <button type="submit">🔍</button>
                </form>
            </div>
        </div>

        <!-- Posts Grid -->
        @if($posts->count() > 0)
            <div class="posts-grid">
                @foreach($posts as $post)
                    <article class="post-card">
                        @if($post->featured_image)
                            <div class="post-image">
                                <img src="{{ asset('storage/' . $post->featured_image) }}" alt="{{ $post->title }}">
                                <div class="post-type-badge">
                                    @switch($post->type)
                                        @case('blog')
                                            📖
                                            @break
                                        @case('news')
                                            📰
                                            @break
                                        @case('highlight')
                                            🏆
                                            @break
                                        @case('tournament')
                                            🎮
                                            @break
                                        @case('update')
                                            🔄
                                            @break
                                    @endswitch
                                </div>
                            </div>
                        @endif

                        <div class="post-content">
                            <div class="post-meta">
                                <span class="post-type type-{{ $post->type }}">{{ ucfirst($post->type) }}</span>
                                <span class="post-date">{{ $post->published_at->format('M j, Y') }}</span>
                            </div>

                            <h2 class="post-title">
                                <a href="{{ route('posts.show', $post->slug) }}">{{ $post->title }}</a>
                            </h2>

                            <p class="post-excerpt">
                                {{ $post->excerpt }}
                            </p>

                            <div class="post-footer">
                                <div class="post-author">
                                    <span>By {{ $post->author->username }}</span>
                                </div>
                                <div class="post-stats">
                                    <span class="views">👁️ {{ number_format($post->views) }}</span>
                                    <span class="reading-time">📖 {{ $post->reading_time }} min read</span>
                                </div>
                            </div>

                            <a href="{{ route('posts.show', $post->slug) }}" class="read-more">
                                Read More →
                            </a>
                        </div>
                    </article>
                @endforeach
            </div>

            <!-- Pagination -->
            <div class="pagination-wrapper">
                {{ $posts->appends(request()->query())->links() }}
            </div>
        @else
            <div class="empty-state">
                <div class="empty-icon">📰</div>
                <h3>No posts found</h3>
                @if(request('search'))
                    <p>No posts found for "{{ request('search') }}". Try adjusting your search terms.</p>
                    <a href="{{ route('posts.index') }}" class="btn-primary">View All Posts</a>
                @elseif(request('type'))
                    <p>No {{ strtolower($postTypes[request('type')]) }} posts available yet.</p>
                    <a href="{{ route('posts.index') }}" class="btn-primary">View All Posts</a>
                @else
                    <p>No posts have been published yet. Check back soon for the latest updates!</p>
                @endif
            </div>
        @endif
    </div>
</div>

<style>
.blog-index {
    padding: 2rem 0 4rem;
    background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
    min-height: 100vh;
    color: #e2e8f0;
}

.blog-header {
    text-align: center;
    margin-bottom: 3rem;
    padding: 2rem;
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 102, 0, 0.2);
    border-radius: 16px;
}

.header-content h1 {
    border: 1px solid rgba(57, 255, 20, 0.2);
    border-radius: 16px;
}

.header-content h1 {
    color: #39ff14;
    font-size: 2.5rem;
    margin-bottom: 1rem;
    font-weight: 700;
}

.header-content p {
    color: #94a3b8;
    font-size: 1.1rem;
    max-width: 600px;
    margin: 0 auto;
}

.blog-filters {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 3rem;
    gap: 2rem;
    flex-wrap: wrap;
}

.filter-tabs {
    display: flex;
    gap: 0.5rem;
    flex-wrap: wrap;
}

.filter-tab {
    padding: 0.75rem 1.25rem;
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 255, 255, 0.1);
    border-radius: 25px;
    color: #94a3b8;
    text-decoration: none;
    font-weight: 500;
    font-size: 0.9rem;
    transition: all 0.3s ease;
    white-space: nowrap;
}

.filter-tab:hover {
    border-color: rgba(255, 102, 0, 0.3);
    color: #ff6600;
}

.filter-tab.active {
    background: linear-gradient(135deg, #ff6600, #cc5500);
    border-color: #ff6600;
    color: white;
}

.search-box {
    display: flex;
    align-items: center;
}

.search-box form {
    display: flex;
    gap: 0.5rem;
}

.search-box input {
    background: rgba(255, 255, 255, 0.1);
    border: 1px solid rgba(255, 255, 255, 0.2);
    border-radius: 25px;
    padding: 0.75rem 1.25rem;
    color: #e2e8f0;
    font-size: 0.9rem;
    min-width: 250px;
}

.search-box input:focus {
    outline: none;
    border-color: #ff6600;
    box-shadow: 0 0 0 3px rgba(255, 102, 0, 0.1);
}

.search-box button {
    background: linear-gradient(135deg, #ff6600, #cc5500);
    border: none;
    border-radius: 50%;
    width: 44px;
    height: 44px;
    color: white;
    cursor: pointer;
    transition: all 0.3s ease;
    display: flex;
    align-items: center;
    justify-content: center;
}

.search-box button:hover {
    transform: scale(1.05);
    box-shadow: 0 4px 15px rgba(255, 102, 0, 0.3);
}

.posts-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(350px, 1fr));
    gap: 2rem;
    margin-bottom: 3rem;
}

.post-card {
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 255, 255, 0.1);
    border-radius: 16px;
    overflow: hidden;
    transition: all 0.3s ease;
    position: relative;
}

.post-card:hover {
    transform: translateY(-5px);
    border-color: rgba(255, 102, 0, 0.3);
    box-shadow: 0 15px 35px rgba(255, 102, 0, 0.1);
}

.post-image {
    position: relative;
    aspect-ratio: 16/9;
    overflow: hidden;
}

.post-image img {
    width: 100%;
    height: 100%;
    object-fit: cover;
    transition: transform 0.3s ease;
}

.post-card:hover .post-image img {
    transform: scale(1.05);
}

.post-type-badge {
    position: absolute;
    top: 1rem;
    right: 1rem;
    background: rgba(0, 0, 0, 0.8);
    backdrop-filter: blur(10px);
    padding: 0.5rem;
    border-radius: 50%;
    font-size: 1.2rem;
}

.post-content {
    padding: 1.5rem;
}

.post-meta {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1rem;
}

.post-type {
    padding: 0.25rem 0.75rem;
    border-radius: 15px;
    font-size: 0.8rem;
    font-weight: 600;
    text-transform: uppercase;
}

.type-blog { background: rgba(59, 130, 246, 0.2); color: #3b82f6; }
.type-news { background: rgba(16, 185, 129, 0.2); color: #10b981; }
.type-highlight { background: rgba(245, 158, 11, 0.2); color: #f59e0b; }
.type-tournament { background: rgba(139, 92, 246, 0.2); color: #8b5cf6; }
.type-update { background: rgba(236, 72, 153, 0.2); color: #ec4899; }

.post-date {
    color: #94a3b8;
    font-size: 0.9rem;
}

.post-title {
    margin: 0 0 1rem 0;
}

.post-title a {
    color: #e2e8f0;
    text-decoration: none;
    font-size: 1.3rem;
    font-weight: 600;
    line-height: 1.4;
    transition: color 0.3s ease;
}

.post-title a:hover {
    color: #ff6600;
}

.post-excerpt {
    color: #94a3b8;
    line-height: 1.6;
    margin-bottom: 1.5rem;
}

.post-footer {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1rem;
    padding-top: 1rem;
    border-top: 1px solid rgba(255, 255, 255, 0.1);
}

.post-author {
    color: #94a3b8;
    font-size: 0.9rem;
}

.post-stats {
    display: flex;
    gap: 1rem;
    color: #94a3b8;
    font-size: 0.8rem;
}

.read-more {
    color: #ff6600;
    text-decoration: none;
    font-weight: 600;
    transition: all 0.3s ease;
}

.read-more:hover {
    color: #cc5500;
    transform: translateX(5px);
}

.empty-state {
    text-align: center;
    padding: 4rem 2rem;
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 255, 255, 0.1);
    border-radius: 16px;
}

.empty-icon {
    font-size: 4rem;
    margin-bottom: 1.5rem;
    opacity: 0.7;
}

.empty-state h3 {
    color: #e2e8f0;
    margin-bottom: 1rem;
    font-size: 1.5rem;
}

.empty-state p {
    color: #94a3b8;
    margin-bottom: 2rem;
    font-size: 1.1rem;
}

.btn-primary {
    background: linear-gradient(135deg, #ff6600, #cc5500);
    color: white;
    border: none;
    padding: 0.75rem 2rem;
    border-radius: 25px;
    text-decoration: none;
    font-weight: 600;
    transition: all 0.3s ease;
    display: inline-block;
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(255, 102, 0, 0.3);
}

.pagination-wrapper {
    margin-top: 3rem;
    display: flex;
    justify-content: center;
}

@media (max-width: 768px) {
    .blog-filters {
        flex-direction: column;
        align-items: stretch;
        gap: 1rem;
    }
    
    .filter-tabs {
        justify-content: center;
    }
    
    .search-box input {
        min-width: 200px;
    }
    
    .posts-grid {
        grid-template-columns: 1fr;
    }
    
    .post-footer {
        flex-direction: column;
        gap: 1rem;
        align-items: flex-start;
    }
}
</style>
@endsection