@extends('layouts.app')

@section('title', 'My Tournaments')

@section('content')
<div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
    <!-- Header -->
    <div class="flex justify-between items-center mb-8">
        <h1 class="text-3xl font-gaming font-bold text-cod-green">🏆 My Tournaments</h1>
        @if(auth()->user()->canHostFreeTournaments())
            <a href="{{ route('tournaments.create') }}" class="bg-cod-green text-black px-6 py-3 rounded-lg font-semibold hover:bg-green-600 transition-colors">
                Host New Tournament
            </a>
        @endif
    </div>

    <!-- Status Filter Tabs -->
    <div class="flex space-x-1 mb-6">
        <a href="{{ route('tournaments.my', ['filter' => 'all'] + request()->except('filter')) }}" 
           class="px-6 py-3 rounded-lg font-semibold transition-colors {{ ($filter ?? 'all') === 'all' ? 'bg-cod-green text-black' : 'bg-cod-gray text-gray-300 hover:bg-cod-light' }}">
            📊 All Tournaments
        </a>
        <a href="{{ route('tournaments.my', ['filter' => 'ongoing'] + request()->except('filter')) }}" 
           class="px-6 py-3 rounded-lg font-semibold transition-colors {{ ($filter ?? 'all') === 'ongoing' ? 'bg-cod-green text-black' : 'bg-cod-gray text-gray-300 hover:bg-cod-light' }}">
            🔥 Ongoing
        </a>
        <a href="{{ route('tournaments.my', ['filter' => 'past'] + request()->except('filter')) }}" 
           class="px-6 py-3 rounded-lg font-semibold transition-colors {{ ($filter ?? 'all') === 'past' ? 'bg-cod-green text-black' : 'bg-cod-gray text-gray-300 hover:bg-cod-light' }}">
            📚 Past Tournaments
        </a>
    </div>

    <!-- Enhanced Filters -->
    <div class="bg-cod-gray rounded-lg p-6 mb-8">
        <form method="GET" action="{{ route('tournaments.my') }}">
            <input type="hidden" name="filter" value="{{ $filter ?? 'all' }}">
            <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                <select name="game_id" class="bg-cod-dark text-gray-100 border border-cod-light rounded-lg px-4 py-2">
                    <option value="">All Games</option>
                    @foreach($games as $game)
                        <option value="{{ $game->id }}" {{ request('game_id') == $game->id ? 'selected' : '' }}>
                            {{ $game->name }}
                        </option>
                    @endforeach
                </select>
                
                <button type="submit" class="bg-cod-green text-black px-6 py-2 rounded-lg font-semibold hover:bg-green-600 transition-colors">
                    🔍 Filter
                </button>
                
                <a href="{{ route('tournaments.my') }}" class="bg-cod-gray text-gray-300 px-6 py-2 rounded-lg font-semibold hover:bg-cod-light transition-colors text-center">
                    Clear Filters
                </a>
            </div>
        </form>
    </div>

    <!-- Tournaments Grid -->
    @if($tournaments->count() > 0)
        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            @foreach($tournaments as $tournament)
                <div class="bg-cod-gray rounded-lg p-6 border-l-4 {{ 
                    in_array($tournament->status, ['completed', 'cancelled']) ? 'border-blue-500' : 
                    ($tournament->status === 'ongoing' ? 'border-yellow-500' : 'border-purple-500') 
                }} hover:bg-cod-light transition-colors">
                    <!-- Tournament Header -->
                    <div class="flex justify-between items-start mb-4">
                        <div class="flex items-center space-x-2">
                            <img src="{{ $tournament->game->icon_url ?? asset('images/default-game.png') }}" 
                                 alt="{{ $tournament->game->name }}" 
                                 class="w-10 h-10 rounded-lg object-cover">
                            <div>
                                <h3 class="font-bold text-gray-100">{{ $tournament->title }}</h3>
                                <div class="text-sm text-gray-400">{{ $tournament->game->name }}</div>
                            </div>
                        </div>
                        <div class="text-right">
                            <div class="text-purple-400 font-bold text-lg">
                                @if($tournament->entry_fee > 0)
                                    ₦{{ number_format($tournament->entry_fee) }}
                                @else
                                    FREE
                                @endif
                            </div>
                            <div class="text-xs {{ 
                                in_array($tournament->status, ['completed', 'cancelled']) ? 'text-blue-400' : 
                                ($tournament->status === 'ongoing' ? 'text-yellow-400' : 'text-purple-400') 
                            }}">
                                {{ ucfirst($tournament->status) }}
                            </div>
                        </div>
                    </div>

                    <!-- Tournament Details -->
                    <div class="mb-4">
                        <div class="flex justify-between items-center text-sm mb-2">
                            <span class="text-gray-400">Host:</span>
                            <span class="text-gray-200 {{ $tournament->host_id === auth()->id() ? 'font-bold text-purple-400' : '' }}">
                                {{ $tournament->host->username }}
                                @if($tournament->host_id === auth()->id())
                                    (You)
                                @endif
                            </span>
                        </div>
                        
                        <div class="flex justify-between items-center text-sm mb-2">
                            <span class="text-gray-400">Participants:</span>
                            <span class="text-gray-200">{{ $tournament->participants->count() }}</span>
                        </div>

                        @if($tournament->prize_pool > 0)
                            <div class="flex justify-between items-center text-sm mb-2">
                                <span class="text-gray-400">Prize Pool:</span>
                                <span class="text-green-400 font-semibold">₦{{ number_format($tournament->prize_pool) }}</span>
                            </div>
                        @endif

                        @if($tournament->starts_at)
                            <div class="flex justify-between items-center text-sm">
                                <span class="text-gray-400">
                                    @if($tournament->status === 'upcoming')
                                        Starts:
                                    @elseif($tournament->status === 'ongoing')
                                        Started:
                                    @else
                                        Started:
                                    @endif
                                </span>
                                <span class="text-gray-200">{{ $tournament->starts_at->format('M j, Y g:i A') }}</span>
                            </div>
                        @endif
                    </div>

                    <!-- Tournament Type -->
                    <div class="mb-4">
                        <span class="inline-block px-3 py-1 text-xs bg-purple-500/20 text-purple-400 rounded-full">
                            {{ ucfirst($tournament->type) }} Tournament
                        </span>
                        
                        <!-- User's Role in Tournament -->
                        @if($tournament->host_id === auth()->id())
                            <span class="inline-block px-3 py-1 text-xs bg-yellow-500/20 text-yellow-400 rounded-full ml-2">
                                Host
                            </span>
                        @else
                            <span class="inline-block px-3 py-1 text-xs bg-blue-500/20 text-blue-400 rounded-full ml-2">
                                Participant
                            </span>
                        @endif
                    </div>

                    <!-- Action Button -->
                    <div class="mt-4">
                        <a href="{{ route('tournaments.show', $tournament) }}" 
                           class="block text-center bg-purple-600 text-white py-2 px-4 rounded-lg font-semibold hover:bg-purple-700 transition-colors">
                            View Tournament
                        </a>
                    </div>

                    <!-- Timestamp -->
                    <div class="mt-3 text-xs text-gray-500 text-center">
                        {{ $tournament->created_at->diffForHumans() }}
                    </div>
                </div>
            @endforeach
        </div>

        <!-- Pagination -->
        <div class="mt-8">
            {{ $tournaments->appends(request()->query())->links() }}
        </div>
    @else
        <!-- Empty State -->
        <div class="text-center py-16">
            <div class="w-24 h-24 bg-cod-gray rounded-full flex items-center justify-center mx-auto mb-6">
                <span class="text-4xl">🏆</span>
            </div>
            <h3 class="text-xl font-bold text-gray-300 mb-2">
                @if(($filter ?? 'all') === 'ongoing')
                    No Ongoing Tournaments
                @elseif(($filter ?? 'all') === 'past')
                    No Past Tournaments
                @else
                    No Tournaments Yet
                @endif
            </h3>
            <p class="text-gray-400 mb-6">
                @if(($filter ?? 'all') === 'ongoing')
                    You don't have any ongoing tournaments at the moment.
                @elseif(($filter ?? 'all') === 'past')
                    You haven't participated in any completed tournaments yet.
                @else
                    You haven't participated in any tournaments yet.
                @endif
            </p>
            @if(auth()->user()->canHostFreeTournaments())
                <a href="{{ route('tournaments.create') }}" class="bg-purple-600 text-white px-6 py-3 rounded-lg font-semibold hover:bg-purple-700 transition-colors">
                    Host Your First Tournament
                </a>
            @else
                <a href="{{ route('tournaments.index') }}" class="bg-purple-600 text-white px-6 py-3 rounded-lg font-semibold hover:bg-purple-700 transition-colors">
                    Browse Tournaments
                </a>
            @endif
        </div>
    @endif
</div>
@endsection