@extends('layouts.app')

@section('title', 'Transfer Credits')

@section('content')
<div class="max-w-2xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
    <!-- Header -->
    <div class="text-center mb-8">
        <h1 class="text-4xl font-gaming font-bold text-cod-green gaming-text-glow mb-4">Transfer Credits</h1>
        <p class="text-gray-400">Send credits to other players instantly</p>
    </div>

    <!-- Balance Overview -->
    <div class="bg-cod-gray rounded-lg p-6 mb-8 gaming-glow">
        <div class="text-center">
            <h3 class="text-lg font-semibold text-gray-300 mb-2">Available Balance</h3>
            <div class="text-3xl font-bold text-green-400 mb-2">₦{{ number_format(auth()->user()->wallet_balance) }}</div>
            <p class="text-sm text-gray-400">Ready to transfer</p>
        </div>
    </div>

    <!-- Transfer Form -->
    <div class="bg-cod-gray rounded-lg p-8 gaming-glow">
        <div class="mb-6">
            <div class="flex items-center justify-center mb-4">
                <div class="w-16 h-16 bg-cod-green rounded-full flex items-center justify-center">
                    <svg class="w-8 h-8 text-black" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 7h12m0 0l-4-4m4 4l-4 4m0 6H4m0 0l4 4m-4-4l4-4"></path>
                    </svg>
                </div>
            </div>
            <h2 class="text-xl font-semibold text-cod-green text-center mb-2">Send Credits</h2>
            <p class="text-gray-400 text-center">Transfer credits to another player's wallet</p>
        </div>

        <form action="{{ route('wallet.transfer.store') }}" method="POST" class="space-y-6" id="transferForm">
            @csrf

            <!-- Recipient Username -->
            <div>
                <label for="recipient_username" class="block text-sm font-medium text-gray-300 mb-2">
                    Recipient Username
                    <span class="text-red-400">*</span>
                </label>
                <div class="relative">
                    <input type="text" 
                           id="recipient_username" 
                           name="recipient_username" 
                           value="{{ old('recipient_username') }}"
                           required
                           class="w-full px-4 py-3 bg-cod-dark border border-cod-light rounded-lg text-gray-100 focus:ring-2 focus:ring-cod-green focus:border-transparent @error('recipient_username') border-red-500 @enderror"
                           placeholder="Enter username of recipient">
                    <div class="absolute inset-y-0 right-0 flex items-center pr-3">
                        <svg class="w-5 h-5 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"></path>
                        </svg>
                    </div>
                </div>
                @error('recipient_username')
                    <p class="mt-1 text-sm text-red-400">{{ $message }}</p>
                @enderror
                <p class="mt-1 text-sm text-gray-400">Enter the exact username of the player you want to send credits to</p>
            </div>

            <!-- Amount -->
            <div>
                <label for="amount" class="block text-sm font-medium text-gray-300 mb-2">
                    Amount
                    <span class="text-red-400">*</span>
                </label>
                <div class="relative">
                    <div class="absolute inset-y-0 left-0 flex items-center pl-3">
                        <span class="text-gray-400">₦</span>
                    </div>
                    <input type="number" 
                           id="amount" 
                           name="amount" 
                           value="{{ old('amount') }}"
                           min="1" 
                           max="100000" 
                           step="0.01"
                           required
                           class="w-full pl-8 pr-4 py-3 bg-cod-dark border border-cod-light rounded-lg text-gray-100 focus:ring-2 focus:ring-cod-green focus:border-transparent @error('amount') border-red-500 @enderror"
                           placeholder="0.00">
                </div>
                @error('amount')
                    <p class="mt-1 text-sm text-red-400">{{ $message }}</p>
                @enderror
                <div class="mt-1 flex justify-between text-sm text-gray-400">
                    <span>Min: ₦1.00</span>
                    <span>Max: ₦100,000.00</span>
                </div>

                <!-- Quick Amount Buttons -->
                <div class="mt-3 grid grid-cols-4 gap-2">
                    <button type="button" onclick="setAmount(100)" class="px-3 py-2 bg-cod-dark border border-cod-light rounded text-sm text-gray-300 hover:bg-cod-light transition-colors">₦100</button>
                    <button type="button" onclick="setAmount(500)" class="px-3 py-2 bg-cod-dark border border-cod-light rounded text-sm text-gray-300 hover:bg-cod-light transition-colors">₦500</button>
                    <button type="button" onclick="setAmount(1000)" class="px-3 py-2 bg-cod-dark border border-cod-light rounded text-sm text-gray-300 hover:bg-cod-light transition-colors">₦1,000</button>
                    <button type="button" onclick="setAmount(5000)" class="px-3 py-2 bg-cod-dark border border-cod-light rounded text-sm text-gray-300 hover:bg-cod-light transition-colors">₦5,000</button>
                </div>
            </div>

            <!-- Note (Optional) -->
            <div>
                <label for="note" class="block text-sm font-medium text-gray-300 mb-2">Note (Optional)</label>
                <textarea id="note" 
                          name="note" 
                          rows="3"
                          maxlength="255"
                          class="w-full px-4 py-3 bg-cod-dark border border-cod-light rounded-lg text-gray-100 focus:ring-2 focus:ring-cod-green focus:border-transparent @error('note') border-red-500 @enderror"
                          placeholder="Add a message with your transfer (optional)">{{ old('note') }}</textarea>
                @error('note')
                    <p class="mt-1 text-sm text-red-400">{{ $message }}</p>
                @enderror
                <p class="mt-1 text-sm text-gray-400">Maximum 255 characters</p>
            </div>

            <!-- Transaction PIN -->
            <div>
                <label for="pin" class="block text-sm font-medium text-gray-300 mb-2">
                    Transaction PIN
                    <span class="text-red-400">*</span>
                </label>
                <input type="password" 
                       id="pin" 
                       name="pin" 
                       maxlength="4"
                       pattern="[0-9]{4}"
                       required
                       class="w-full px-4 py-3 bg-cod-dark border border-cod-light rounded-lg text-gray-100 focus:ring-2 focus:ring-cod-green focus:border-transparent text-center text-2xl tracking-widest @error('pin') border-red-500 @enderror"
                       placeholder="••••">
                @error('pin')
                    <p class="mt-1 text-sm text-red-400">{{ $message }}</p>
                @enderror
                <p class="mt-1 text-sm text-gray-400">Enter your 4-digit transaction PIN to confirm</p>
            </div>

            <!-- Security Notice -->
            <div class="bg-yellow-500/10 border border-yellow-500/20 rounded-lg p-4">
                <div class="flex items-start">
                    <svg class="w-5 h-5 text-yellow-400 mt-0.5 mr-3 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L3.732 16.5c-.77.833.192 2.5 1.732 2.5z"></path>
                    </svg>
                    <div>
                        <h4 class="text-yellow-400 font-medium mb-1">Transfer Information</h4>
                        <ul class="text-sm text-yellow-300 space-y-1">
                            <li>• Transfers are instant and cannot be reversed</li>
                            <li>• Verify the recipient username before confirming</li>
                            <li>• Your PIN is required to complete the transfer</li>
                            <li>• Contact support if you need assistance</li>
                        </ul>
                    </div>
                </div>
            </div>

            <!-- Action Buttons -->
            <div class="flex space-x-4">
                <a href="{{ route('wallet.index') }}" 
                   class="flex-1 bg-cod-light text-cod-green text-center px-6 py-3 rounded-lg font-semibold hover:bg-cod-dark transition-colors">
                    Cancel
                </a>
                <button type="submit" 
                        class="flex-1 bg-cod-green text-black px-6 py-3 rounded-lg font-semibold hover:bg-green-600 transition-colors"
                        id="submitBtn">
                    Transfer Credits
                </button>
            </div>
        </form>
    </div>

    <!-- Transfer History Link -->
    <div class="mt-8 text-center">
        <a href="{{ route('wallet.transfer.history') }}" class="text-cod-green hover:text-green-400 font-medium">
            View Transfer History →
        </a>
    </div>
</div>

<script>
// Amount quick-set function
function setAmount(amount) {
    document.getElementById('amount').value = amount;
}

// Auto-format PIN input
document.addEventListener('DOMContentLoaded', function() {
    const pinInput = document.getElementById('pin');
    const amountInput = document.getElementById('amount');
    const form = document.getElementById('transferForm');
    const submitBtn = document.getElementById('submitBtn');
    
    // PIN formatting
    pinInput.addEventListener('input', function(e) {
        this.value = this.value.replace(/[^0-9]/g, '');
        if (this.value.length > 4) {
            this.value = this.value.slice(0, 4);
        }
    });
    
    pinInput.addEventListener('keypress', function(e) {
        if (!/[0-9]/.test(e.key) && !['Backspace', 'Delete', 'Tab', 'Enter'].includes(e.key)) {
            e.preventDefault();
        }
    });
    
    // Amount validation
    amountInput.addEventListener('input', function(e) {
        const value = parseFloat(this.value) || 0;
        const maxBalance = {!! json_encode(auth()->user()->wallet_balance) !!};
        
        if (value > maxBalance) {
            this.setCustomValidity('Amount cannot exceed your available balance');
        } else if (value < 1) {
            this.setCustomValidity('Minimum transfer amount is ₦1.00');
        } else if (value > 100000) {
            this.setCustomValidity('Maximum transfer amount is ₦100,000.00');
        } else {
            this.setCustomValidity('');
        }
    });
    
    // Form submission confirmation
    form.addEventListener('submit', function(e) {
        const amount = document.getElementById('amount').value;
        const recipient = document.getElementById('recipient_username').value;
        
        if (!confirm(`Are you sure you want to transfer ₦${parseFloat(amount).toLocaleString()} to ${recipient}?`)) {
            e.preventDefault();
            return false;
        }
        
        // Disable submit button to prevent double submission
        submitBtn.disabled = true;
        submitBtn.textContent = 'Processing...';
    });
});
</script>
@endsection