<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\User;
use App\Models\Level;
use App\Services\LevelService;
use App\Http\Controllers\Admin\UserController;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;

class AdminUserControllerTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        
        // Run the thirty level system seeder to ensure all levels exist
        $this->artisan('db:seed', ['--class' => 'Database\\Seeders\\ThirtyLevelSystemSeeder']);
    }

    /** @test */
    public function controller_fetches_all_levels_and_processes_them_correctly()
    {
        // Create a regular user to view
        $user = User::factory()->create([
            'level' => 1
        ]);

        // Test the controller logic directly
        $levelService = new LevelService();
        $controller = new UserController($levelService);
        
        // Use reflection to test the show method logic
        $levels = Level::all();
        $processedLevels = $levelService->processLevelsForDropdown($levels);
        
        // Verify that all 31 levels are processed
        $this->assertCount(31, $processedLevels);

        // Verify specific levels exist in the processed levels
        $levelNames = $processedLevels->pluck('name')->toArray();
        
        // Check that we have Level 1 through Level 30
        for ($i = 1; $i <= 30; $i++) {
            $this->assertContains("Level {$i}", $levelNames);
        }
        
        // Check that Creator level exists
        $this->assertContains('Creator', $levelNames);

        // Verify that the processed levels have the correct display names with limits
        $level1 = $processedLevels->where('name', 'Level 1')->first();
        $this->assertEquals('Level 1 - Beginner (₦1,000 limit)', $level1->display_name_with_limits);

        $level30 = $processedLevels->where('name', 'Level 30')->first();
        $this->assertEquals('Level 30 - Legend (₦4,600,000 limit)', $level30->display_name_with_limits);

        $creator = $processedLevels->where('name', 'Creator')->first();
        $this->assertEquals('Creator - Content & Tournament Host (No limit)', $creator->display_name_with_limits);
    }

    /** @test */
    public function all_levels_are_created_by_seeder()
    {
        // Verify that exactly 31 levels exist (30 regular + Creator)
        $this->assertCount(31, Level::all());
        
        // Verify that Level 1 through Level 30 exist
        for ($i = 1; $i <= 30; $i++) {
            $this->assertDatabaseHas('levels', ['name' => "Level {$i}"]);
        }
        
        // Verify that Creator level exists
        $this->assertDatabaseHas('levels', ['name' => 'Creator']);
        
        // Verify that the first level has correct wager limit
        $level1 = Level::where('name', 'Level 1')->first();
        $this->assertEquals(1000, $level1->max_wager_amount);
        
        // Verify that the last level has correct wager limit
        $level30 = Level::where('name', 'Level 30')->first();
        $this->assertEquals(4600000, $level30->max_wager_amount);
        
        // Verify that Creator has no wager limit
        $creator = Level::where('name', 'Creator')->first();
        $this->assertNull($creator->max_wager_amount);
    }

    /** @test */
    public function level_service_processes_all_levels_correctly()
    {
        $levelService = new LevelService();
        $levels = Level::all();
        $processedLevels = $levelService->processLevelsForDropdown($levels);
        
        // Test a few specific levels
        $level5 = $processedLevels->where('name', 'Level 5')->first();
        $this->assertEquals('Level 5 - Novice (₦100,000 limit)', $level5->display_name_with_limits);
        
        $level15 = $processedLevels->where('name', 'Level 15')->first();
        $this->assertEquals('Level 15 - Advanced (₦850,000 limit)', $level15->display_name_with_limits);
        
        $level25 = $processedLevels->where('name', 'Level 25')->first();
        $this->assertEquals('Level 25 - Master (₦2,600,000 limit)', $level25->display_name_with_limits);
    }
}