<?php

namespace Tests\Feature;

use App\Models\User;
use App\Models\Level;
use App\Models\WalletTransaction;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class AdminUserManagementTest extends TestCase
{
    use RefreshDatabase;

    protected $admin;
    protected $user;
    protected $levels;

    protected function setUp(): void
    {
        parent::setUp();
        
        // Seed levels
        $this->artisan('db:seed', ['--class' => 'ThirtyLevelSystemSeeder']);
        $this->levels = Level::all();
        
        // Create admin user
        $this->admin = User::factory()->create([
            'username' => 'admin',
            'email' => 'admin@test.com',
            'is_organizer' => true,
            'role' => 'admin',
            'level' => 1
        ]);
        
        // Create regular user
        $this->user = User::factory()->create([
            'username' => 'testuser',
            'email' => 'user@test.com',
            'wallet_balance' => 1000.00,
            'xp' => 100,
            'level' => 2
        ]);
    }

    /** @test */
    public function admin_can_access_user_management_page()
    {
        $response = $this->actingAs($this->admin)
            ->get('/admin/users');

        $response->assertStatus(200);
        $response->assertSee('User Management');
        $response->assertSee($this->user->username);
    }

    /** @test */
    public function admin_can_assign_level_to_user()
    {
        $creatorLevel = $this->levels->where('name', 'Creator')->first();
        
        $response = $this->actingAs($this->admin)
            ->post("/admin/users/{$this->user->id}/assign-level", [
                'level_id' => $creatorLevel->id
            ]);

        $response->assertRedirect();
        $response->assertSessionHas('success');
        
        $this->user->refresh();
        $this->assertEquals($creatorLevel->id, $this->user->level);
        
        // Check if role was updated for Creator level
        $this->assertEquals('creator', $this->user->role);
    }

    /** @test */
    public function admin_can_credit_user_wallet()
    {
        $initialBalance = $this->user->wallet_balance;
        $creditAmount = 500.00;
        
        $response = $this->actingAs($this->admin)
            ->post("/admin/users/{$this->user->id}/credit-wallet", [
                'amount' => $creditAmount,
                'description' => 'Test credit'
            ]);

        $response->assertRedirect();
        $response->assertSessionHas('success');
        
        $this->user->refresh();
        $this->assertEquals($initialBalance + $creditAmount, $this->user->wallet_balance);
        
        // Check transaction was created
        $this->assertDatabaseHas('wallet_transactions', [
            'user_id' => $this->user->id,
            'type' => 'admin_credit',
            'amount' => $creditAmount,
            'status' => 'completed'
        ]);
    }

    /** @test */
    public function admin_can_debit_user_wallet()
    {
        $initialBalance = $this->user->wallet_balance;
        $debitAmount = 200.00;
        
        $response = $this->actingAs($this->admin)
            ->post("/admin/users/{$this->user->id}/debit-wallet", [
                'amount' => $debitAmount,
                'description' => 'Test debit'
            ]);

        $response->assertRedirect();
        $response->assertSessionHas('success');
        
        $this->user->refresh();
        $this->assertEquals($initialBalance - $debitAmount, $this->user->wallet_balance);
        
        // Check transaction was created
        $this->assertDatabaseHas('wallet_transactions', [
            'user_id' => $this->user->id,
            'type' => 'admin_debit',
            'amount' => -$debitAmount,
            'status' => 'completed'
        ]);
    }

    /** @test */
    public function admin_cannot_debit_more_than_user_balance()
    {
        $response = $this->actingAs($this->admin)
            ->post("/admin/users/{$this->user->id}/debit-wallet", [
                'amount' => $this->user->wallet_balance + 100,
                'description' => 'Test excessive debit'
            ]);

        $response->assertSessionHasErrors(['amount']);
    }

    /** @test */
    public function admin_can_view_user_details_with_activity()
    {
        // Create some wallet transactions for activity
        WalletTransaction::create([
            'user_id' => $this->user->id,
            'type' => 'deposit',
            'amount' => 100,
            'status' => 'completed',
            'description' => 'Test deposit'
        ]);
        
        $response = $this->actingAs($this->admin)
            ->get("/admin/users/{$this->user->id}");

        $response->assertStatus(200);
        $response->assertSee('User Details');
        $response->assertSee($this->user->username);
        $response->assertSee('Level Assignment');
        $response->assertSee('Wallet Management');
        $response->assertSee('User Activity Log');
    }

    /** @test */
    public function admin_can_filter_users_by_level()
    {
        $level5 = $this->levels->where('name', 'Level 5')->first();
        
        $response = $this->actingAs($this->admin)
            ->get("/admin/users?level={$level5->id}");

        $response->assertStatus(200);
    }

    /** @test */
    public function admin_can_search_users()
    {
        $response = $this->actingAs($this->admin)
            ->get("/admin/users?search={$this->user->username}");

        $response->assertStatus(200);
        $response->assertSee($this->user->username);
    }

    /** @test */
    public function non_admin_cannot_access_admin_routes()
    {
        $nonAdmin = User::factory()->create(['is_organizer' => false]);
        
        $response = $this->actingAs($nonAdmin)
            ->get('/admin/users');

        $response->assertStatus(403);
    }

    /** @test */
    public function admin_actions_are_logged()
    {
        $this->actingAs($this->admin)
            ->post("/admin/users/{$this->user->id}/credit-wallet", [
                'amount' => 100,
                'description' => 'Test credit for logging'
            ]);

        // Check admin action was logged
        $this->assertDatabaseHas('wallet_transactions', [
            'user_id' => $this->user->id,
            'type' => 'admin_action',
            'description' => 'Admin Action: wallet_credited'
        ]);
    }
}