<?php

namespace Tests\Feature;

use Tests\TestCase;
use Illuminate\Support\Facades\View;

class BladeCreateTemplatesSyntaxTest extends TestCase
{
    /**
     * List of create templates to test for syntax errors.
     */
    protected $createTemplates = [
        'challenges.create',
        'tournaments.create',
        'admin.tournaments.create',
        'admin.challenges.create',
        'admin.games.create',
        'admin.posts.create',
        'admin.popup-notifications.create',
        'admin.notifications.create-provider',
        'creator.blogs.create',
    ];
    
    /**
     * Test that all create Blade templates have valid syntax.
     */
    public function test_all_create_templates_have_valid_blade_syntax()
    {
        foreach ($this->createTemplates as $template) {
            $this->assertTemplateSyntax($template);
        }
    }
    
    /**
     * Test specific template for valid syntax.
     */
    protected function assertTemplateSyntax($templateName)
    {
        try {
            $view = View::getFinder()->find($templateName);
            $contents = file_get_contents($view);
            
            // Check for basic Blade syntax errors
            $this->assertNotEmpty($contents, "Template {$templateName} should not be empty");
            
            // Verify @error/@enderror directive balance
            $errorCount = substr_count($contents, '@error');
            $enderrorCount = substr_count($contents, '@enderror');
            $this->assertEquals($errorCount, $enderrorCount, 
                "All @error directives should have matching @enderror directives in template {$templateName}");
            
            // Check for stray @enderror without @error
            $lines = explode("\n", $contents);
            $inErrorBlock = false;
            $lineNumber = 0;
            
            foreach ($lines as $line) {
                $lineNumber++;
                
                if (strpos($line, '@error') !== false) {
                    $this->assertFalse($inErrorBlock, 
                        "Found nested or overlapping @error directive on line {$lineNumber} in template {$templateName}");
                    $inErrorBlock = true;
                }
                
                if (strpos($line, '@enderror') !== false) {
                    $this->assertTrue($inErrorBlock, 
                        "Found @enderror without matching @error on line {$lineNumber} in template {$templateName}");
                    $inErrorBlock = false;
                }
            }
            
            $this->assertFalse($inErrorBlock, "All @error blocks should be properly closed in template {$templateName}");
            
            // Check for duplicate @endsection directives
            $endsectionCount = substr_count($contents, '@endsection');
            $this->assertLessThanOrEqual(1, $endsectionCount, 
                "Template {$templateName} should have only one @endsection directive");
            
            // Check for orphaned content after @endsection
            $foundEndsection = false;
            $contentAfterEndsection = false;
            
            foreach ($lines as $line) {
                $trimmedLine = trim($line);
                
                if ($trimmedLine === '@endsection') {
                    $foundEndsection = true;
                    continue;
                }
                
                // If we've found @endsection and there's meaningful content after it
                if ($foundEndsection && !empty($trimmedLine) && 
                    !preg_match('/^\/\*|^\*|^\*\/|^\/\/|^@push|^@endpush|^<script|^<\/script/', $trimmedLine)) {
                    $contentAfterEndsection = true;
                    break;
                }
            }
            
            $this->assertFalse($contentAfterEndsection, 
                "Template {$templateName} should not have content after @endsection (except for @push sections)");
                
        } catch (\Exception $e) {
            // If template doesn't exist, skip it
            if (strpos($e->getMessage(), 'View') !== false && strpos($e->getMessage(), 'not found') !== false) {
                $this->markTestSkipped("Template {$templateName} does not exist");
            } else {
                $this->fail("Template {$templateName} has syntax errors: " . $e->getMessage());
            }
        }
    }
    
    /**
     * Test for balanced section directives across all create templates.
     */
    public function test_create_templates_have_balanced_section_directives()
    {
        $sectionDirectives = [
            '@section' => '@endsection',
            '@if' => '@endif',
            '@foreach' => '@endforeach',
            '@for' => '@endfor',
            '@while' => '@endwhile',
            '@push' => '@endpush',
        ];
        
        foreach ($this->createTemplates as $template) {
            try {
                $view = View::getFinder()->find($template);
                $contents = file_get_contents($view);
                
                foreach ($sectionDirectives as $start => $end) {
                    $startCount = substr_count($contents, $start);
                    $endCount = substr_count($contents, $end);
                    
                    $this->assertEquals($startCount, $endCount, 
                        "Template {$template}: All {$start} directives should have matching {$end} directives. Found {$startCount} {$start} and {$endCount} {$end}");
                }
                
            } catch (\Exception $e) {
                if (strpos($e->getMessage(), 'View') !== false && strpos($e->getMessage(), 'not found') !== false) {
                    $this->markTestSkipped("Template {$template} does not exist");
                }
            }
        }
    }
    
    /**
     * Test that create templates don't have common syntax errors.
     */
    public function test_create_templates_common_syntax_errors()
    {
        foreach ($this->createTemplates as $template) {
            try {
                $view = View::getFinder()->find($template);
                $contents = file_get_contents($view);
                
                // Check for common syntax errors
                // Removed incorrect assertion for '{{ old(' syntax; valid usage should not fail the test.
                
                // Check for properly closed PHP tags in Blade
                $phpOpenCount = substr_count($contents, '<?php');
                $phpCloseCount = substr_count($contents, '?>');
                if ($phpOpenCount > 0) {
                    $this->assertEquals($phpOpenCount, $phpCloseCount,
                        "Template {$template}: All <?php tags should have matching ?> tags");
                }
                
            } catch (\Exception $e) {
                if (strpos($e->getMessage(), 'View') !== false && strpos($e->getMessage(), 'not found') !== false) {
                    $this->markTestSkipped("Template {$template} does not exist");
                }
            }
        }
    }
}