<?php

namespace Tests\Feature;

use Tests\TestCase;
use Illuminate\Support\Facades\File;

class BladeGlobalSyntaxTest extends TestCase
{
    /**
     * Test all Blade templates in the application for basic syntax errors.
     */
    public function test_all_blade_templates_have_valid_syntax()
    {
        $bladeFiles = File::allFiles(resource_path('views'));
        $bladeFiles = array_filter($bladeFiles, function ($file) {
            return $file->getExtension() === 'php' && str_contains($file->getFilename(), '.blade.');
        });
        
        $errors = [];
        
        foreach ($bladeFiles as $file) {
            $relativePath = str_replace(resource_path('views/'), '', $file->getPathname());
            $contents = File::get($file->getPathname());
            
            // Check for balanced @error/@enderror directives
            $errorCount = substr_count($contents, '@error');
            $enderrorCount = substr_count($contents, '@enderror');
            
            if ($errorCount !== $enderrorCount) {
                $errors[] = "{$relativePath}: Unbalanced @error/@enderror directives ({$errorCount} @error, {$enderrorCount} @enderror)";
            }
            
            // Check for stray @enderror without @error
            $lines = explode("\n", $contents);
            $inErrorBlock = false;
            $lineNumber = 0;
            
            foreach ($lines as $line) {
                $lineNumber++;
                
                if (strpos($line, '@error') !== false) {
                    if ($inErrorBlock) {
                        $errors[] = "{$relativePath}:{$lineNumber}: Nested @error directive found";
                    }
                    $inErrorBlock = true;
                }
                
                if (strpos($line, '@enderror') !== false) {
                    if (!$inErrorBlock) {
                        $errors[] = "{$relativePath}:{$lineNumber}: @enderror without matching @error";
                    }
                    $inErrorBlock = false;
                }
            }
            
            if ($inErrorBlock) {
                $errors[] = "{$relativePath}: Unclosed @error block";
            }
            
            // Check for multiple @endsection directives
            $endsectionCount = substr_count($contents, '@endsection');
            if ($endsectionCount > 1) {
                $errors[] = "{$relativePath}: Multiple @endsection directives found ({$endsectionCount})";
            }
            
            // Check for orphaned content after @endsection
            if ($endsectionCount === 1) {
                $foundEndsection = false;
                $contentAfterEndsection = false;
                
                foreach ($lines as $line) {
                    $trimmedLine = trim($line);
                    
                    if ($trimmedLine === '@endsection') {
                        $foundEndsection = true;
                        continue;
                    }
                    
                    // If we've found @endsection and there's meaningful content after it
                    if ($foundEndsection && !empty($trimmedLine) && 
                        !preg_match('/^\/\*|^\*|^\*\/|^\/\/|^@push|^@endpush|^<script|^<\/script/', $trimmedLine)) {
                        $contentAfterEndsection = true;
                        break;
                    }
                }
                
                if ($contentAfterEndsection) {
                    $errors[] = "{$relativePath}: Content found after @endsection";
                }
            }
        }
        
        if (!empty($errors)) {
            $this->fail("Blade syntax errors found:\n" . implode("\n", $errors));
        }
        
        $this->assertTrue(true, 'All Blade templates have valid syntax');
    }
    
    /**
     * Test for balanced common Blade directives.
     */
    public function test_balanced_blade_directives()
    {
        $bladeFiles = File::allFiles(resource_path('views'));
        $bladeFiles = array_filter($bladeFiles, function ($file) {
            return $file->getExtension() === 'php' && str_contains($file->getFilename(), '.blade.');
        });
        
        $directivePairs = [
            '@if' => '@endif',
            '@foreach' => '@endforeach',
            '@for' => '@endfor',
            '@while' => '@endwhile',
            '@switch' => '@endswitch',
        ];
        
        $errors = [];
        
        foreach ($bladeFiles as $file) {
            $relativePath = str_replace(resource_path('views/'), '', $file->getPathname());
            $contents = File::get($file->getPathname());
            
            foreach ($directivePairs as $start => $end) {
                $startCount = substr_count($contents, $start);
                $endCount = substr_count($contents, $end);
                
                // Account for @elseif, @else in @if blocks
                if ($start === '@if') {
                    $elseifCount = substr_count($contents, '@elseif');
                    $elseCount = substr_count($contents, '@else');
                    // @elseif and @else are part of @if blocks, so adjust accordingly
                    // We can't easily validate this without a proper parser, so we'll skip complex validation
                    continue;
                }
                
                if ($startCount !== $endCount) {
                    $errors[] = "{$relativePath}: Unbalanced {$start}/{$end} directives ({$startCount} {$start}, {$endCount} {$end})";
                }
            }
        }
        
        if (!empty($errors)) {
            $this->fail("Unbalanced Blade directives found:\n" . implode("\n", $errors));
        }
        
        $this->assertTrue(true, 'All Blade directive pairs are balanced');
    }
}