<?php

namespace Tests\Feature;

use Tests\TestCase;
use Illuminate\View\Factory;
use Illuminate\Support\Facades\View;

class BladeTournamentSyntaxTest extends TestCase
{
    /**
     * Test that tournament creation Blade templates have valid syntax.
     */
    public function test_tournament_create_blade_syntax_is_valid()
    {
        // Test public tournament create template
        $publicTournamentTemplate = 'tournaments.create';
        
        try {
            // Try to compile the Blade template without rendering
            $view = View::getFinder()->find($publicTournamentTemplate);
            $contents = file_get_contents($view);
            
            // Check for basic Blade syntax errors
            $this->assertNotEmpty($contents, 'Tournament create template should not be empty');
            
            // Verify @error/@enderror directive balance
            $errorCount = substr_count($contents, '@error');
            $enderrorCount = substr_count($contents, '@enderror');
            $this->assertEquals($errorCount, $enderrorCount, 
                'All @error directives should have matching @enderror directives in tournament create template');
            
            // Check for stray @enderror without @error
            $lines = explode("\n", $contents);
            $inErrorBlock = false;
            $lineNumber = 0;
            
            foreach ($lines as $line) {
                $lineNumber++;
                
                if (strpos($line, '@error') !== false) {
                    $this->assertFalse($inErrorBlock, 
                        "Found nested or overlapping @error directive on line {$lineNumber}");
                    $inErrorBlock = true;
                }
                
                if (strpos($line, '@enderror') !== false) {
                    $this->assertTrue($inErrorBlock, 
                        "Found @enderror without matching @error on line {$lineNumber}");
                    $inErrorBlock = false;
                }
            }
            
            $this->assertFalse($inErrorBlock, 'All @error blocks should be properly closed');
            
            // Test that the template can be read by Blade compiler
            $this->assertTrue(true, 'Tournament create template syntax is valid');
            
        } catch (\Exception $e) {
            $this->fail('Tournament create template has syntax errors: ' . $e->getMessage());
        }
    }
    
    /**
     * Test that admin tournament creation Blade template has valid syntax.
     */
    public function test_admin_tournament_create_blade_syntax_is_valid()
    {
        // Test admin tournament create template
        $adminTournamentTemplate = 'admin.tournaments.create';
        
        try {
            // Try to compile the Blade template without rendering
            $view = View::getFinder()->find($adminTournamentTemplate);
            $contents = file_get_contents($view);
            
            // Check for basic Blade syntax errors
            $this->assertNotEmpty($contents, 'Admin tournament create template should not be empty');
            
            // Verify @error/@enderror directive balance
            $errorCount = substr_count($contents, '@error');
            $enderrorCount = substr_count($contents, '@enderror');
            $this->assertEquals($errorCount, $enderrorCount, 
                'All @error directives should have matching @enderror directives in admin tournament create template');
            
            // Check for stray @enderror without @error
            $lines = explode("\n", $contents);
            $inErrorBlock = false;
            $lineNumber = 0;
            
            foreach ($lines as $line) {
                $lineNumber++;
                
                if (strpos($line, '@error') !== false) {
                    $this->assertFalse($inErrorBlock, 
                        "Found nested or overlapping @error directive on line {$lineNumber}");
                    $inErrorBlock = true;
                }
                
                if (strpos($line, '@enderror') !== false) {
                    $this->assertTrue($inErrorBlock, 
                        "Found @enderror without matching @error on line {$lineNumber}");
                    $inErrorBlock = false;
                }
            }
            
            $this->assertFalse($inErrorBlock, 'All @error blocks should be properly closed');
            
            // Test that the template can be read by Blade compiler
            $this->assertTrue(true, 'Admin tournament create template syntax is valid');
            
        } catch (\Exception $e) {
            $this->fail('Admin tournament create template has syntax errors: ' . $e->getMessage());
        }
    }
    
    /**
     * Test tournament template for structural integrity.
     */
    public function test_tournament_template_structure()
    {
        $publicTournamentTemplate = 'tournaments.create';
        
        try {
            $view = View::getFinder()->find($publicTournamentTemplate);
            $contents = file_get_contents($view);
            
            // Check for duplicate @endsection directives
            $endsectionCount = substr_count($contents, '@endsection');
            $this->assertLessThanOrEqual(1, $endsectionCount, 
                'Tournament template should have only one @endsection directive');
            
            // Check for orphaned content after @endsection
            $lines = explode("\n", $contents);
            $foundEndsection = false;
            $contentAfterEndsection = false;
            
            foreach ($lines as $line) {
                $trimmedLine = trim($line);
                
                if ($trimmedLine === '@endsection') {
                    $foundEndsection = true;
                    continue;
                }
                
                // If we've found @endsection and there's meaningful content after it
                if ($foundEndsection && !empty($trimmedLine) && !preg_match('/^\/\*|^\*|^\*\/|^\/\//', $trimmedLine)) {
                    $contentAfterEndsection = true;
                    break;
                }
            }
            
            $this->assertFalse($contentAfterEndsection, 
                'Tournament template should not have content after @endsection');
                
        } catch (\Exception $e) {
            $this->fail('Tournament template structure test failed: ' . $e->getMessage());
        }
    }
}