<?php

namespace Tests\Feature;

use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class DashboardRoutingTest extends TestCase
{
    use RefreshDatabase;

    /** @test */
    public function home_route_shows_public_content_for_guests()
    {
        $response = $this->get('/');
        
        $response->assertStatus(200);
        // Check for home page specific content
        $response->assertSee('Welcome to');
        $response->assertSee('AGAMEC');
        $response->assertSee('The Ultimate Competitive Gaming Platform');
        $response->assertSee('Platform Dominance'); // Public stats section
    }

    /** @test */
    public function home_route_shows_public_content_for_authenticated_users()
    {
        $user = User::factory()->create();
        
        $response = $this->actingAs($user)->get('/');
        
        $response->assertStatus(200);
        // Authenticated users should still see the public home page at root URL
        $response->assertSee('Welcome to');
        $response->assertSee('AGAMEC');
        $response->assertSee('The Ultimate Competitive Gaming Platform');
        $response->assertSee('Platform Dominance'); // Public stats section
    }

    /** @test */
    public function dashboard_route_redirects_guests_to_login()
    {
        $response = $this->get('/dashboard');
        
        $response->assertRedirect('/login');
    }

    /** @test */
    public function dashboard_route_shows_user_dashboard_for_authenticated_users()
    {
        $user = User::factory()->create(['username' => 'testuser']);
        
        $response = $this->actingAs($user)->get('/dashboard');
        
        $response->assertStatus(200);
        // Check for dashboard-specific content, not home page content
        $response->assertSee('Your Gaming Dashboard');
        $response->assertSee('Welcome back, testuser!');
        $response->assertSee('Quick Actions');
        $response->assertSee('Battles Fought'); // User stats section
        
        // Should NOT see home page specific content
        $response->assertDontSee('Welcome to');
        $response->assertDontSee('The Ultimate Competitive Gaming Platform');
        $response->assertDontSee('Platform Dominance'); // Public stats section shouldn't be on user dashboard
    }

    /** @test */
    public function dashboard_and_home_serve_different_content()
    {
        $user = User::factory()->create(['username' => 'testuser']);
        
        // Get home page content
        $homeResponse = $this->actingAs($user)->get('/');
        $homeContent = $homeResponse->getContent();
        
        // Get dashboard content  
        $dashboardResponse = $this->actingAs($user)->get('/dashboard');
        $dashboardContent = $dashboardResponse->getContent();
        
        // They should be different
        $this->assertNotEquals($homeContent, $dashboardContent);
        
        // Home should have public marketing content
        $this->assertStringContainsString('Welcome to', $homeContent);
        $this->assertStringContainsString('AGAMEC', $homeContent);
        
        // Dashboard should have user-specific content
        $this->assertStringContainsString('Your Gaming Dashboard', $dashboardContent);
        $this->assertStringContainsString('Welcome back, testuser!', $dashboardContent);
    }

    /** @test */
    public function admin_dashboard_is_separate_from_user_dashboard()
    {
        $user = User::factory()->create(['role' => 'highest']); // Assuming 'highest' role has admin access
        
        $userDashboardResponse = $this->actingAs($user)->get('/dashboard');
        $adminDashboardResponse = $this->actingAs($user)->get('/admin/dashboard');
        
        // Both should return 200 but with different content
        $userDashboardResponse->assertStatus(200);
        
        $userDashboardContent = $userDashboardResponse->getContent();
        
        // User dashboard should have user-specific content
        $this->assertStringContainsString('Your Gaming Dashboard', $userDashboardContent);
        
        // User dashboard should NOT have admin-specific content
        $this->assertStringNotContainsString('Admin Dashboard', $userDashboardContent);
        $this->assertStringNotContainsString('Platform management and oversight', $userDashboardContent);
    }
}