<?php

namespace Tests\Feature;

use App\Models\Challenge;
use App\Models\Dispute;
use App\Models\Game;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Tests\TestCase;

class EnhancedChallengeSystemTest extends TestCase
{
    use RefreshDatabase, WithFaker;

    /**
     * Test challenge creation with new fields
     */
    public function test_challenge_creation_with_game_info()
    {
        $user = User::factory()->create(['wallet_balance' => 1000]);
        $game = Game::factory()->create();

        $challengeData = [
            'game_id' => $game->id,
            'wager_amount' => 100,
            'best_of' => 3,
            'rules_text' => 'Test rules',
            'creator_game_username' => 'TestPlayer123',
            'group_link' => 'https://example.com/lobby/123',
            'group_id' => 'LOBBY123',
        ];

        $challenge = Challenge::create(array_merge($challengeData, [
            'creator_id' => $user->id
        ]));

        $this->assertDatabaseHas('challenges', [
            'creator_id' => $user->id,
            'creator_game_username' => 'TestPlayer123',
            'group_link' => 'https://example.com/lobby/123',
            'group_id' => 'LOBBY123',
        ]);
    }

    /**
     * Test result claim matching logic
     */
    public function test_result_claims_matching()
    {
        $creator = User::factory()->create();
        $accepter = User::factory()->create();
        $game = Game::factory()->create();

        $challenge = Challenge::create([
            'creator_id' => $creator->id,
            'accepter_id' => $accepter->id,
            'game_id' => $game->id,
            'wager_amount' => 100,
            'best_of' => 1,
            'creator_game_username' => 'Creator123',
            'creator_claimed_result' => 'win',
            'accepter_claimed_result' => 'loss',
        ]);

        $this->assertTrue($challenge->hasBothClaimedResults());
        $this->assertTrue($challenge->resultsMatch());
        $this->assertFalse($challenge->bothClaimVictory());
        $this->assertEquals($creator->id, $challenge->getWinnerFromClaims()->id);
    }

    /**
     * Test both claim victory scenario
     */
    public function test_both_claim_victory_scenario()
    {
        $creator = User::factory()->create();
        $accepter = User::factory()->create();
        $game = Game::factory()->create();

        $challenge = Challenge::create([
            'creator_id' => $creator->id,
            'accepter_id' => $accepter->id,
            'game_id' => $game->id,
            'wager_amount' => 100,
            'best_of' => 1,
            'creator_game_username' => 'Creator123',
            'creator_claimed_result' => 'win',
            'accepter_claimed_result' => 'win',
        ]);

        $this->assertTrue($challenge->hasBothClaimedResults());
        $this->assertFalse($challenge->resultsMatch());
        $this->assertTrue($challenge->bothClaimVictory());
    }

    /**
     * Test dispute evidence functionality
     */
    public function test_dispute_evidence_functionality()
    {
        $creator = User::factory()->create();
        $game = Game::factory()->create();

        $challenge = Challenge::create([
            'creator_id' => $creator->id,
            'game_id' => $game->id,
            'wager_amount' => 100,
            'best_of' => 1,
            'creator_game_username' => 'Creator123',
        ]);

        $dispute = Dispute::create([
            'challenge_id' => $challenge->id,
            'opened_by' => $creator->id,
            'reason' => 'Both claim victory',
            'creator_evidence_path' => 'evidence/creator-proof.jpg',
        ]);

        $this->assertTrue($dispute->hasCreatorEvidence());
        $this->assertFalse($dispute->hasAccepterEvidence());
        $this->assertFalse($dispute->hasBothEvidence());

        $dispute->update(['accepter_evidence_path' => 'evidence/accepter-proof.jpg']);

        $this->assertTrue($dispute->hasAccepterEvidence());
        $this->assertTrue($dispute->hasBothEvidence());
    }

    /**
     * Test challenge model helper methods
     */
    public function test_challenge_helper_methods()
    {
        $creator = User::factory()->create();
        $accepter = User::factory()->create();
        $game = Game::factory()->create();

        $challenge = Challenge::create([
            'creator_id' => $creator->id,
            'accepter_id' => $accepter->id,
            'game_id' => $game->id,
            'wager_amount' => 100,
            'best_of' => 1,
            'creator_game_username' => 'Creator123',
            'creator_claimed_result' => 'win',
            'accepter_claimed_result' => 'loss',
        ]);

        // Test reset functionality
        $challenge->resetResultClaims();
        $challenge->refresh();

        $this->assertEquals('none', $challenge->creator_claimed_result);
        $this->assertEquals('none', $challenge->accepter_claimed_result);
        $this->assertFalse($challenge->hasBothClaimedResults());
    }

    /**
     * Test result claiming flow for accepted challenges
     */
    public function test_result_claiming_flow_for_accepted_challenges()
    {
        $creator = User::factory()->create(['wallet_balance' => 1000]);
        $accepter = User::factory()->create(['wallet_balance' => 1000]);
        $game = Game::factory()->create();

        $challenge = Challenge::create([
            'creator_id' => $creator->id,
            'accepter_id' => $accepter->id,
            'game_id' => $game->id,
            'wager_amount' => 100,
            'best_of' => 1,
            'creator_game_username' => 'Creator123',
            'status' => 'accepted', // Challenge has been accepted and match played
        ]);

        // Test creator claiming victory
        $response = $this->actingAs($creator)
            ->post(route('challenges.claim-result', $challenge), [
                'result' => 'win'
            ]);

        $response->assertRedirect();
        $challenge->refresh();
        $this->assertEquals('win', $challenge->creator_claimed_result);
        $this->assertEquals('none', $challenge->accepter_claimed_result);

        // Test accepter claiming loss (matching result)
        $response = $this->actingAs($accepter)
            ->post(route('challenges.claim-result', $challenge), [
                'result' => 'loss'
            ]);

        $response->assertRedirect();
        $response->assertSessionHas('success', 'Match resolved! You forfeited and your opponent wins. Funds have been distributed.');
        
        $challenge->refresh();
        $this->assertEquals('win', $challenge->creator_claimed_result);
        $this->assertEquals('loss', $challenge->accepter_claimed_result);
        $this->assertEquals('completed', $challenge->status);
        $this->assertEquals('creator_win', $challenge->result);
    }

    /**
     * Test dispute creation when both players claim victory
     */
    public function test_dispute_creation_on_both_claim_victory()
    {
        $creator = User::factory()->create(['wallet_balance' => 1000]);
        $accepter = User::factory()->create(['wallet_balance' => 1000]);
        $game = Game::factory()->create();

        $challenge = Challenge::create([
            'creator_id' => $creator->id,
            'accepter_id' => $accepter->id,
            'game_id' => $game->id,
            'wager_amount' => 100,
            'best_of' => 1,
            'creator_game_username' => 'Creator123',
            'status' => 'accepted',
        ]);

        // Both players claim victory
        $this->actingAs($creator)
            ->post(route('challenges.claim-result', $challenge), ['result' => 'win']);

        $response = $this->actingAs($accepter)
            ->post(route('challenges.claim-result', $challenge), ['result' => 'win']);

        $response->assertRedirect();
        $response->assertSessionHas('info', 'Both players claim victory. Please upload evidence to resolve the dispute.');
        
        $challenge->refresh();
        $this->assertEquals('disputed', $challenge->status);
        $this->assertTrue($challenge->bothClaimVictory());
        $this->assertNotNull($challenge->dispute);
    }

    /**
     * Test that result claiming is restricted to challenge participants
     */
    public function test_result_claiming_restricted_to_participants()
    {
        $creator = User::factory()->create(['wallet_balance' => 1000]);
        $accepter = User::factory()->create(['wallet_balance' => 1000]);
        $outsider = User::factory()->create();
        $game = Game::factory()->create();

        $challenge = Challenge::create([
            'creator_id' => $creator->id,
            'accepter_id' => $accepter->id,
            'game_id' => $game->id,
            'wager_amount' => 100,
            'best_of' => 1,
            'creator_game_username' => 'Creator123',
            'status' => 'accepted',
        ]);

        $response = $this->actingAs($outsider)
            ->post(route('challenges.claim-result', $challenge), ['result' => 'win']);

        $response->assertRedirect();
        $response->assertSessionHas('error', 'You are not part of this challenge.');
    }
}