<?php

namespace Tests\Feature;

use App\Enums\UserRole;
use App\Models\User;
use App\Models\Challenge;
use App\Models\Tournament;
use App\Models\Game;
use App\Models\Level;
use App\Policies\ChallengePolicy;
use App\Policies\TournamentPolicy;
use App\Http\Controllers\LeaderboardController;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;
use Illuminate\Http\Request;

class FixValidationTest extends TestCase
{
    use RefreshDatabase;

    // Constants for test level IDs for clarity
    private const LEVEL_1_ID = 1;
    private const LEVEL_29_ID = 2; // Level 29 record created in setUp() gets ID 2

    protected function setUp(): void
    {
        parent::setUp();
        
        // Seed some basic levels for testing
        \App\Models\Level::create([
            'name' => 'Level 1',
            'min_xp' => 0,
            'min_matches' => 0,
            'min_games_played' => 0,
            'min_agreement_rate' => 0.6,
            'requires_kyc' => false,
            'can_create_challenges' => true,
            'can_host_free_tournaments' => false,
            'can_host_paid_tournaments' => false,
            'max_wager_amount' => 20000,
        ]);
        
        \App\Models\Level::create([
            'name' => 'Level 29',
            'min_xp' => 2800,
            'min_matches' => 58,
            'min_games_played' => 200,
            'min_agreement_rate' => 0.95,
            'requires_kyc' => true,
            'can_create_challenges' => true,
            'can_host_free_tournaments' => true,
            'can_host_paid_tournaments' => true,
            'max_wager_amount' => 200000,
        ]);
    }

    /** @test */
    public function challenge_policy_allows_basic_user_to_create()
    {
        $user = User::factory()->create([
            'role' => UserRole::BASIC_USER,
            'xp' => 0,
            'games_played' => 0,
            'is_organizer' => false // Explicitly set to false to ensure test passes
        ]);
        $policy = new ChallengePolicy();
        
        $this->assertTrue($policy->create($user));
    }

    /** @test */
    public function challenge_policy_allows_premium_user_to_create()
    {
        $user = User::factory()->create([
            'role' => UserRole::PREMIUM_USER,
            'xp' => 2800,
            'games_played' => 200,
            'is_organizer' => false // Explicitly set to false to ensure test passes
        ]);
        $policy = new ChallengePolicy();
        
        $this->assertTrue($policy->create($user));
    }

    /** @test */
    public function challenge_policy_allows_creator_to_create()
    {
        $user = User::factory()->create([
            'role' => UserRole::CREATOR,
            'xp' => 3000,
            'games_played' => 1000,
            'is_organizer' => false // Explicitly set to false to ensure test passes
        ]);
        $policy = new ChallengePolicy();
        
        $this->assertTrue($policy->create($user));
    }

    /** @test */
    public function challenge_policy_allows_moderator_to_create()
    {
        $user = User::factory()->create([
            'role' => UserRole::MODERATOR,
            'xp' => 0,
            'games_played' => 0,
            'is_organizer' => false // Explicitly set to false to ensure test passes
        ]);
        $policy = new ChallengePolicy();
        
        $this->assertTrue($policy->create($user));
    }

    /** @test */
    public function challenge_policy_denies_admin_to_create()
    {
        $user = User::factory()->create([
            'role' => UserRole::ADMIN,
            'is_organizer' => true
        ]);
        $policy = new ChallengePolicy();
        
        $this->assertFalse($policy->create($user));
    }

    /** @test */
    public function tournament_policy_allows_premium_user_to_create()
    {
        $user = User::factory()->create([
            'role' => UserRole::PREMIUM_USER,
            'xp' => 2800,
            'games_played' => 200,
            'level' => self::LEVEL_29_ID, // References Level 29 record created in setUp()
            'is_organizer' => false // Explicitly set to false to ensure deterministic test
        ]);
        $policy = new TournamentPolicy();
        
        $this->assertTrue($policy->create($user));
    }

    /** @test */
    public function tournament_policy_allows_creator_to_create()
    {
        $user = User::factory()->create([
            'role' => UserRole::CREATOR,
            'xp' => 3000,
            'games_played' => 1000,
            'is_organizer' => false // Explicitly set to false to ensure deterministic test
        ]);
        $policy = new TournamentPolicy();
        
        $this->assertTrue($policy->create($user));
    }

    /** @test */
    public function tournament_policy_denies_moderator_without_level_to_create()
    {
        // Create a moderator at level 1 - should NOT be able to create tournaments
        $user = User::factory()->create([
            'role' => UserRole::MODERATOR,
            'level' => self::LEVEL_1_ID, // References Level 1 record created in setUp()
            'xp' => 0,
            'games_played' => 0,
            'is_organizer' => false // Explicitly set to false to ensure deterministic test
        ]);
        $policy = new TournamentPolicy();
        
        $this->assertFalse($policy->create($user));
    }

    /** @test */
    public function tournament_policy_allows_moderator_with_level_29_to_create()
    {
        // Create a moderator at level 29 - should be able to create tournaments
        $user = User::factory()->create([
            'role' => UserRole::MODERATOR,
            'level' => self::LEVEL_29_ID, // References Level 29 record created in setUp()
            'xp' => 2800,
            'games_played' => 200,
            'is_organizer' => false // Explicitly set to false to ensure deterministic test
        ]);
        $policy = new TournamentPolicy();
        
        $this->assertTrue($policy->create($user));
    }

    /** @test */
    public function tournament_policy_denies_basic_user_to_create()
    {
        $user = User::factory()->create([
            'role' => UserRole::BASIC_USER,
            'xp' => 0,
            'games_played' => 0,
            'level' => self::LEVEL_1_ID, // References Level 1 record created in setUp()
            'is_organizer' => false // Explicitly set to false to ensure deterministic test
        ]);
        $policy = new TournamentPolicy();
        
        $this->assertFalse($policy->create($user));
    }

    /** @test */
    public function tournament_policy_allows_admin_to_create()
    {
        $user = User::factory()->create([
            'role' => UserRole::ADMIN,
            'is_organizer' => true
        ]);
        $policy = new TournamentPolicy();
        
        $this->assertTrue($policy->create($user));
    }

    /** @test */
    public function leaderboard_controller_handles_all_time_period_without_sql_error()
    {
        // Create some test users and challenges
        $user1 = User::factory()->create(['status' => 'active']);
        $user2 = User::factory()->create(['status' => 'active']);

        $controller = new LeaderboardController();
        
        // Use reflection to test the private method directly
        $reflection = new \ReflectionClass($controller);
        $method = $reflection->getMethod('getWinsLeaderboard');
        $method->setAccessible(true);
        
        // Test all_time period
        $dateRange = ['start' => null, 'end' => null];
        $result = $method->invoke($controller, 'all_time', $dateRange);
        
        // Should not throw any SQL errors and return a collection
        $this->assertIsIterable($result);
    }

    /** @test */
    public function leaderboard_controller_handles_daily_period_without_sql_error()
    {
        // Create some test users
        $user1 = User::factory()->create(['status' => 'active']);
        $user2 = User::factory()->create(['status' => 'active']);

        $controller = new LeaderboardController();
        
        // Use reflection to test the private method directly
        $reflection = new \ReflectionClass($controller);
        $method = $reflection->getMethod('getWinsLeaderboard');
        $method->setAccessible(true);
        
        // Test daily period
        $dateRange = ['start' => today()->startOfDay(), 'end' => today()->endOfDay()];
        $result = $method->invoke($controller, 'daily', $dateRange);
        
        // Should not throw SQL parameter binding errors
        $this->assertIsIterable($result);
    }

    /** @test */
    public function leaderboard_controller_handles_weekly_period_without_sql_error()
    {
        // Create some test users
        $user1 = User::factory()->create(['status' => 'active']);
        $user2 = User::factory()->create(['status' => 'active']);

        $controller = new LeaderboardController();
        
        // Use reflection to test the private method directly
        $reflection = new \ReflectionClass($controller);
        $method = $reflection->getMethod('getWinsLeaderboard');
        $method->setAccessible(true);
        
        // Test weekly period
        $dateRange = ['start' => now()->startOfWeek(), 'end' => now()->endOfWeek()];
        $result = $method->invoke($controller, 'weekly', $dateRange);
        
        // Should not throw SQL parameter binding errors
        $this->assertIsIterable($result);
    }

    /** @test */
    public function leaderboard_controller_handles_monthly_period_without_sql_error()
    {
        // Create some test users
        $user1 = User::factory()->create(['status' => 'active']);
        $user2 = User::factory()->create(['status' => 'active']);

        $controller = new LeaderboardController();
        
        // Use reflection to test the private method directly
        $reflection = new \ReflectionClass($controller);
        $method = $reflection->getMethod('getWinsLeaderboard');
        $method->setAccessible(true);
        
        // Test monthly period
        $dateRange = ['start' => now()->startOfMonth(), 'end' => now()->endOfMonth()];
        $result = $method->invoke($controller, 'monthly', $dateRange);
        
        // Should not throw SQL parameter binding errors
        $this->assertIsIterable($result);
    }
}