<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\User;
use App\Models\Friend;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Auth;

class FriendSuggestionsAmbiguityTest extends TestCase
{
    use RefreshDatabase;

    /**
     * Test that the SQL ambiguity in friends relationship is fixed
     */
    public function test_friends_pluck_does_not_have_sql_ambiguity()
    {
        // Create two test users
        $user1 = User::factory()->create(['username' => 'user1', 'status' => 'active']);
        $user2 = User::factory()->create(['username' => 'user2', 'status' => 'active']);
        
        // Create an accepted friendship between them
        Friend::create([
            'user_id' => $user1->id,
            'friend_id' => $user2->id,
            'status' => 'accepted',
            'request_sent_at' => now(),
            'responded_at' => now(),
        ]);
        
        // Also create the reciprocal friendship (as done in the Friend model)
        Friend::create([
            'user_id' => $user2->id,
            'friend_id' => $user1->id,
            'status' => 'accepted',
            'request_sent_at' => now(),
            'responded_at' => now(),
        ]);
        
        // This should not throw an SQL ambiguity error
        // Previously: pluck('id') was ambiguous between users.id and friends.id
        // Now: pluck('users.id') specifies which table's id column to use
        $friendIds = $user1->friends()->pluck('users.id');
        
        // Should have one friend
        $this->assertCount(1, $friendIds);
        $this->assertTrue($friendIds->contains($user2->id));
    }

    /**
     * Test that the friend suggestions page loads without SQL errors
     */
    public function test_friend_suggestions_page_loads_without_sql_error()
    {
        // Create test users
        $user1 = User::factory()->create(['username' => 'user1', 'status' => 'active']);
        $user2 = User::factory()->create(['username' => 'user2', 'status' => 'active']);
        
        // Create a friendship so the problematic code path is executed
        Friend::create([
            'user_id' => $user1->id,
            'friend_id' => $user2->id,
            'status' => 'accepted',
            'request_sent_at' => now(),
            'responded_at' => now(),
        ]);
        
        // Login as user1
        Auth::login($user1);
        
        // This should not produce an SQL error about ambiguous 'id' column
        $response = $this->get('/friends/suggestions');
        
        // Should not get a 500 error due to SQL ambiguity
        $response->assertStatus(200);
    }
}