<?php

namespace Tests\Feature;

use App\Models\Challenge;
use App\Models\Game;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class ImmediateResolutionTest extends TestCase
{
    use RefreshDatabase;

    /**
     * Test immediate resolution when player claims loss
     */
    public function test_immediate_resolution_when_player_claims_loss()
    {
        $creator = User::factory()->create(['wallet_balance' => 1000]);
        $accepter = User::factory()->create(['wallet_balance' => 1000]);
        $game = Game::factory()->create();

        $challenge = Challenge::create([
            'creator_id' => $creator->id,
            'accepter_id' => $accepter->id,
            'game_id' => $game->id,
            'wager_amount' => 100,
            'best_of' => 1,
            'creator_game_username' => 'Creator123',
            'status' => 'accepted',
        ]);

        // Creator claims loss - should immediately resolve in favor of accepter
        $response = $this->actingAs($creator)
            ->post(route('challenges.claim-result', $challenge), ['result' => 'loss']);

        $response->assertRedirect();
        $response->assertSessionHas('success', 'Match resolved! You forfeited and your opponent wins. Funds have been distributed.');
        
        $challenge->refresh();
        $this->assertEquals('loss', $challenge->creator_claimed_result);
        $this->assertEquals('none', $challenge->accepter_claimed_result); // Accepter didn't need to respond
        $this->assertEquals('completed', $challenge->status);
        $this->assertEquals('accepter_win', $challenge->result);
    }

    /**
     * Test immediate resolution when accepter claims loss
     */
    public function test_immediate_resolution_when_accepter_claims_loss()
    {
        $creator = User::factory()->create(['wallet_balance' => 1000]);
        $accepter = User::factory()->create(['wallet_balance' => 1000]);
        $game = Game::factory()->create();

        $challenge = Challenge::create([
            'creator_id' => $creator->id,
            'accepter_id' => $accepter->id,
            'game_id' => $game->id,
            'wager_amount' => 100,
            'best_of' => 1,
            'creator_game_username' => 'Creator123',
            'status' => 'accepted',
        ]);

        // Accepter claims loss - should immediately resolve in favor of creator
        $response = $this->actingAs($accepter)
            ->post(route('challenges.claim-result', $challenge), ['result' => 'loss']);

        $response->assertRedirect();
        $response->assertSessionHas('success', 'Match resolved! You forfeited and your opponent wins. Funds have been distributed.');
        
        $challenge->refresh();
        $this->assertEquals('none', $challenge->creator_claimed_result); // Creator didn't need to respond
        $this->assertEquals('loss', $challenge->accepter_claimed_result);
        $this->assertEquals('completed', $challenge->status);
        $this->assertEquals('creator_win', $challenge->result);
    }

    /**
     * Test that "I Won" still waits for opponent response
     */
    public function test_win_claim_still_waits_for_opponent()
    {
        $creator = User::factory()->create(['wallet_balance' => 1000]);
        $accepter = User::factory()->create(['wallet_balance' => 1000]);
        $game = Game::factory()->create();

        $challenge = Challenge::create([
            'creator_id' => $creator->id,
            'accepter_id' => $accepter->id,
            'game_id' => $game->id,
            'wager_amount' => 100,
            'best_of' => 1,
            'creator_game_username' => 'Creator123',
            'status' => 'accepted',
        ]);

        // Creator claims win - should wait for accepter
        $response = $this->actingAs($creator)
            ->post(route('challenges.claim-result', $challenge), ['result' => 'win']);

        $response->assertRedirect();
        $response->assertSessionHas('success', 'Your result has been recorded. Waiting for your opponent.');
        
        $challenge->refresh();
        $this->assertEquals('win', $challenge->creator_claimed_result);
        $this->assertEquals('none', $challenge->accepter_claimed_result);
        $this->assertEquals('accepted', $challenge->status); // Still not completed
    }

    /**
     * Test mixed scenario: one wins, then other admits loss
     */
    public function test_mixed_scenario_win_then_loss()
    {
        $creator = User::factory()->create(['wallet_balance' => 1000]);
        $accepter = User::factory()->create(['wallet_balance' => 1000]);
        $game = Game::factory()->create();

        $challenge = Challenge::create([
            'creator_id' => $creator->id,
            'accepter_id' => $accepter->id,
            'game_id' => $game->id,
            'wager_amount' => 100,
            'best_of' => 1,
            'creator_game_username' => 'Creator123',
            'status' => 'accepted',
        ]);

        // Creator claims win first
        $this->actingAs($creator)
            ->post(route('challenges.claim-result', $challenge), ['result' => 'win']);

        // Accepter then claims loss - should immediately resolve
        $response = $this->actingAs($accepter)
            ->post(route('challenges.claim-result', $challenge), ['result' => 'loss']);

        $response->assertRedirect();
        $response->assertSessionHas('success', 'Match resolved! You forfeited and your opponent wins. Funds have been distributed.');
        
        $challenge->refresh();
        $this->assertEquals('win', $challenge->creator_claimed_result);
        $this->assertEquals('loss', $challenge->accepter_claimed_result);
        $this->assertEquals('completed', $challenge->status);
        $this->assertEquals('creator_win', $challenge->result);
    }

    /**
     * Test both players claim loss scenario (mutual forfeit)
     * Note: This is a rare edge case that would require manual intervention
     * since the first loss claim normally resolves immediately
     */
    public function test_both_claim_loss_creates_dispute()
    {
        $creator = User::factory()->create(['wallet_balance' => 1000]);
        $accepter = User::factory()->create(['wallet_balance' => 1000]);
        $game = Game::factory()->create();

        // Create a challenge with both players already having claimed loss
        // This simulates the rare case where both players somehow claim loss
        // (could happen with simultaneous submissions or manual data manipulation)
        $challenge = Challenge::create([
            'creator_id' => $creator->id,
            'accepter_id' => $accepter->id,
            'game_id' => $game->id,
            'wager_amount' => 100,
            'best_of' => 1,
            'creator_game_username' => 'Creator123',
            'status' => 'accepted',
            'creator_claimed_result' => 'loss',
            'accepter_claimed_result' => 'loss',
        ]);

        // Manually trigger the logic to handle this scenario
        // by attempting to claim result when both already claim loss
        $response = $this->actingAs($creator)
            ->post(route('challenges.claim-result', $challenge), ['result' => 'loss']);

        // Since creator already claimed loss, this should be prevented
        $response->assertRedirect();
        $response->assertSessionHas('error', 'You have already claimed a result for this challenge.');
    }

    /**
     * Test theoretical both claim loss scenario using direct model manipulation
     */
    public function test_both_claim_loss_dispute_via_model()
    {
        $creator = User::factory()->create(['wallet_balance' => 1000]);
        $accepter = User::factory()->create(['wallet_balance' => 1000]);
        $game = Game::factory()->create();

        $challenge = Challenge::create([
            'creator_id' => $creator->id,
            'accepter_id' => $accepter->id,
            'game_id' => $game->id,
            'wager_amount' => 100,
            'best_of' => 1,
            'creator_game_username' => 'Creator123',
            'status' => 'accepted',
            'creator_claimed_result' => 'loss',
            'accepter_claimed_result' => 'loss',
        ]);

        // Test the helper method
        $this->assertTrue($challenge->bothClaimLoss());
        
        // In the real system, this would create a dispute if it somehow occurred
        // But our immediate resolution prevents this from happening naturally
    }

    /**
     * Test Challenge model helper methods
     */
    public function test_challenge_helper_methods()
    {
        $creator = User::factory()->create();
        $accepter = User::factory()->create();
        $game = Game::factory()->create();

        // Test bothClaimLoss method
        $challenge = Challenge::create([
            'creator_id' => $creator->id,
            'accepter_id' => $accepter->id,
            'game_id' => $game->id,
            'wager_amount' => 100,
            'best_of' => 1,
            'creator_game_username' => 'Creator123',
            'creator_claimed_result' => 'loss',
            'accepter_claimed_result' => 'loss',
        ]);

        $this->assertTrue($challenge->bothClaimLoss());
        $this->assertFalse($challenge->bothClaimVictory());
        $this->assertTrue($challenge->hasPlayerClaimedLoss());

        // Test hasPlayerClaimedLoss with only one loss
        $challenge->update(['accepter_claimed_result' => 'none']);
        $this->assertTrue($challenge->hasPlayerClaimedLoss());
        
        // Test getWinnerFromLossClaimant
        $this->assertEquals($accepter->id, $challenge->getWinnerFromLossClaimant()->id);

        // Test with accepter claiming loss
        $challenge->update([
            'creator_claimed_result' => 'none',
            'accepter_claimed_result' => 'loss'
        ]);
        $this->assertEquals($creator->id, $challenge->getWinnerFromLossClaimant()->id);
    }
}