<?php

namespace Tests\Feature;

use App\Models\Level;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class LevelSystemTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        
        // Seed the 30-level system for testing
        $this->artisan('db:seed', ['--class' => 'ThirtyLevelSystemSeeder']);
    }

    /**
     * Test level progression based on games played.
     */
    public function test_level_progression_with_games_played()
    {
        $user = User::factory()->create([
            'xp' => 300,
            'games_played' => 0,
            'level' => 1,
            'role' => 'level1',
        ]);

        // User should be at Level 1 initially
        $this->assertEquals(1, $user->level);
        $currentLevel = $user->getCurrentLevel();
        $this->assertEquals('Level 1', $currentLevel->name);

        // Add games played to reach Level 2 requirements (10 games)
        $user->addGamesPlayed(10);
        $user->refresh();

        // Should advance to Level 2
        $this->assertEquals(2, $user->level);
    }

    /**
     * Test wager limits for different levels.
     */
    public function test_wager_limits_by_level()
    {
        $user = User::factory()->create([
            'xp' => 0,
            'games_played' => 0,
            'level' => 1,
            'role' => 'level1', // Ensure the user has a proper role
        ]);

        // Level 1 should have ₦1,000 limit
        $this->assertEquals(1000.00, $user->getMaxWagerAmount());

        // Advance to Level 2 (₦5,000 limit)
        $user->addGamesPlayed(10);
        $user->addXP(100);
        $user->checkLevelUp();
        $user->refresh();

        $this->assertEquals(5000.00, $user->getMaxWagerAmount());
    }

    /**
     * Test tournament creation permissions.
     */
    public function test_tournament_creation_permissions()
    {
        // Level 28 user - should not be able to create tournaments
        $user28 = User::factory()->create([
            'xp' => 2700,
            'games_played' => 890,
            'level' => 28,
            'role' => 'level1', // Ensure non-privileged role
            'is_organizer' => false,
        ]);

        $this->assertFalse($user28->canCreateTournamentsNewSystem());

        // Level 29 user - should be able to create tournaments
        $user29 = User::factory()->create([
            'xp' => 2800,
            'games_played' => 965,
            'level' => 29,
            'role' => 'level1', // Ensure non-privileged role
            'is_organizer' => false,
        ]);

        $this->assertTrue($user29->canCreateTournamentsNewSystem());

        // Creator user - should be able to create tournaments
        $creator = User::factory()->create([
            'role' => 'creator',
            'is_organizer' => false,
        ]);

        $this->assertTrue($creator->canCreateTournamentsNewSystem());

        // Admin user - should be able to create tournaments
        $admin = User::factory()->create([
            'role' => 'admin',
            'is_organizer' => true,
        ]);

        $this->assertTrue($admin->canCreateTournamentsNewSystem());
    }

    /**
     * Test Creator request system.
     */
    public function test_creator_request_system()
    {
        $user = User::factory()->create([
            'creator_request_status' => 'none',
            'role' => 'level1',
            'is_organizer' => false,
        ]);

        // User should be able to request Creator status
        $this->assertTrue($user->canRequestCreatorStatus());

        // Submit Creator request
        $success = $user->requestCreatorStatus('I want to create content for the platform');
        $this->assertTrue($success);

        $user->refresh();
        $this->assertEquals('pending', $user->creator_request_status);
        $this->assertTrue($user->hasPendingCreatorRequest());

        // Should not be able to request again while pending
        $this->assertFalse($user->canRequestCreatorStatus());

        // Admin approves request
        $user->approveCreatorRequest();
        $user->refresh();

        $this->assertEquals('approved', $user->creator_request_status);
        $this->assertEquals('creator', $user->role);
        $this->assertTrue($user->isCreator());
    }

    /**
     * Test Creator has unlimited wager amount.
     */
    public function test_creator_unlimited_wager()
    {
        $creator = User::factory()->create([
            'role' => 'creator',
            'is_organizer' => false,
        ]);

        $this->assertNull($creator->getMaxWagerAmount());
        $this->assertTrue($creator->canCreateChallengeWithAmountNewSystem(999999999));
    }

    /**
     * Test challenge amount validation with new system.
     */
    public function test_challenge_amount_validation()
    {
        $user = User::factory()->create([
            'xp' => 100,
            'games_played' => 10,
            'level' => 2,
            'role' => 'level1', // Ensure non-privileged role
            'is_organizer' => false,
        ]);

        // Level 2 has ₦5,000 limit
        $this->assertTrue($user->canCreateChallengeWithAmountNewSystem(4000));
        $this->assertTrue($user->canCreateChallengeWithAmountNewSystem(5000));
        $this->assertFalse($user->canCreateChallengeWithAmountNewSystem(6000));
    }

    /**
     * Test admin challenge restrictions.
     */
    public function test_admin_challenge_restrictions()
    {
        $admin = User::factory()->create([
            'role' => 'admin',
            'is_organizer' => true,
        ]);

        // Admin should NOT be able to create challenges
        $this->assertFalse($admin->canCreateChallengeWithAmountNewSystem(1000));
        $this->assertFalse($admin->canCreateChallengeWithAmount(1000));

        // But admin should be able to create tournaments
        $this->assertTrue($admin->canCreateTournamentsNewSystem());
        $this->assertTrue($admin->canCreateTournaments());
    }

    /**
     * Test level 30 achievements.
     */
    public function test_level_30_achievements()
    {
        $user = User::factory()->create([
            'xp' => 2900,
            'games_played' => 1040,
            'level' => 1, // Start at level 1 and let system detect current level
            'role' => 'level1', // Ensure non-privileged role
            'is_organizer' => false,
        ]);

        // Force level check
        $user->checkLevelUp();
        $user->refresh();

        $currentLevel = $user->getCurrentLevel();
        $this->assertEquals('Level 30', $currentLevel->name);
        
        // Level 30 should be able to create tournaments
        $this->assertTrue($user->canCreateTournamentsNewSystem());
        
        // Level 30 should have high wager limit
        $this->assertEquals(4600000.00, $user->getMaxWagerAmount());
    }
}
