<?php

namespace Tests\Feature;

use App\Models\Challenge;
use App\Models\Game;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Tests\TestCase;

class MatchProofTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        Storage::fake('public');
    }

    public function test_user_can_submit_match_proof()
    {
        $creator = User::factory()->create(['wallet_balance' => 1000]);
        $accepter = User::factory()->create(['wallet_balance' => 1000]);
        $game = Game::factory()->create();

        $challenge = Challenge::create([
            'creator_id' => $creator->id,
            'accepter_id' => $accepter->id,
            'game_id' => $game->id,
            'wager_amount' => 100,
            'best_of' => 1,
            'status' => 'accepted',
            'creator_game_username' => 'TestPlayer123',
        ]);

        $file = UploadedFile::fake()->image('proof.jpg');

        $response = $this->actingAs($creator)
            ->post(route('challenges.submit-proof', $challenge), [
                'proof_file' => $file,
                'notes' => 'Here is my proof of victory!'
            ]);

        $response->assertRedirect(route('challenges.show', $challenge));
        $response->assertSessionHas('success', 'Proof submitted successfully!');

        // Verify proof was saved to database
        $this->assertDatabaseHas('match_proofs', [
            'challenge_id' => $challenge->id,
            'user_id' => $creator->id,
            'notes' => 'Here is my proof of victory!'
        ]);

        // Verify file was stored
        Storage::disk('public')->assertExists('match_proofs/' . $file->hashName());

        // Verify challenge now has proofs
        $challenge->refresh();
        $this->assertEquals(1, $challenge->matchProofs->count());
        
        $proof = $challenge->matchProofs->first();
        $this->assertEquals($creator->id, $proof->user_id);
        $this->assertEquals('Here is my proof of victory!', $proof->notes);
        $this->assertStringContainsString('/storage/match_proofs/', $proof->file_url);
    }

    public function test_proof_displays_correctly_in_view()
    {
        $creator = User::factory()->create(['wallet_balance' => 1000]);
        $accepter = User::factory()->create(['wallet_balance' => 1000]);
        $game = Game::factory()->create();

        $challenge = Challenge::create([
            'creator_id' => $creator->id,
            'accepter_id' => $accepter->id,
            'game_id' => $game->id,
            'wager_amount' => 100,
            'best_of' => 1,
            'status' => 'accepted',
            'creator_game_username' => 'TestPlayer123',
        ]);

        // Create a proof directly
        $challenge->matchProofs()->create([
            'user_id' => $creator->id,
            'file_url' => '/storage/match_proofs/test-proof.jpg',
            'notes' => 'Test proof notes'
        ]);

        $response = $this->actingAs($creator)
            ->get(route('challenges.show', $challenge));

        $response->assertStatus(200);
        $response->assertSee($creator->username);
        $response->assertSee('Test proof notes');
        $response->assertSee('/storage/match_proofs/test-proof.jpg');
        $response->assertDontSee('No match proofs uploaded yet');
    }

    public function test_no_proof_message_shows_when_no_proofs_exist()
    {
        $creator = User::factory()->create(['wallet_balance' => 1000]);
        $accepter = User::factory()->create(['wallet_balance' => 1000]);
        $game = Game::factory()->create();

        $challenge = Challenge::create([
            'creator_id' => $creator->id,
            'accepter_id' => $accepter->id,
            'game_id' => $game->id,
            'wager_amount' => 100,
            'best_of' => 1,
            'status' => 'accepted',
            'creator_game_username' => 'TestPlayer123',
        ]);

        $response = $this->actingAs($creator)
            ->get(route('challenges.show', $challenge));

        $response->assertStatus(200);
        $response->assertSee('No match proofs uploaded yet');
    }

    public function test_complete_proof_upload_flow()
    {
        $creator = User::factory()->create(['wallet_balance' => 1000]);
        $accepter = User::factory()->create(['wallet_balance' => 1000]);
        $game = Game::factory()->create();

        $challenge = Challenge::create([
            'creator_id' => $creator->id,
            'accepter_id' => $accepter->id,
            'game_id' => $game->id,
            'wager_amount' => 100,
            'best_of' => 1,
            'status' => 'accepted',
            'creator_game_username' => 'TestPlayer123',
        ]);

        // Initially, should show no proofs message
        $response = $this->actingAs($creator)
            ->get(route('challenges.show', $challenge));
        $response->assertSee('No match proofs uploaded yet');

        // Upload a proof
        $file = UploadedFile::fake()->image('proof.jpg');
        $response = $this->actingAs($creator)
            ->post(route('challenges.submit-proof', $challenge), [
                'proof_file' => $file,
                'notes' => 'Here is my proof of victory!'
            ]);

        $response->assertRedirect(route('challenges.show', $challenge));
        $response->assertSessionHas('success', 'Proof submitted successfully!');

        // After upload, should show the proof immediately
        $response = $this->actingAs($creator)
            ->get(route('challenges.show', $challenge));
        
        $response->assertStatus(200);
        $response->assertSee($creator->username);
        $response->assertSee('Here is my proof of victory!');
        $response->assertDontSee('No match proofs uploaded yet');
        
        // Should see the uploaded image
        $proof = $challenge->fresh()->matchProofs->first();
        $response->assertSee($proof->file_url);
    }
}